package ts.query;

import ts.tester.UnitTest;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.*;

public class QueryConnectionTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryConnectionTest.class, args);
  }

  public static class MyConnection extends QueryConnection {
    static final long serialVersionUID = -1L;
    long openTimeMillis = 0L;
    private boolean isClosed = true;
    private long limitTimeMillis;
    public MyConnection(QueryConnectionConfig cfg) { super(cfg); }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran)
    { super(cfg, tran); }
    @Override
    protected long getOpenTimeMillis() { return this.openTimeMillis; }
    @Override
    public void open() throws ReasonedException {
      this.openTimeMillis = System.currentTimeMillis();
      this.isClosed = false;
    }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() throws ReasonedException {}
    @Override
    public void close() throws ReasonedException { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  public void constructor_config()
  {
    MSG("接続設定オブジェクトを引数にとるコンストラクタの確認。");
    MSG("- 接続設定ファイルを使用しない場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();

    MyConnection conn = new MyConnection(config);
    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    EQUAL(conn.getOpenTimeMillis(), 0L);
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), 0);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    NOTNULL(conn.getQueryHistory());

    try {
      conn.open();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), 0L);
    FALSE(conn.isClosed());
    TRUE (conn.isOpened());
    NOTNULL(conn.getQueryHistory());

    try {
      conn.close();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), 0L);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    NOTNULL(conn.getQueryHistory());
  }

  public void constructor_config_HasConfigFile()
  {
    MSG("接続設定オブジェクトを引数にとるコンストラクタの確認。");
    MSG("- 接続設定ファイルありの場合。");

    QueryConnectionConfig config = new QueryConnectionConfig(
      "QueryConnectionTest_constructor_config_HasConfigFile");

    MyConnection conn = new MyConnection(config);
    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    EQUAL(conn.getOpenTimeMillis(), 0L);
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), 0L);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    NOTNULL(conn.getQueryHistory());

    try {
      conn.open();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), conn.getOpenTimeMillis() + 20000L);
    FALSE(conn.isClosed());
    TRUE (conn.isOpened());
    NOTNULL(conn.getQueryHistory());

    try {
      conn.close();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), conn.getOpenTimeMillis() + 20000L);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    NOTNULL(conn.getQueryHistory());
  }

  public void constructor_config_tran()
  {
    MSG("接続設定オブジェクトとトランザクションを引数にとるコンストラクタの確認。");
    MSG("- 接続設定ファイルなしの場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();

    QueryTransaction tran = new QueryTransaction();
    tran.setTimeoutMillis(45678L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    MyConnection conn = new MyConnection(config, tran);
    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    EQUAL(conn.getOpenTimeMillis(), 0L);
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), tran.getLimitTimeMillis());
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.open();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), tran.getLimitTimeMillis());
    FALSE(conn.isClosed());
    TRUE (conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.close();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(), "");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 0L);
    EQUAL(conn.getLimitTimeMillis(), tran.getLimitTimeMillis());
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    tran.end();
  }

  public void constructor_config_tran_HasConfigFile()
  {
    MSG("接続設定オブジェクトとトランザクションを引数にとるコンストラクタの確認。");
    MSG("- 接続設定ファイルありの場合。");
    MSG("- 接続タイムアウトよりもトランザクション・タイムアウトの方が遅い場合。");

    QueryConnectionConfig config = new QueryConnectionConfig(
      "QueryConnectionTest_constructor_config_HasConfigFile");

    QueryTransaction tran = new QueryTransaction();
    tran.setTimeoutMillis(30000L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    long TRAN_LIMIT = tran.getLimitTimeMillis();

    MyConnection conn = new MyConnection(config, tran);
    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    EQUAL(conn.getOpenTimeMillis(), 0L);
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), TRAN_LIMIT);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.open();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), conn.getOpenTimeMillis() + 20000L);
    FALSE(conn.isClosed());
    TRUE (conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.close();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), conn.getOpenTimeMillis() + 20000L);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    tran.end();
  }

  public void constructor_config_tran_HasConfigFile_1()
  {
    MSG("接続設定オブジェクトとトランザクションを引数にとるコンストラクタの確認。");
    MSG("- 接続設定ファイルありの場合。");
    MSG("- 接続タイムアウトよりもトランザクション・タイムアウトの方が早い場合。");

    QueryConnectionConfig config = new QueryConnectionConfig(
      "QueryConnectionTest_constructor_config_HasConfigFile");

    QueryTransaction tran = new QueryTransaction();
    tran.setTimeoutMillis(10000L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    long TRAN_LIMIT = tran.getLimitTimeMillis();

    MyConnection conn = new MyConnection(config, tran);
    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    EQUAL(conn.getOpenTimeMillis(), 0L);
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), TRAN_LIMIT);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.open();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), TRAN_LIMIT);
    FALSE(conn.isClosed());
    TRUE (conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    try {
      conn.close();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(conn.getConnectionId(),
      "QueryConnectionTest_constructor_config_HasConfigFile");
    EQUAL(conn.getConfig(), config);
    POSITIVE(conn.getOpenTimeMillis());
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(conn.getLimitTimeMillis(), TRAN_LIMIT);
    TRUE (conn.isClosed());
    FALSE(conn.isOpened());
    EQUAL(conn.getQueryHistory(), tran.getQueryHistory());

    tran.end();
  }
}
