package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import java.io.*;
import java.nio.channels.*;
import java.util.*;

public class QueryExecutionConfigTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryExecutionConfigTest.class, args);
  }

  File DATA_DIR = new File("test/data/");
  File CFG_DIR = new File(DATA_DIR, "ts/query/configs/execution");
  File SAMPLE_DIR = new File(DATA_DIR, "ts/query/QueryExecutionConfig/");

  static class MyExecution implements IQueryExecution {
    private QueryExecutionConfig config;
    public MyExecution(QueryExecutionConfig cfg, IQueryConnection conn)
    { this.config = cfg; }
    @Override
    public String getExecutionId() { return this.config.getExecutionId(); }
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException {}
  }

  static class MyExecution2 implements IQueryExecution {
    @Override
    public String getExecutionId() { return ""; }
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException {}
  }

  static class MyExecution3 implements IQueryExecution {
    public MyExecution3(QueryExecutionConfig cfg, IQueryConnection conn)
    { throw new RuntimeException("ERR0"); }
    @Override
    public String getExecutionId() { return ""; }
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException {}
  }

  static class MyConnection implements IQueryConnection {
    static final long serialVersionUID = -1L;
    private QueryConnectionConfig cfg;
    private boolean isClosed = true;
    public MyConnection(QueryConnectionConfig cfg) { this.cfg = cfg; }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran)
    { this(cfg); }
    @Override
    public String getConnectionId() { return cfg.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return 0L; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
    @Override
    public void open() throws ReasonedException {
      this.isClosed = false;
    }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() throws ReasonedException {}
    @Override
    public void close() throws ReasonedException { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  static class MyTransaction extends QueryTransaction {
    public MyTransaction() {}
    @Override
    public IQueryConnection getQueryConnection(String connId)
    throws ReasonedException, ReasonedRuntimeException {
      return new MyConnection(new QueryConnectionConfig());
    }
  }


  public void constructor()
  {
    MSG("デフォルト・コンストラクタ。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    EQUAL(config.getExecutionId(), "");
    NOTNULL(config.getResource());
    NULL(AbstractResource.class.cast(config.getResource()).getPath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getExecutionClass(), "");
    EQUAL(config.getConnectionId(), "");
    EQUAL(config.getLimitFetchCount(), 0);
    EQUAL(config.getLimitSpentTime(), 0L);
  }

  public void constructor_execId()
  {
    MSG("実行IDを引数にとるコンストラクタ。");

    String EID = "QueryExecutionConfigTest_constructor_execId";

    QueryExecutionConfig config = new QueryExecutionConfig(EID);
    EQUAL(config.getExecutionId(), EID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, EID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getExecutionClass(), "ts.query.QueryExecutionConfigTest$MyExecution1");
    EQUAL(config.getConnectionId(), "ccc");
    EQUAL(config.getLimitFetchCount(), 1000);
    EQUAL(config.getLimitSpentTime(), 100000L);
  }

  public void constructor_execId_NullOrEmpty()
  {
    MSG("引数がヌルや空文字列の場合。");

    try {
      new QueryExecutionConfig(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[execution Id=null]");
    }

    try {
      new QueryExecutionConfig("");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[execution Id=]");
    }
  }

  public void loadResource_LoadXmlFile()
  {
    MSG("XML形式の設定ファイルのロードの確認。");

    String EID = "QueryExecutionConfigTest_loadResource_LoadXmlFile";

    QueryExecutionConfig config = new QueryExecutionConfig(EID);
    EQUAL(config.getExecutionId(), EID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, EID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getExecutionClass(),
      "ts.query.QueryExecutionConfigTest$MyExecution2");
    EQUAL(config.getConnectionId(), "ddd");
    EQUAL(config.getLimitFetchCount(), 2000);
    EQUAL(config.getLimitSpentTime(), 200000L);

    EQUAL(config.getResource().getFirstValue("ts-query.execution.class"),
      "ts.query.QueryExecutionConfigTest$MyExecution2");
    EQUAL(config.getResource().getFirstValue("ts-query.execution.connection.id")
      ,"ddd");
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.limit.fetchcount"), "2000"); 
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.limit.spenttime"), "200000"); 
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.aaa.bbb.ccc"), "xxxx"); 

    EQUAL(config.typedGetter().getString("ts-query.execution.class"),
      "ts.query.QueryExecutionConfigTest$MyExecution2");
    EQUAL(config.typedGetter().getString("ts-query.execution.connection.id"),
      "ddd");
    EQUAL(config.typedGetter().getString("ts-query.execution.limit.fetchcount"),
      "2000");
    EQUAL(config.typedGetter().getInteger(
      "ts-query.execution.limit.fetchcount"), 2000);
    EQUAL(config.typedGetter().getString("ts-query.execution.limit.spenttime"),
      "200000");
    EQUAL(config.typedGetter().getLong(
      "ts-query.execution.limit.spenttime"), 200000L);
    EQUAL(config.typedGetter().getString("ts-query.execution.aaa.bbb.ccc"),
      "xxxx");

    EQUAL(config.typedGetter().getList("ts-query.execution.class"),
      Arrays.asList("ts.query.QueryExecutionConfigTest$MyExecution2"));
    EQUAL(config.typedGetter().getList("ts-query.execution.connection.id"),
      Arrays.asList("ddd"));
    EQUAL(config.typedGetter().getList("ts-query.execution.limit.fetchcount"),
      Arrays.asList("2000"));
    EQUAL(config.typedGetter().getList("ts-query.execution.limit.spenttime"),
      Arrays.asList("200000"));
    EQUAL(config.typedGetter().getList("ts-query.execution.aaa.bbb.ccc"),
      Arrays.asList("xxxx"));
  }

  public void loadResource_LoadPropFile()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードの確認。");

    String EID = "QueryExecutionConfigTest_loadResource_LoadPropFile";

    QueryExecutionConfig config = new QueryExecutionConfig(EID);
    EQUAL(config.getExecutionId(), EID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, EID + ".properties").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getExecutionClass(),
      "ts.query.QueryExecutionConfigTest$MyExecution4");
    EQUAL(config.getConnectionId(), "eee");
    EQUAL(config.getLimitFetchCount(), 4444);
    EQUAL(config.getLimitSpentTime(), 444444L);

    EQUAL(config.getResource().getFirstValue("ts-query.execution.class"),
      "ts.query.QueryExecutionConfigTest$MyExecution4");
    EQUAL(config.getResource().getFirstValue("ts-query.execution.connection.id")
      ,"eee");
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.limit.fetchcount"), "4444"); 
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.limit.spenttime"), "444444"); 
    EQUAL(config.getResource().getFirstValue(
      "ts-query.execution.aaa.bbb.ccc"), "yyy"); 

    EQUAL(config.typedGetter().getString("ts-query.execution.class"),
      "ts.query.QueryExecutionConfigTest$MyExecution4");
    EQUAL(config.typedGetter().getString("ts-query.execution.connection.id"),
      "eee");
    EQUAL(config.typedGetter().getString("ts-query.execution.limit.fetchcount"),
      "4444");
    EQUAL(config.typedGetter().getInteger(
      "ts-query.execution.limit.fetchcount"), 4444);
    EQUAL(config.typedGetter().getString("ts-query.execution.limit.spenttime"),
      "444444");
    EQUAL(config.typedGetter().getLong(
      "ts-query.execution.limit.spenttime"), 444444L);
    EQUAL(config.typedGetter().getString("ts-query.execution.aaa.bbb.ccc"),
      "yyy");
  }

  public void loadResource_FailToLoadConfig_Xml()
  {
    MSG("XML形式の設定ファイルのロードに失敗した場合。");

    String EID = "QueryExecutionConfigTest_loadResource_FailToLoadConfig_Xml";

    try {
      new QueryExecutionConfig(EID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.FailToLoadExecutionConfigFile);
    }
  }

  public void loadResource_FailToLoadConfig_Prop()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードに失敗した場合。");

    String EID = "QueryExecutionConfigTest_loadResource_FailToLoadConfig_Prop";
    File file = new File(CFG_DIR, EID + ".properties");

    RandomAccessFile raf = null;
    FileLock lock = null;

    try {
      raf = new RandomAccessFile(file, "rw");
      FileChannel channel = raf.getChannel();
      lock = channel.lock(0, Long.MAX_VALUE, false);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      new QueryExecutionConfig(EID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.FailToLoadExecutionConfigFile);
    }
    catch (Throwable e) {
      NG(e);
    }
    finally {
      try {
        if (lock != null) try { lock.release(); } catch (Exception e) {}
        if (raf  != null) try { raf.close(); } catch (Exception e) {}
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void loadResource_ConfigFileNotFound()
  {
    MSG("設定ファイルがXML形式でもJavaプロパティ形式でも存在しなかった場合。");

    try {
      new QueryExecutionConfig("xx");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.ExecutionConfigFileNotFound);
    }
  }

  public void getExecuteClass()
  {
    MSG("実行クラス名の設定と取得の確認。");

    QueryExecutionConfig config;

    config = new QueryExecutionConfig();
    EQUAL(config.getExecutionClass(), "");

    config.getResource().setFirstValue("ts-query.execution.class", "X");
    EQUAL(config.getExecutionClass(), "X");

    config.getResource().setFirstValue("ts-query.execution.class", "Y");
    EQUAL(config.getExecutionClass(), "Y");

    config.getResource().setFirstValue("ts-query.execution.class", "Z");
    EQUAL(config.getExecutionClass(), "Z");

    config.getResource().setFirstValue("ts-query.execution.class", "");
    EQUAL(config.getExecutionClass(), "");

    config.getResource().removeChildren("ts-query.execution.class");
    EQUAL(config.getExecutionClass(), "");
  }

  public void getConnectionId()
  {
    MSG("接続先IDの設定と取得の確認。");

    QueryExecutionConfig config;

    config = new QueryExecutionConfig();
    EQUAL(config.getExecutionClass(), "");

    config.getResource().setFirstValue("ts-query.execution.connection.id", "C");
    EQUAL(config.getConnectionId(), "C");

    config.getResource().setFirstValue("ts-query.execution.connection.id", "D");
    EQUAL(config.getConnectionId(), "D");

    config.getResource().setFirstValue("ts-query.execution.connection.id", "E");
    EQUAL(config.getConnectionId(), "E");

    config.getResource().setFirstValue("ts-query.execution.connection.id", "");
    EQUAL(config.getConnectionId(), "");

    config.getResource().removeChildren("ts-query.execution.connection.id");
    EQUAL(config.getConnectionId(), "");
  }

  public void getLimitFetchCount()
  {
    MSG("処理件数の制限値の取得の確認。");

    QueryExecutionConfig config;

    config = new QueryExecutionConfig();
    EQUAL(config.getLimitFetchCount(), 0);

    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "1000");
    EQUAL(config.getLimitFetchCount(), 1000);

    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "9999");
    EQUAL(config.getLimitFetchCount(), 9999);

    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "-111");
    EQUAL(config.getLimitFetchCount(), -111);

    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "-0");
    EQUAL(config.getLimitFetchCount(), 0);

    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "");
    EQUAL(config.getLimitFetchCount(), 0);

    config.getResource().removeChildren("ts-query.execution.limit.fetchcount");
    EQUAL(config.getLimitFetchCount(), 0);
  }

  public void getLimitFetchCount_IllegalLimitFetchCount()
  {
    MSG("処理件数の制限値に不正な値を設定した場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "abc");

    try {
      config.getLimitFetchCount();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.IllegalLimitFetchCount);
    }
  }

  public void getLimitSpentTime()
  {
    MSG("処理時間の制限値の取得の確認。");

    QueryExecutionConfig config;

    config = new QueryExecutionConfig();
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "1000");
    EQUAL(config.getLimitSpentTime(), 1000L);

    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "9999");
    EQUAL(config.getLimitSpentTime(), 9999L);

    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "-111");
    EQUAL(config.getLimitSpentTime(), -111L);

    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "-0");
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "");
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().removeChildren("ts-query.execution.limit.spenttime");
    EQUAL(config.getLimitSpentTime(), 0L);
  }

  public void getLimitSpentTime_IllegalLimitSpentTime()
  {
    MSG("処理時間の制限値に不正な値を設定した場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "abc");

    try {
      config.getLimitSpentTime();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.IllegalLimitSpentTime);
    }
  }

  public void create()
  {
    MSG("IQueryExecutionオブジェクトを作成するメソッドの確認。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      MyExecution exec = config.create();
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQueryExecution exec = config.create();
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }
 
  public void create_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecutionXXX");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }

  public void create_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution2");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      config.create();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.ExecutionConstructorNotFound);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_FailToCreate()
  {
    MSG("IQueryExecutionオブジェクトの作成に失敗した場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution3");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.FailToCreateExecution);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create");

    try {
      MyExecution2 exec = config.create();
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_FailToCreateConnection()
  {
    MSG("コネクション・クラスの作成に失敗した場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");
    config.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionConfigTest_create_XXX");

    try {
      MyExecution exec = config.create();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran()
  {
    MSG("トランザクションを引数にとるIQueryExecutionオブジェクトを作成するメソッドの確認。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");

    try {
      MyExecution exec = config.create(tran);
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQueryExecution exec = config.create(tran);
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_Null()
  {
    MSG("引数ヌルの場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();

    try {
      config.create((IQueryTransaction) null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e);
    }
    catch (ReasonedException e) {
      NG(e);
    }
  }

  public void create_tran_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }
 
  public void create_tran_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecutionXXX");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }

  public void create_tran_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution2");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.ExecutionConstructorNotFound);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_FailToCreate()
  {
    MSG("IQueryExecutionオブジェクトの作成に失敗した場合。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution3");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.FailToCreateExecution);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    IQueryTransaction tran = new MyTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");

    try {
      MyExecution2 exec = config.create(tran);
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_FailToCreateConnection()
  {
    MSG("コネクション・クラスの作成に失敗した場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");

    try {
      MyExecution exec = config.create(tran);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_conn()
  {
    MSG("コネクションを引数にとるIQueryExecutionオブジェクトを作成するメソッドの確認。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");

    try {
      MyExecution exec = config.create(conn);
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQueryExecution exec = config.create(conn);
      NOTNULL(exec);
      EQUAL(exec.getExecutionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_conn_Null()
  {
    MSG("引数ヌルの場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();

    try {
      config.create((IQueryConnection) null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e);
    }
    catch (ReasonedException e) {
      NG(e);
    }
  }

  public void create_conn_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();

    try {
      config.create(conn);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }
 
  public void create_conn_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecutionXXX");

    try {
      config.create(conn);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionClassNotFound);
    }
  }

  public void create_conn_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution2");

    try {
      config.create(conn);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.ExecutionConstructorNotFound);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_conn_FailToCreate()
  {
    MSG("IQueryExecutionオブジェクトの作成に失敗した場合。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution3");

    try {
      config.create(conn);
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.FailToCreateExecution);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_conn_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    IQueryConnection conn = new MyConnection(new QueryConnectionConfig());

    QueryExecutionConfig config = new QueryExecutionConfig();
    config.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionConfigTest$MyExecution");

    try {
      MyExecution2 exec = config.create(conn);
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void outputSampleXml_pw()
  {
    MSG("XML形式の実行設定ファイルの出力メソッドの確認。");

    String EID = "QueryExecutionConfigTest_outputSampleXml_pw";

    File file = new File(SAMPLE_DIR, EID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
        "Windows-31J"));

      QueryExecutionConfig config = new QueryExecutionConfig();
      config.outputSampleXml(pw, "Windows-31J");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, EID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryExecutionConfig config = new QueryExecutionConfig(EID);
      EQUAL(config.getExecutionClass(), "...");
      EQUAL(config.getConnectionId(), "...");
      EQUAL(config.getLimitFetchCount(), 0);
      EQUAL(config.getLimitSpentTime(), 0L);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void outputSampleXml_pw_Null()
  {
    MSG("引数がヌルの場合。");

    PrintWriter pw = null;
    try {
      QueryExecutionConfig config = new QueryExecutionConfig();
      config.outputSampleXml(null, "Windows-31J");
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      pw = new PrintWriter(new StringWriter());
      QueryExecutionConfig config = new QueryExecutionConfig();
      config.outputSampleXml(pw, null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }
  }

  public void outputSampleXmlEntries_pw_Inheritance()
  {
    MSG("XMLエントリを出力するメソッドの継承の確認。");

    String EID =
      "QueryExecutionConfigTest_outputSampleXmlEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, EID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "UTF8"));
      
      QueryExecutionConfig config = new QueryExecutionConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSampleXmlEntries(PrintWriter pw) throws IOException {
          pw.println("      <eeee>");
          pw.println("        <ffff>FFFF</ffff>");
          pw.println("      </eeee>");
        }
      };
      config.outputSampleXml(pw, "UTF8");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, EID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryExecutionConfig config = new QueryExecutionConfig(EID);
      EQUAL(config.getExecutionClass(), "...");
      EQUAL(config.getConnectionId(), "...");
      EQUAL(config.getLimitFetchCount(), 0);
      EQUAL(config.getLimitSpentTime(), 0L);
      EQUAL(config.typedGetter().getString("ts-query.execution.eeee.ffff"),
        "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw()
  {
    MSG("プロパティ形式の実行設定ファイルの出力メソッドの確認。");

    String EID = "QueryExecutionConfigTest_outputSampleProp_pw";

    File file = new File(SAMPLE_DIR, EID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
        "Windows-31J"));

      QueryExecutionConfig config = new QueryExecutionConfig();
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, EID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryExecutionConfig config = new QueryExecutionConfig(EID);
      EQUAL(config.getExecutionClass(), "...");
      EQUAL(config.getConnectionId(), "...");
      EQUAL(config.getLimitFetchCount(), 0);
      EQUAL(config.getLimitSpentTime(), 0L);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      QueryExecutionConfig config = new QueryExecutionConfig();
      config.outputSampleProp(null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSamplePropEntries_pw_Inheritance()
  {
    MSG("プロパティ・エントリを出力するメソッドの継承の確認。");

    String EID =
      "QueryExecutionConfigTest_outputSamplePropEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, EID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
        "UTF8"));

      QueryExecutionConfig config = new QueryExecutionConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSamplePropEntries(PrintWriter pw) throws IOException {
          pw.println("ts-query.execution.eeee.ffff = FFFF");
        }
      };
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, EID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryExecutionConfig config = new QueryExecutionConfig(EID);
      EQUAL(config.getExecutionClass(), "...");
      EQUAL(config.getConnectionId(), "...");
      EQUAL(config.getLimitFetchCount(), 0);
      EQUAL(config.getLimitSpentTime(), 0L);
      EQUAL(config.typedGetter().getString("ts-query.execution.eeee.ffff"),
        "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void executeCommand_args_config_FileTypeIsXml()
  {
    MSG("実行設定ファイルのサンプルを出力するメソッドの確認(XML)。");

    String EID =
      "QueryExecutionConfigTest_executeCommand_args_config_FileTypeIsXml";

    QueryExecutionConfig config = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, EID + ".xml").getPath(), "XmL", "UTF8"},
      config), 0);
  }

  public void executeCommand_args_config_FileTypeIsProp()
  {
    MSG("実行設定ファイルのサンプルを出力するメソッドの確認(Prop)。");

    String EID =
      "QueryExecutionConfigTest_executeCommand_args_config_FileTypeIsProp";

    QueryExecutionConfig config = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, EID + ".properties").getPath(), "PrOp", "UTF8"},
      config), 0);
  }

  public void executeCommand_args_config_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    EQUAL(QueryExecutionConfig.executeCommand(null, config), 1);
  }

  public void executeCommand_args_config_ZeroArgs()
  {
    MSG("引数の配列の要素数がゼロの場合。");

    QueryExecutionConfig config = new QueryExecutionConfig();
    EQUAL(QueryExecutionConfig.executeCommand(new String[0], config), 1);
  }

  public void executeCommand_args_config_IllegalArgsCount()
  {
    MSG("引数の配列の要素数が不正な場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"p"}, cfg), 1);
    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"p","q"}, cfg), 1);
    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"p","q","xml"},
      cfg), 1);
    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"p","q","xml",
      "UTF8", "t"}, cfg), 1);
    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"p","q","xml",
      "UTF8", "t", "u"}, cfg), 1);
  }

  public void executeCommand_args_config_IllegalOutputType()
  {
    MSG("出力ファイルの形式が不正な場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XXX", "UTF8"}, cfg), 1);
  }

  public void executeCommand_args_config_IllegalEncoding()
  {
    MSG("出力ファイルの文字エンコーディングが不正な場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XML", "UUUU"}, cfg), 2);
  }

  public void executeCommand_args_config_OutputFilePathIsBad()
  {
    MSG("出力ファイルのパスが不正な場合。");
    MSG("- 指定したパスが既存のディレクトリだった場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();

    EQUAL(QueryExecutionConfig.executeCommand(new String[]{"pgm",
      SAMPLE_DIR.getPath(), "XML", "UTF8"}, cfg), 3);
  }
}
