package ts.query;

import ts.tester.UnitTest;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.*;

public class QueryExecutionTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryExecutionTest.class, args);
  }

  enum ENUM { AAA }

  static boolean HAS_RESULT_TABLE = false;
  static boolean FAIL_TO_EXECUTE = false;
  static StringBuilder LOG = null;
  static void CLEAR_LOG() { LOG = new StringBuilder(); }

  public static class MyExecution extends QueryExecution {
    public MyExecution(QueryExecutionConfig config) 
    throws ReasonedException, ReasonedRuntimeException { super(config); }
    public MyExecution(QueryExecutionConfig config, IQueryConnection conn) 
    { super(config, conn); }
    protected void prepareContent(
      QueryExecutionContent content, Map<String,Object> inputMap
    ) throws ReasonedException, ReasonedRuntimeException {
      if (HAS_RESULT_TABLE) {
        content.addOutput(new QueryExecutionOutput("COL"));
      }
      LOG.append("[prepareContent")
         .append("[connection id=").append(content.getConnectionId())
         .append("][execution id=").append(content.getExecutionId())
         .append("]]");
    }
    protected void executeContent(QueryExecutionContent content)
    throws ReasonedException, ReasonedRuntimeException {
      LOG.append("[executeContent")
         .append("[connection id=").append(content.getConnectionId())
         .append("][execution id=").append(content.getExecutionId())
         .append("]]");
      if (FAIL_TO_EXECUTE) throw new ReasonedException(ENUM.AAA);
    }
  }

  static long CONNECTION_LIMIT_TIME = 0L;

  public static class MyConnection implements IQueryConnection {
    private boolean isClosed = true;
    private QueryConnectionConfig cfg;
    private IQueryHistory history;
    private long limitTimeMillis;
    public MyConnection(QueryConnectionConfig cfg) {
      this.cfg = cfg;
      this.history = new QueryHistory();
      this.limitTimeMillis = 0L;
    }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran) {
      this.cfg = cfg;
      this.history = tran.getQueryHistory();
      this.limitTimeMillis = tran.getLimitTimeMillis();
    }
    @Override
    public String getConnectionId() { return this.cfg.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return CONNECTION_LIMIT_TIME; }
    @Override
    public IQueryHistory getQueryHistory() { return this.history; }
    @Override
    public void open() throws ReasonedException { this.isClosed = false; }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() {}
    @Override
    public void close() { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  public static class ErrConnection extends MyConnection {
    public ErrConnection(QueryConnectionConfig cfg) {
      super(cfg);
      throw new RuntimeException();
    }
  }


  protected void preInvocation(String method)
  {
    LOG = new StringBuilder();
    HAS_RESULT_TABLE = false;
    FAIL_TO_EXECUTE = false;
  }

  public void constructor_config()
  {
    MSG("実行設定オブジェクトを引数コンストラクタの確認。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    EQUAL(exec.getConfig(), cfg);
    EQUAL(exec.getExecutionId(), "");
    TRUE (exec.getQueryConnection() instanceof IQueryConnection);
    TRUE (exec.getQueryConnection() instanceof MyConnection);
    MyConnection conn = exec.getQueryConnection();
    EQUAL(conn.getConnectionId(), "QueryExecutionTest_constructor_config");

    cfg = new QueryExecutionConfig("QueryExecutionTest_constructor_config1");
    try {
      exec = new MyExecution(cfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }
    EQUAL(exec.getConfig(), cfg);
    EQUAL(exec.getExecutionId(), "QueryExecutionTest_constructor_config1");
    TRUE (exec.getQueryConnection() instanceof IQueryConnection);
    TRUE (exec.getQueryConnection() instanceof MyConnection);
    conn = exec.getQueryConnection();
    EQUAL(conn.getConnectionId(), "QueryExecutionTest_constructor_config");
  }

  public void constructor_config_Null()
  {
    MSG("引数がヌルの場合。");

    MyExecution exec = null;
    try {
      exec = new MyExecution(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e);
    }
  }

  public void constructor_config_IllegalConnectionConfig()
  {
    MSG("このオブジェクトが使用する接続設定が不正だった場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "XXX");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionConfigFileNotFound);
    }

    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config_IllegalConnectionConfig");

    try {
      exec = new MyExecution(execCfg);
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.FailToLoadConnectionConfigFile);
    }
  }

  public void constructor_config_FailToCreateConnection()
  {
    MSG("このオブジェクトが使用するクエリ・コネクションの作成に失敗した場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config_FailToCreateQueryConnection");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.FailToCreateConnection);
    }
  }

  public void constructor_config_conn()
  {
    MSG("実行設定オブジェクトとクエリ・コネクションを引数にとるコンストラクタの確認。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "XXX");

    QueryConnectionConfig connCfg = new QueryConnectionConfig(
      "QueryExecutionTest_constructor_config");
    
    MyConnection conn = null;
    try {
      conn = connCfg.create();
    }
    catch (Exception e) {
      NG(e);
    }

    MyExecution exec = new MyExecution(execCfg, conn);

    EQUAL(exec.getConfig(), execCfg);
    EQUAL(exec.getExecutionId(), "");
    EQUAL(exec.getQueryConnection(), conn);
    EQUAL(exec.getQueryConnection().getConnectionId(),
      "QueryExecutionTest_constructor_config");

    execCfg = new QueryExecutionConfig(
      "QueryExecutionTest_constructor_config1");
    exec = new MyExecution(execCfg, conn);
    EQUAL(exec.getConfig(), execCfg);
    EQUAL(exec.getExecutionId(), "QueryExecutionTest_constructor_config1");
    EQUAL(exec.getQueryConnection(), conn);
    EQUAL(exec.getQueryConnection().getConnectionId(),
      "QueryExecutionTest_constructor_config");
  }

  public void constructor_config_conn_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "XXX");

    QueryConnectionConfig connCfg = new QueryConnectionConfig(
      "QueryExecutionTest_constructor_config");
    
    MyConnection conn = null;
    try {
      conn = connCfg.create();
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      new MyExecution(null, conn);
      NG();
    }
    catch (AssertionError e) {
      OK(e);
    }

    try {
      new MyExecution(execCfg, null);
      NG();
    }
    catch (AssertionError e) {
      OK(e);
    }
  }

  public void calcLimitTimeMillis_beginTm()
  {
    MSG("終了時間の制限値を算出するメソッドの確認。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "3000");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    }
    catch (Exception e) {
      NG(e);
      return;
    }

    MyConnection conn = exec.getQueryConnection();
    CONNECTION_LIMIT_TIME = 12345L;
    EQUAL(conn.getLimitTimeMillis(), 12345L);

    MSG("- コネクションの制限時刻 ＞ 開始時刻＋制限時間の場合。");

    EQUAL(exec.calcLimitTimeMillis(123L), 3123L);

    MSG("- コネクションの制限時刻 ＝ 開始時刻＋制限時間の場合。");

    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "12222");
    EQUAL(exec.calcLimitTimeMillis(123L), 12345);

    MSG("- コネクションの制限時刻 ＜ 開始時刻＋制限時間の場合。");

    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "20000");
    EQUAL(exec.calcLimitTimeMillis(123L), 12345L);
  }

  public void calcLimitTimeMillis_beginTm_ZeroOrNegative()
  {
    MSG("開始時刻や制限時間が負の場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "3000");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    }
    catch (Exception e) {
      NG(e);
      return;
    }


    MyConnection conn = exec.getQueryConnection();

    MSG("- コネクションの制限時刻>0、開始時刻<=0, 制限時間>0の場合。");

    CONNECTION_LIMIT_TIME = 12345L;
    EQUAL(conn.getLimitTimeMillis(), 12345L);
    EQUAL(exec.calcLimitTimeMillis(0L), 12345L);
    EQUAL(exec.calcLimitTimeMillis(-1L), 12345L);

    MSG("- コネクションの制限時刻>0、開始時刻>0, 制限時間<=0の場合。");

    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "0000");

    EQUAL(exec.calcLimitTimeMillis(222L), 12345L);

    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "-1");
    EQUAL(exec.calcLimitTimeMillis(222L), 12345L);

    MSG("- コネクションの制限時刻<=0、開始時刻>0, 制限時間<=0の場合。");

    CONNECTION_LIMIT_TIME = 0L;
    EQUAL(conn.getLimitTimeMillis(), 0L);
    EQUAL(exec.calcLimitTimeMillis(222L), 0L);

    CONNECTION_LIMIT_TIME = -5L;
    EQUAL(conn.getLimitTimeMillis(), -5L);
    EQUAL(exec.calcLimitTimeMillis(222L), -5L);

    MSG("- コネクションの制限時刻<=0、開始時刻<=0, 制限時間<=0の場合。");

    CONNECTION_LIMIT_TIME = 0L;
    EQUAL(conn.getLimitTimeMillis(), 0L);
    EQUAL(exec.calcLimitTimeMillis(0L), 0L);
    EQUAL(exec.calcLimitTimeMillis(-1L), 0L);

    CONNECTION_LIMIT_TIME = -5L;
    EQUAL(conn.getLimitTimeMillis(), -5L);
    EQUAL(exec.calcLimitTimeMillis(0L), -5L);
    EQUAL(exec.calcLimitTimeMillis(-1L), -5L);

    MSG("- コネクションの制限時刻<=0、開始時刻>0, 制限時間>0の場合。");

    execCfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "3000");

    CONNECTION_LIMIT_TIME = 0L;
    EQUAL(conn.getLimitTimeMillis(), 0L);
    EQUAL(exec.calcLimitTimeMillis(222L), 3222L);

    CONNECTION_LIMIT_TIME = -5L;
    EQUAL(conn.getLimitTimeMillis(), -5L);
    EQUAL(exec.calcLimitTimeMillis(222L), 3222L);

    MSG("- コネクションの制限時刻=0、開始時刻<=0, 制限時間>0の場合。");

    CONNECTION_LIMIT_TIME = 0L;
    EQUAL(conn.getLimitTimeMillis(), 0L);
    EQUAL(exec.calcLimitTimeMillis(0L), 0L);

    CONNECTION_LIMIT_TIME = -5L;
    EQUAL(conn.getLimitTimeMillis(), -5L);
    EQUAL(exec.calcLimitTimeMillis(0L), -5L);

    CONNECTION_LIMIT_TIME = 0L;
    EQUAL(conn.getLimitTimeMillis(), 0L);
    EQUAL(exec.calcLimitTimeMillis(-1L), 0L);

    CONNECTION_LIMIT_TIME = -5L;
    EQUAL(conn.getLimitTimeMillis(), -5L);
    EQUAL(exec.calcLimitTimeMillis(-1L), -5L);
  }

  public void calcLimitFetchCount()
  {
    MSG("取得データ件数の制限値を算出するメソッドの確認。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "1000");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    EQUAL(exec.calcLimitFetchCount(), 1000);

    execCfg.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "  100  ");
    EQUAL(exec.calcLimitFetchCount(), 100);

    execCfg.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "0");
    EQUAL(exec.calcLimitFetchCount(), 0);

    execCfg.getResource().setFirstValue("ts-query.execution.limit.fetchcount",
      "-1");
    EQUAL(exec.calcLimitFetchCount(), -1);
  }

  public void checkTimeout_content()
  {
    MSG("タイムアウトの判定を行うメソッドの確認。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    content.setLimitTimeMillis(System.currentTimeMillis() -1L);

    try {
      exec.checkTimeout(content);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryExecution.Error.Timeout);
    }

    content.setLimitTimeMillis(System.currentTimeMillis() + 1000L);

    try {
      exec.checkTimeout(content);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void checkTimeout_content_ZeroOrNegative()
  {
    MSG("制限時刻がゼロ以下の場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    content.setLimitTimeMillis(0L);

    try {
      exec.checkTimeout(content);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    content.setLimitTimeMillis(-1L);

    try {
      exec.checkTimeout(content);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    content.setLimitTimeMillis(-100L);

    try {
      exec.checkTimeout(content);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void checkFetch_content_count()
  {
    MSG("取得データ件数オーバーの判定を行うメソッドの確認。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    content.setLimitFetchCount(100);
    content.addOutput(new QueryExecutionOutput("AAA"));
    TRUE(content.hasResultTable());

    try {
      exec.checkFetchCount(content, 101);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryExecution.Error.FetchCountOver);
    }

    try {
      exec.checkFetchCount(content, 100);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 99);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 0);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, -1);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void checkFetch_content_count_ZeroOrNegative()
  {
    MSG("制限件数がゼロ又は負の場合。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    content.setLimitFetchCount(0);
    content.addOutput(new QueryExecutionOutput("AAA"));
    TRUE(content.hasResultTable());

    try {
      exec.checkFetchCount(content, 101);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 100);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 99);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 0);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, -1);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    content.setLimitFetchCount(-10);
    content.addOutput(new QueryExecutionOutput("AAA"));
    TRUE(content.hasResultTable());

    try {
      exec.checkFetchCount(content, 101);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 100);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 99);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, 0);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.checkFetchCount(content, -1);
      OK();
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void disposeContent_content_result()
  {
    MSG("実行内容オブジェクトの後始末をするメソッドの確認。");
    MSG("- 結果テーブルを持っている場合は実行結果オブジェクトに設定。");

    QueryExecutionConfig execCfg = new QueryExecutionConfig();
    execCfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    execCfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(execCfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    QueryResult rslt = new QueryResult("q0");

    content.setExecutionId("e0");
    FALSE(content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    NULL(rslt.getResultTable("e0"));
    EQUAL(rslt.getAllResultIds().size(), 0);

    Table<String,Serializable> t0 = exec.newResultTable(content);
    content.setResultTable(t0);
    FALSE(content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    NULL(rslt.getResultTable("e0"));
    EQUAL(rslt.getAllResultIds().size(), 0);

    content.setFetchCount(1);
    FALSE(content.hasResultTable());
    EQUAL(content.getFetchCount(), 1);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    NULL(rslt.getResultTable("e0"));
    EQUAL(rslt.getAllResultIds().size(), 0);

    content.setUpdateCount(10);
    FALSE(content.hasResultTable());
    EQUAL(content.getFetchCount(), 1);
    EQUAL(content.getUpdateCount(), 10);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    NULL(rslt.getResultTable("e0"));
    EQUAL(rslt.getAllResultIds().size(), 0);

    content.setFetchCount(0);
    content.setUpdateCount(0);
    FALSE(content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    NULL(rslt.getResultTable("e0"));
    EQUAL(rslt.getAllResultIds().size(), 0);

    content.addOutput(new QueryExecutionOutput("AAA"));
    TRUE (content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    EQUAL(rslt.getResultTable("e0"), t0);
    EQUAL(rslt.getAllResultIds(), Arrays.asList("e0"));

    content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    Table<String,Serializable> t1 = exec.newResultTable(content);
    content.setExecutionId("e7");
    content.setResultTable(t1);
    content.addOutput(new QueryExecutionOutput("BBB"));
    TRUE (content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    EQUAL(rslt.getResultTable("e0"), t0);
    EQUAL(rslt.getResultTable("e7"), t1);
    EQUAL(rslt.getAllResultIds(), Arrays.asList("e0", "e7"));

    content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    Table<String,Serializable> t2 = exec.newResultTable(content);
    content.setExecutionId("e3");
    content.setResultTable(t2);
    content.addOutput(new QueryExecutionOutput("CCC"));
    TRUE (content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    EQUAL(rslt.getResultTable("e0"), t0);
    EQUAL(rslt.getResultTable("e7"), t1);
    EQUAL(rslt.getResultTable("e3"), t2);
    EQUAL(rslt.getAllResultIds(), Arrays.asList("e0", "e7", "e3"));

    content = new QueryExecutionContent();
    content.setExecutionId(exec.getConfig().getExecutionId());
    content.setConnectionId(exec.getQueryConnection().getConnectionId());
    Table<String,Serializable> t3 = exec.newResultTable(content);
    content.setExecutionId("e7");
    content.setResultTable(t3);
    content.addOutput(new QueryExecutionOutput("DDD"));
    TRUE (content.hasResultTable());
    EQUAL(content.getFetchCount(), 0);
    EQUAL(content.getUpdateCount(), 0);
    TRUE (rslt.isSuccess());
    exec.disposeContent(content, rslt);
    EQUAL(rslt.getResultTable("e0"), t0);
    EQUAL(rslt.getResultTable("e7"), t3);
    EQUAL(rslt.getResultTable("e3"), t2);
    EQUAL(rslt.getAllResultIds(), Arrays.asList("e0", "e7", "e3"));
  }

  public void execute_inputMap()
  {
    MSG("クエリ実行内容を実行するメソッドの確認。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();

    MSG("- 結果データ・テーブルなし。");

    IQueryResult rslt = null;
    try {
      rslt = exec.execute(inputMap);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
    EQUAL(rslt.getQueryId(), "");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    MSG("- 結果データ・テーブルあり。");

    HAS_RESULT_TABLE = true;

    try {
      rslt = exec.execute(inputMap);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
    EQUAL(rslt.getQueryId(), "");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 1);
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    NOTNULL(rslt.getResultTable(""));
  }

  public void execute_inputMap_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    IQueryResult rslt = null;
    try {
      rslt = exec.execute(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void execute_inputMap_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");
    cfg.getResource().setFirstValue("ts-query.execution.limit.spenttime", "1");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();

    IQueryResult rslt = null;
    try {
      rslt = exec.execute(inputMap);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryExecution.Error.Timeout);
    }
  }

  public void execute_inputMap_FailToExecute()
  {
    MSG("実行中にエラーが発生した場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();

    FAIL_TO_EXECUTE = true;

    IQueryResult rslt = null;
    try {
      rslt = exec.execute(inputMap);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ENUM.AAA);
    }
  }
  
  public void execute_inputMap_result()
  {
    MSG("実行内容を実行するメソッドの確認。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();

    MSG("- 結果データ・テーブルあり。");

    IQueryResult rslt = new QueryResult("q0");
    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    MSG("- 結果データ・テーブルあり。");

    HAS_RESULT_TABLE = true;

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 1);
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    NOTNULL(rslt.getResultTable(""));
  }

  public void execute_inputMap_result_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    try {
      exec.execute(null, rslt);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    try {
      exec.execute(inputMap, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e.toString());
    }
  }

  public void execute_inputMap_result_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");
    cfg.getResource().setFirstValue("ts-query.execution.limit.spenttime", "1");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    try {
      exec.execute(inputMap, rslt);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryExecution.Error.Timeout);
    }
  }

  public void execute_inputMap_result_FailToExecute()
  {
    MSG("実行中にエラーが発生した場合。");

    QueryExecutionConfig cfg = new QueryExecutionConfig();
    cfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.QueryExecutionTest$MyExecution");
    cfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "QueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(cfg);
    } catch (Exception e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    FAIL_TO_EXECUTE = true;

    try {
      exec.execute(inputMap, rslt);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ENUM.AAA);
    }
  }
}
