/*
 * QueryTransactionManager class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

/**
 * トランザクション・オブジェクトの作成や取得、破棄を管理するためのクラス。
 *
 * @author 佐藤隆之
 * @version $Id: QueryTransactionManager.java,v 1.25 2012-03-09 16:13:05 tayu Exp $
 */
public class QueryTransactionManager implements IQueryTransactionManager
{
  /**
   * スレッド・ローカルなトランザクション・オブジェクトを格納する{@link
   * ThreadLocal}オブジェクト。
   */
  private final static ThreadLocal<IQueryTransaction> THREAD_LOCAL =
    new ThreadLocal<IQueryTransaction>();

  /**
   * スレッド・セーフなトランザクション・オブジェクトを格納するためのマップ・
   * オブジェクト。
   */
  private final static Map<String,IQueryTransaction> TRANSACTION_MAP =
    new HashMap<String,IQueryTransaction>();

  /**
   * デフォルト・コンストラクタ。
   * <br>
   * このクラスはインスタンス化して使用することはないため、アクセス指定子を
   * <tt>private</tt>に指定している。
   */
  QueryTransactionManager()
  {}

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction createThreadLocalTransaction()
    throws ReasonedRuntimeException
  {
    return createThreadLocalTransaction(QueryTransaction.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction createThreadLocalTransaction(
    Class<? extends IQueryTransaction> tranClass
  ) throws ReasonedRuntimeException
  {
    IQueryTransaction tran = THREAD_LOCAL.get();

    if (tran != null && tran.getState() != IQueryTransaction.State.Ended) {
      throw new ReasonedRuntimeException(
        IQueryTransactionManager.Error.ThreadLocalAlreadyExists,
        "[transaction class=" + tran.getClass().getName() + "]");
    }

    if (tranClass == null) {
      throw new ReasonedRuntimeException(
        IQueryTransactionManager.Error.FailToCreateThreadLocal,
        "[transaction class=null]");
    }

    try {
      THREAD_LOCAL.set(tranClass.newInstance());
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(
        IQueryTransactionManager.Error.FailToCreateThreadLocal,
        "[transaction class=" + tranClass.getName() + "]", e);
    }

    return new ThreadLocalTransaction();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction getThreadLocalTransaction()
  {
    return new ThreadLocalTransaction();
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction createThreadSafeTransaction(String key)
    throws ReasonedRuntimeException
  {
    return createThreadSafeTransaction(key, QueryTransaction.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction createThreadSafeTransaction(
    String key, Class<? extends IQueryTransaction>  tranClass
  ) throws ReasonedRuntimeException
  {
    assert (key != null && tranClass != null) :
      (key == null) ? "@param:key is null." :
      (tranClass == null) ? "@param:tranClass is null." : "";

    synchronized (TRANSACTION_MAP) {
      IQueryTransaction tran = TRANSACTION_MAP.get(key);
      if (tran != null && tran.getState() != IQueryTransaction.State.Ended) {
        throw new ReasonedRuntimeException(
          IQueryTransactionManager.Error.ThreadSafeAlreadyExists,
          "[transaction class=" + tran.getClass().getName() + "]");
      }

      try {
        TRANSACTION_MAP.put(key, tranClass.newInstance());
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(
          IQueryTransactionManager.Error.FailToCreateThreadSafe,
          "[transaction class=" + tranClass.getName() + "]", e);
      }

      return new ThreadSafeTransaction(key);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryTransaction getThreadSafeTransaction(String key)
  {
    return new ThreadSafeTransaction(key);
  }


  /**
   * トランザクション・オブジェクトをスレッド・ローカル化するためのラップ・
   * クラス。
   */
  private static final class ThreadLocalTransaction implements IQueryTransaction
  {
    /**
     * 基になるトランザクション・オブジェクト。
     * このオブジェクトの終了時に基のトランザクション・オブジェクトはスレッド・
     * ローカル・オブジェクトから削除され、そこから取得できなくなってしまうの
     * で、この変数に退避して使用する。
     */
    private final IQueryTransaction transaction;

    /**
     * デフォルト・コンストラクタ。
     *
     * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
     *   存在しない場合。
     */
    ThreadLocalTransaction() throws ReasonedRuntimeException
    {
      IQueryTransaction tran = THREAD_LOCAL.get();
      if (tran == null) {
        throw new ReasonedRuntimeException(
          IQueryTransactionManager.Error.ThreadLocalNotExist);
      }
      this.transaction = tran;
    }

    /**
     * 基になったトランザクション・オブジェクトを取得する。
     *
     * @return 基になったトランザクション・オブジェクト。
     * @throws ReasonedRuntimeException このオブジェクトが保持している基に
     *   なったトランザクション・オブジェクトが既に破棄されている場合。
     */
    IQueryTransaction getInner()
    {
      if (this.transaction == THREAD_LOCAL.get()) {
        return this.transaction;
      }
      else if (this.transaction.getState() == IQueryTransaction.State.Ended) {
        return this.transaction;
      }
      else if (THREAD_LOCAL.get() == null) {
        throw new ReasonedRuntimeException(
          IQueryTransactionManager.Error.ThreadLocalNotExist);
      }
      else {
        throw new ReasonedRuntimeException(
          IQueryTransactionManager.Error.ThreadLocalAlreadyDiscarded);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin() throws ReasonedException, ReasonedRuntimeException
    {
      getInner().begin();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void commit() throws ReasonedException, ReasonedRuntimeException
    {
      getInner().commit();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void rollback() throws ReasonedRuntimeException
    {
      getInner().rollback();
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void end() throws ReasonedRuntimeException
    {
      IQueryTransaction tran = getInner();
      THREAD_LOCAL.remove();
      tran.end();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public State getState()
    {
      return getInner().getState();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getBeginTimeMillis()
    {
      return getInner().getBeginTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getLimitTimeMillis()
    {
      return getInner().getLimitTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getTimeoutMillis()
    {
      return getInner().getTimeoutMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setTimeoutMillis(long millis)
    {
      getInner().setTimeoutMillis(millis);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryHistory getQueryHistory()
    {
      return getInner().getQueryHistory();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryConnection getQueryConnection(String connId)
      throws ReasonedException, ReasonedRuntimeException
    {
      return getInner().getQueryConnection(connId);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryExecution getQueryExecution(String execId)
      throws ReasonedException, ReasonedRuntimeException
    {
      return getInner().getQueryExecution(execId);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQuery getQuery(String queryId)
      throws ReasonedException, ReasonedRuntimeException
    {
      return getInner().getQuery(queryId);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode()
    {
      return getInner().hashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj)
    {
      if (obj == null) {
        return false;
      }
      else if (! (obj instanceof ThreadLocalTransaction)) {
        return false;
      }
      else {
        ThreadLocalTransaction tran = ThreadLocalTransaction.class.cast(obj);
        return getInner().equals(tran.getInner());
      }
    }
  }

  /**
   * トランザクション・オブジェクトをスレッド・セーフ化するためのラップ・
   * クラス。
   */
  private static final class ThreadSafeTransaction implements IQueryTransaction
  {
    /** このトランザクション・オブジェクトに結びつけられたキー。 */
    private final String key;

    /** 
     * 基になるトランザクション・オブジェクト。
     * このオブジェクトの終了時に基のトランザクション・オブジェクトは
     * トランザクション・マップから削除され、そこから取得できなくなってしまう
     * ので、この変数に退避して使用する。
     */
    private final IQueryTransaction transaction;

    /**
     * 結びつけるキーを引数にとるコンストラクタ。
     *
     * @param key キー。
     * @throws ReasonedRuntimeException 指定されたキーに結びつけられたスレッド・
     *   セーフなトランザクションが存在しない場合。
     */
    ThreadSafeTransaction(String key) throws ReasonedRuntimeException
    {
      synchronized (TRANSACTION_MAP) {
        IQueryTransaction tran = TRANSACTION_MAP.get(key);
        if (tran == null) {
          throw new ReasonedRuntimeException(
            IQueryTransactionManager.Error.ThreadSafeNotExist, "[key="+key+"]");
        }
        this.key = key;
        this.transaction = tran;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin() throws ReasonedException, ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        this.transaction.begin();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void commit() throws ReasonedException, ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        this.transaction.commit();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void rollback() throws ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        this.transaction.rollback();
      }
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void end() throws ReasonedRuntimeException
    {
      synchronized (TRANSACTION_MAP) {
        IQueryTransaction tran = TRANSACTION_MAP.get(this.key);
        if (tran != null && tran.equals(this.transaction)) {
          TRANSACTION_MAP.remove(this.key);
        }
      }

      synchronized (this.transaction) {
        this.transaction.end();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public State getState()
    {
      synchronized (this.transaction) {
        return this.transaction.getState();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getBeginTimeMillis()
    {
      synchronized (this.transaction) {
        return this.transaction.getBeginTimeMillis();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getLimitTimeMillis()
    {
      synchronized (this.transaction) {
        return this.transaction.getLimitTimeMillis();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setTimeoutMillis(long millis)
    {
      synchronized (this.transaction) {
        this.transaction.setTimeoutMillis(millis);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getTimeoutMillis()
    {
      synchronized (this.transaction) {
        return this.transaction.getTimeoutMillis();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryHistory getQueryHistory()
    {
      return new ThreadSafeQueryHistory(
        this.transaction.getQueryHistory());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryConnection getQueryConnection(String connId)
      throws ReasonedException, ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        return this.transaction.getQueryConnection(connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQueryExecution getQueryExecution(String execId)
      throws ReasonedException, ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        return this.transaction.getQueryExecution(execId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IQuery getQuery(String queryId)
      throws ReasonedException, ReasonedRuntimeException
    {
      synchronized (this.transaction) {
        return this.transaction.getQuery(queryId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode()
    {
      return this.transaction.hashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj)
    {
      if (obj == null) {
        return false;
      }
      else if (! (obj instanceof ThreadSafeTransaction)) {
        return false;
      }
      else {
        ThreadSafeTransaction tran = ThreadSafeTransaction.class.cast(obj);
        return this.transaction.equals(tran.transaction);
      }
    }
  }

  /**
   * クエリ実行履歴オブジェクトをスレッド・セーフ化するためのラップ・クラス。
   */
  private static final class ThreadSafeQueryHistory implements IQueryHistory
  {
    /** シリアル・バージョン番号。 */
    static final long serialVersionUID = 2962083979616856555L;

    /** 基になるクエリ実行履歴オブジェクト。 */
    private final IQueryHistory inner;

    /**
     * 基になるクエリ実行履歴オブジェクトを引数にとるコンストラクタ。
     *
     * @param rsltLst 基になるクエリ実行履歴オブジェクト。
     */
    ThreadSafeQueryHistory(IQueryHistory history)
    {
      this.inner = history;
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public Map<Item,Serializable> appendNew()
    {
      synchronized (this.inner) {
        return this.inner.appendNew();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getAll()
    {
      synchronized (this.inner) {
        return this.inner.getAll();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByQueryId(String queryId)
    {
      synchronized (this.inner) {
        return this.inner.getByQueryId(queryId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByConnectionId(String connId)
    {
      synchronized (this.inner) {
        return this.inner.getByConnectionId(connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByConnectionId(String queryId,
      String connId)
    {
      synchronized (this.inner) {
        return this.inner.getByConnectionId(queryId, connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByExecutionId(String execId)
    {
      synchronized (this.inner) {
        return this.inner.getByExecutionId(execId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByExecutionId(String connId,
      String execId)
    {
      synchronized (this.inner) {
        return this.inner.getByExecutionId(connId, execId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByExecutionId(String queryId,
      String connId, String execId)
    {
      synchronized (this.inner) {
        return this.inner.getByExecutionId(queryId, connId, execId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map<Item,Serializable>> getByIsSuccess(boolean isSuccess)
    {
      synchronized (this.inner) {
        return this.inner.getByIsSuccess(isSuccess);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map<Item,Serializable> getByException(ReasonedException exc)
    {
      synchronized (this.inner) {
        return this.inner.getByException(exc);
      }
    }
  }
}
