package jp.sourceforge.tsukuyomi.openid.discovery;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class UrlIdentifier implements Identifier {
	private static final long serialVersionUID = 1740585702985022532L;

	private static final Log LOG = LogFactory.getLog(UrlIdentifier.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	private static final Set<Character> UNRESERVED_CHARACTERS =
		new HashSet<Character>();

	static {
		for (char c = 'a'; c <= 'z'; c++) {
			UNRESERVED_CHARACTERS.add(c);
		}
		for (char c = 'A'; c <= 'A'; c++) {
			UNRESERVED_CHARACTERS.add(c);
		}
		for (char c = '0'; c <= '9'; c++) {
			UNRESERVED_CHARACTERS.add(c);
		}

		UNRESERVED_CHARACTERS.add(Character.valueOf('-'));
		UNRESERVED_CHARACTERS.add(Character.valueOf('.'));
		UNRESERVED_CHARACTERS.add(Character.valueOf('_'));
		UNRESERVED_CHARACTERS.add(Character.valueOf('~'));
	}

	private URL urlIdentifier;

	public UrlIdentifier(String identifier) throws IdentifierException {
		urlIdentifier = normalize(identifier);
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		final UrlIdentifier that = (UrlIdentifier) o;

		return urlIdentifier.equals(that.urlIdentifier);
	}

	@Override
	public int hashCode() {
		return urlIdentifier.hashCode();
	}

	public String getIdentifier() {
		return urlIdentifier.toExternalForm();
	}

	@Override
	public String toString() {
		return urlIdentifier.toExternalForm();
	}

	public URL getUrl() {
		return urlIdentifier;
	}

	public static URL normalize(String text) throws IdentifierException {
		try {
			URI uri = new URI(text);
			URL url = uri.normalize().toURL();

			String protocol = url.getProtocol().toLowerCase();
			String host = url.getHost().toLowerCase();
			int port = url.getPort();
			String path = normalizeUrlEncoding(url.getPath());
			String query = normalizeUrlEncoding(url.getQuery());
			String fragment = normalizeUrlEncoding(url.getRef());

			if (port == url.getDefaultPort()) {
				port = -1;
			}

			// start building the 'file' part for the URL constructor...
			String file = path;

			if ("".equals(file)) {
				file = "/";
			}

			if (query != null) {
				file = file + "?" + query;
			}

			if (fragment != null) {
				file = file + "#" + fragment;
			}

			URL normalized = new URL(protocol, host, port, file);

			if (DEBUG) {
				LOG.debug("Normalized: " + text + " to: " + normalized);
			}

			return normalized;
		} catch (MalformedURLException e) {
			throw new IdentifierException("Invalid URL identifier", e);
		} catch (URISyntaxException e) {
			throw new IdentifierException("Invalid URL identifier", e);
		}

	}

	private static String normalizeUrlEncoding(String text) {
		if (text == null) {
			return null;
		}

		int len = text.length();
		StringBuffer normalized = new StringBuffer(len);

		for (int i = 0; i < len; i++) {
			char current = text.charAt(i);

			if (current == '%' && i < len - 2) {
				String percentCode = text.substring(i, i + 3).toUpperCase();

				try {
					String str = URLDecoder.decode(percentCode, "ISO-8859-1");
					char chr = str.charAt(0);

					if (UNRESERVED_CHARACTERS.contains(chr)) {
						normalized.append(chr);
					} else {
						normalized.append(percentCode);
					}
				} catch (UnsupportedEncodingException e) {
					normalized.append(percentCode);
				}

				i += 2;
			} else {
				normalized.append(current);
			}
		}

		return normalized.toString();
	}
}
