/*
 * Copyright 2006-2007 Sxip Identity Corporation
 */

package jp.sourceforge.tsukuyomi.openid.discovery.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Arrays;
import java.util.List;

import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryException;
import jp.sourceforge.tsukuyomi.openid.discovery.HtmlResolver;
import jp.sourceforge.tsukuyomi.openid.discovery.HtmlResult;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierException;
import jp.sourceforge.tsukuyomi.openid.discovery.UrlIdentifier;
import jp.sourceforge.tsukuyomi.openid.http.HttpClientManager;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.htmlparser.Node;
import org.htmlparser.Parser;
import org.htmlparser.filters.TagNameFilter;
import org.htmlparser.nodes.TagNode;
import org.htmlparser.util.NodeIterator;
import org.htmlparser.util.NodeList;
import org.htmlparser.util.ParserException;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class HtmlResolverImpl implements HtmlResolver {
	private static final Log LOG = LogFactory.getLog(HtmlResolverImpl.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();
	private HttpClientManager httpClientManager;

	/**
	 * Performs HTML discovery on the supplied URL identifier.
	 * 
	 * @param identifier
	 *            The URL identifier.
	 * @return HTML discovery data obtained from the URL.
	 * @throws IdentifierException
	 */
	public HtmlResult discover(UrlIdentifier identifier)
			throws DiscoveryException, IdentifierException {
		// initialize the results of the HTML discovery
		HtmlResult result = new HtmlResult();

		// get the HTML data (and set the claimed identifier)
		String htmlData = call(identifier.getUrl(), result);

		parseHtml(htmlData, result);

		LOG.info("HTML discovery succeeded on: " + identifier);

		return result;
	}

	/**
	 * Performs a HTTP call on the provided URL identifier.
	 * 
	 * @param url
	 *            The URL identifier.
	 * @param result
	 *            The HTML discovery result, in which the claimed identifier is
	 *            set to the input URL after following redirects.
	 * @return The retrieved HTML data.
	 * @throws IdentifierException
	 */
	private String call(URL url, HtmlResult result) throws DiscoveryException,
			IdentifierException {
		HttpClient client = httpClientManager.getHttpClient();

		GetMethod get = new GetMethod(url.toString());
		get.setFollowRedirects(true);

		try {
			if (DEBUG) {
				LOG.debug("Fetching " + url + "...");
			}

			int statusCode = client.executeMethod(get);
			if (statusCode != HttpStatus.SC_OK) {
				throw new DiscoveryException("GET failed on "
					+ url
					+ " Received status code: "
					+ statusCode);
			}

			result.setClaimed(new UrlIdentifier(get.getURI().toString()));

			InputStream htmlInput = get.getResponseBodyAsStream();
			if (htmlInput == null) {
				throw new DiscoveryException(
					"Cannot open inputstream for GET response from " + url);
			}

			byte data[] = new byte[httpClientManager.getMaxHtmlSize()];

			int totalRead = 0;
			int currentRead = 0;
			while (totalRead < httpClientManager.getMaxHtmlSize()) {
				currentRead =
					htmlInput.read(data, totalRead, httpClientManager
						.getMaxHtmlSize()
						- totalRead);

				if (currentRead == -1) {
					break;
				}

				totalRead += currentRead;
			}

			htmlInput.close();

			if (totalRead <= 0) {
				throw new DiscoveryException("No HTML data read from " + url);
			}

			if (DEBUG) {
				LOG.debug("Read " + totalRead + " bytes.");
			}

			return new String(data, 0, totalRead);

		} catch (IOException e) {
			throw new DiscoveryException("Fatal transport error: ", e);
		} finally {
			get.releaseConnection();
		}
	}

	/**
	 * Parses the HTML data and stores in the result the discovered openid
	 * information.
	 * 
	 * @param htmlData
	 *            HTML data obtained from the URL identifier.
	 * @param result
	 *            The HTML result.
	 */
	private void parseHtml(String htmlData, HtmlResult result)
			throws DiscoveryException {
		URL idp1Endpoint = null;
		URL idp2Endpoint = null;

		if (DEBUG) {
			LOG.debug("Parsing HTML data:\n" + htmlData);
		}

		try {
			Parser parser = Parser.createParser(htmlData, null);

			NodeList heads = parser.parse(new TagNameFilter("HEAD"));
			if (heads.size() != 1) {
				throw new DiscoveryException(
					"HTML response must have exactly one HEAD element, "
						+ "found "
						+ heads.size()
						+ " : "
						+ heads.toHtml());
			}
			Node head = heads.elementAt(0);
			for (NodeIterator i = head.getChildren().elements(); i
				.hasMoreNodes();) {
				Node node = i.nextNode();
				if (node instanceof TagNode) {
					TagNode link = (TagNode) node;
					String href = link.getAttribute("href");

					String rel = link.getAttribute("rel");
					if (rel == null) {
						continue;
					}
					List<String> relations = Arrays.asList(rel.split(" "));
					if (relations == null) {
						continue;
					}

					if (relations.contains("openid.server")) {
						if (result.getIdp1Endpoint() != null) {
							throw new DiscoveryException(
								"More than one openid.server entries found");
						}

						if (DEBUG) {
							LOG
								.debug("Found OpenID1 endpoint: "
									+ idp1Endpoint);
						}

						result.setEndpoint1(href);
					}

					if (relations.contains("openid.delegate")) {
						if (result.getDelegate1() != null) {
							throw new DiscoveryException(
								"More than one openid.delegate entries found");
						}

						if (DEBUG) {
							LOG.debug("Found OpenID1 delegate: " + href);
						}

						result.setDelegate1(href);
					}
					if (relations.contains("openid2.provider")) {
						if (result.getIdp2Endpoint() != null) {
							throw new DiscoveryException(
								"More than one openid.server entries found");
						}

						if (DEBUG) {
							LOG
								.debug("Found OpenID2 endpoint: "
									+ idp2Endpoint);
						}

						result.setEndpoint2(href);
					}

					if (relations.contains("openid2.local_id")) {
						if (result.getDelegate2() != null) {
							throw new DiscoveryException(
								"More than one openid2.local_id entries found");
						}

						if (DEBUG) {
							LOG.debug("Found OpenID2 localID: " + href);
						}

						result.setDelegate2(href);
					}
				}
			}

			if (DEBUG) {
				LOG.debug("HTML discovery result:\n" + result);
			}
		} catch (ParserException e) {
			throw new DiscoveryException("Error parsing HTML message", e);
		}
	}

	public void setHttpClientManager(HttpClientManager httpClientManager) {
		this.httpClientManager = httpClientManager;
	}
}
