/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.examples.modules.minicomposer;

import java.io.*;
import java.util.*;

import org.openide.ErrorManager;
import org.openide.TopManager;
import org.openide.util.NbBundle;

public final class Score implements Serializable {
    private static final long serialVersionUID = 76675765298773L;
    /** Names in text for the tones. */
    public static final String[] TONES_SHORT = new String[] {
        "-", "C", "D", "d", "E", "e", "F", "G", "g", "A", "a", "B", "b"
    };
    private static final String[] KEYS4_TONES_LONG = new String[] {
        "rest", "c", "d", "d_flat", "e", "e_flat", "f", "g", "g_flat", "a", "a_flat", "b", "b_flat"
    };
    /** Display names of the tones. */
    public static final String[] TONES_LONG = new String[KEYS4_TONES_LONG.length];
    /** Default tone to start with in a GUI. */
    public static final int DEFAULT_TONE = 1;
    /** Names in text for the octaves. */
    public static final String[] OCTAVES_SHORT = new String[] {
        "--", "-", ".", "+", "++"
    };
    private static final String[] KEYS4_OCTAVES_LONG = new String[] {
        "low", "middle_low", "middle", "middle_high", "high"
    };
    /** Display names of octaves. */
    public static final String[] OCTAVES_LONG = new String[KEYS4_OCTAVES_LONG.length];
    /** Default octave to display. */
    public static final int DEFAULT_OCTAVE = 2;
    /** Frequency of middle C. */
    public static final float MIDDLE_C_HERTZ = 440.0f;
    /** Position of C in the tone list. */
    public static final int WHERE_IS_C_TONE = 1;
    /** Position of the middle octave in the octave list. */
    public static final int WHERE_IS_MIDDLE_OCTAVE = 2;
    /** Frequency increment of one half-tone. */
    public static final float HALF_STEP = (float)Math.pow(2.0, 1.0 / 12.0);
    /** Names in text for durations. */
    public static final String[] DURATIONS_SHORT = new String[] {
        "1", "2", "4"
    };
    private static final String[] KEYS4_DURATIONS_LONG = new String[] {
        "quarter", "half", "full"
    };
    /** Display names of durations. */
    public static final String[] DURATIONS_LONG = new String[KEYS4_DURATIONS_LONG.length];
    /** Lengths of durations in seconds. */
    public static final float[] DURATION_SECONDS = new float[] {
        0.25f, 0.5f, 1.0f
    };
    /** Index of default duration to select. */
    public static final int DEFAULT_DURATION = 0;
    static {
        ResourceBundle bundle = NbBundle.getBundle(Score.class);
        for (int i = 0; i < TONES_LONG.length; i++)
            TONES_LONG[i] = bundle.getString("TONE_" + KEYS4_TONES_LONG[i]);
        for (int i = 0; i < OCTAVES_LONG.length; i++)
            OCTAVES_LONG[i] = bundle.getString("OCTAVE_" + KEYS4_OCTAVES_LONG[i]);
        for (int i = 0; i < DURATIONS_LONG.length; i++)
            DURATIONS_LONG[i] = bundle.getString("DURATION_" + KEYS4_DURATIONS_LONG[i]);
    }
    /** One note (or rest) in a score. */
    public static final class Note implements Serializable, Cloneable {
        private static final long serialVersionUID = 9674532758L;
        private final int tone;
        private final int octave;
        private final int duration;
        public Note(int tone, int octave, int duration) {
            this.tone = tone;
            this.octave = octave;
            this.duration = duration;
        }
        public int getTone() {
            return tone;
        }
        public int getOctave() {
            return octave;
        }
        public int getDuration() {
            return duration;
        }
        public Note cloneNote() {
            try {
                return (Note)clone();
            } catch (CloneNotSupportedException cnse) {
                throw new IllegalStateException(cnse.toString());
            }
        }
        public String toString() {
            return "Note@" + System.identityHashCode(this) + "[" + TONES_SHORT[tone] + "/" + OCTAVES_SHORT[octave] + "/" + DURATIONS_SHORT[duration] + "]";
        }
    }
    private final List notes; // List<Note>
    /** Create a new score programmatically.
     * Specify the tone, octave, and duration of each note.
     * @param notes a list of type Note
     */
    public Score(List notes) {
        this.notes = Collections.unmodifiableList(new ArrayList(notes));
    }
    /** Number of notes (and rests) in the score. */
    public int getSize() {
        return notes.size();
    }
    /** Get the note at a given position. */
    public Note getNote(int pos) throws IndexOutOfBoundsException {
        return (Note)notes.get(pos);
    }
    public boolean equals(Object o) {
        if (o == null || !(o instanceof Score)) return false;
        Score s = (Score)o;
        return notes.equals(((Score)o).notes);
    }
    public int hashCode() {
        return notes.hashCode();
    }
    public String toString() {
        return "Score[size=" + getSize() + "]";
    }
    /** Parse a score from a text stream. */
    public static Score parse(Reader r) throws IOException {
        BufferedReader reader = new BufferedReader(r);
        List notes = new LinkedList(); // List<Note>
        String line;
        while ((line = reader.readLine()) != null) {
            StringTokenizer tok = new StringTokenizer(line, "/");
            if (tok.hasMoreTokens()) {
                String toneToken = tok.nextToken();
                if (tok.hasMoreTokens()) {
                    String octaveToken = tok.nextToken();
                    if (tok.hasMoreTokens()) {
                        String durationToken = tok.nextToken();
                        if (tok.hasMoreTokens()) {
                            throw mkExc(NbBundle.getMessage(Score.class, "EXC_more_than_3_items_on_line", line));
                        } else {
                            int tone = find(toneToken, Score.TONES_SHORT);
                            if (tone == -1) throw mkExc(NbBundle.getMessage(Score.class, "EXC_unknown_tone", toneToken));
                            int octave = find(octaveToken, Score.OCTAVES_SHORT);
                            if (octave == -1) throw mkExc(NbBundle.getMessage(Score.class, "EXC_unknown_octave", octaveToken));
                            int duration = find(durationToken, Score.DURATIONS_SHORT);
                            if (duration == -1) throw mkExc(NbBundle.getMessage(Score.class, "EXC_unknown_duration", durationToken));
                            notes.add(new Note(tone, octave, duration));
                        }
                    } else {
                        throw mkExc(NbBundle.getMessage(Score.class, "EXC_only_2_items_on_line", line));
                    }
                } else {
                    throw mkExc(NbBundle.getMessage(Score.class, "EXC_only_one_item_on_line", line));
                }
            } else {
                throw mkExc(NbBundle.getMessage(Score.class, "EXC_no_items_on_line", line));
            }
        }
        return new Score(notes);
    }
    private static IOException mkExc(String message) {
        IOException ioe = new IOException(message);
        TopManager.getDefault().getErrorManager().annotate(ioe, ErrorManager.WARNING, null, message, null, null);
        return ioe;
    }
    /** Write a score to a text stream. */
    public static void generate(Score s, Writer w) throws IOException {
        int len = s.getSize();
        for (int i = 0; i < len; i++) {
            Note n = s.getNote(i);
            w.write(TONES_SHORT[n.getTone()]);
            w.write((int)'/');
            w.write(OCTAVES_SHORT[n.getOctave()]);
            w.write((int)'/');
            w.write(DURATIONS_SHORT[n.getDuration()]);
            w.write((int)'\n');
        }
    }
    private static int find(String key, String[] list) {
        for (int i = 0; i < list.length; i++)
            if (key.equals(list[i]))
                return i;
        return -1;
    }
}
