import threading,subprocess,os,signal,tempfile,itertools
import shoutbrowser,config
from favoritelistdb import FavoriteListDataBase

try:
    import urwid, urwid.raw_display
except:
    import sys
    print 'urwid module could not import!'
    print 'if urwid is not installed,please install!'
    if not config.is_module_installed('easy_install'):
        print 'you can not use "easy_install" ?'
        print 'please install them,and...'
    print 'type below'
    print 'easy_install urwid'
    sys.exit(-1)

version ='0.0.1'
Screen = urwid.raw_display.Screen

class ListBoxItemWalker(urwid.PollingListWalker):
    _super = None
    def __init__(self,contents,focus_callback):
        self._super = super(self.__class__,self)
        self._super.__init__(contents)
        self.focus_callback = focus_callback

    def _activate_item(self,widget,is_active):
        if widget:
            widget.activate(is_active)

    def set_focus(self,position):
        focus_out_item = self.get_focus()[0]
        self._activate_item(focus_out_item,False)
        self._super.set_focus(position)
        focus_in_item = self.get_focus()[0]
        self._activate_item(focus_in_item,True)
        self.focus_callback(focus_in_item)

class ListBoxItem(urwid.Text):
    __key_event_handler = {}
    def __init__(self, palette,label, user_data, key_event={}):
        self.__super = super(self.__class__,self)
        self.__super.__init__(label)
        self.__key_event_handler = key_event
        self.__palette = palette
        self.user_data = user_data
        
    def activate(self,is_active):
        if is_active:
            self.set_text(
                (self.__palette.ITEM_REVERSE,
                 self.get_text()[0])
                )
        else:
            self.set_text(self.get_text()[0])

    def selectable(self):
        return True

    def keypress(self, (maxcol,), key):
        handler = self.__key_event_handler.get(key,None)
        propagate_key = key
        if handler:
            handler(self.get_text()[0], self.user_data)

        return propagate_key

class StationListBox(object):
    def __init__(self,(screen,notify,palette,dialog,)):
        self._station_cache =[]
        self._station_info = {}
        self.dialog = dialog
        self.screen = screen
        self.notify = notify
        self.palette = palette
        self.__thread = None

    def __del__(self):
        if self.__thread:
            self.__thread.terminate()

    def add_station(self,name,id,info):
        self._station_cache.append((name,id))
        self._station_info[id]=info

    def _on_focus(self,item):
        if self.notify and item:
            self.notify.print_info(
                shoutbrowser.translate_info(
                    self._station_info[item.user_data])
                )

    def on_player_activate(self,name,id):
        PlayerDialog(self.dialog,self.notify).build(name,id)

    def build(self,event_handler):
        listbox =  urwid.ListBox(
            ListBoxItemWalker(
                [ListBoxItem(self.palette,name,id,
                             event_handler)
                 for name ,id in self._station_cache]
                ,self._on_focus)
            )
        listbox.set_focus(0)
        return urwid.Frame(urwid.LineBox( listbox ),
                           urwid.Text('Stations'))

class FavoriteStationListBox(StationListBox):
    def __init__(self,remove_event_callback,*args):
        self.__super = super(self.__class__,self)
        self.__remove_event_callback = remove_event_callback
        self.__super.__init__(args)

    def on_remove_favorite(self,name,id):
        self.notify.print_info('Remove from Favorite: %s' % name) 
        FavoriteListDataBase().remove_station(id)
        self.__remove_event_callback()

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'delete':self.on_remove_favorite,
                        'r':self.on_remove_favorite}
        return self.__super.build(event_handler)

class SearchedStationListBox(StationListBox):
    def __init__(self,*args):
        self.__super = super(self.__class__,self)
        self.__super.__init__(args)

    def on_add_favorite(self,name,id):
        self.notify.print_info('Add to Favorite: %s' % name)
        codec,bitrate,genre = self._station_info[id][0:3]
        FavoriteListDataBase().add_station(id,name,codec,bitrate,genre)

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'a':self.on_add_favorite}
        return self.__super.build(event_handler)

class GenreListBox:
    _selected_item = None
    def __init__(self,palette):
        self.palette = palette
    def __create_genre_widget(self,name,callback,active_genre):
        widget = urwid.Button(name,callback)
        if active_genre and name == active_genre:
            widget = urwid.AttrWrap(widget,self.palette.ITEM_REVERSE)
        return widget

    def build(self,callback,active=None):
        listbox = urwid.ListBox(
            [self.__create_genre_widget(name,callback,active)
             for name in shoutbrowser.major_genre])
        if active:
            listbox.set_focus(shoutbrowser.major_genre.index(active))
        return urwid.Frame(urwid.LineBox( listbox ),urwid.Text('Genres'))

class Notification:
    def __init__(self,palette):
        self.palette = palette

    def set_output(self,print_func):
        self.print_func = print_func

    def print_info(self,text):
        self._set_status(text,self.palette.MSG_INFO)

    def print_warn(self,text):
        self._set_status(text,self.palette.MSG_WARN)

    def print_error(self,text):
        self._set_status(text,self.palette.MSG_ERR)

    def _set_status(self,text,state):
        info = urwid.AttrWrap(
            urwid.Text(text),state)
        self.print_func(info)

    def clean(self):
        self.print_func(None)

class ColorPalette:
    MSG_INFO = 'info'
    MSG_WARN = 'warn'
    MSG_ERR  = 'error'
    MENU = 'menu'
    MENU_REVERSE ='menu_reverse'
    ITEM_REVERSE = 'reverse'
    TEXTBOX = 'tbox'
    DEFAULT = 'default'
    DIALOG = 'dialog'
    def __init__(self,screen):
        screen.register_palette([
                (self.DEFAULT, 'white', 'black', 'standout'),
                (self.DIALOG, 'black', 'light gray', 'standout'),
                (self.MENU, 'black', 'dark cyan', 'standout'),
                (self.MENU_REVERSE, 'white', 'dark blue',('bold','underline')),
                (self.ITEM_REVERSE, 'white', 'dark blue', 'bold'),
                (self.TEXTBOX,   'white', 'dark green','underline'),
                (self.MSG_INFO, 'light green', 'black', 'underline'),
                (self.MSG_WARN,  'yellow', 'black','bold'),
                (self.MSG_ERR,   'light red', 'black',('bold','underline'))
                ])

class Application(object):
    def __init__(self,*args):
        self.ui,self.view,self._notify,self._palette,self._dialog = args[0]

class GenreBrowser(Application):
    __thread = None
    def __init__(self,*args):
        super(self.__class__,self).__init__(args)

    def __del__(self):
        if self.__thread:
            self.__thread.terminate()

    def get_content(self,genre=''):
        stations = SearchedStationListBox(self.ui,self._notify,self._palette,
                                          self._dialog)
        if genre:
            try:
                shoutbrowser.get_stations(stations.add_station,genre)
            except:
                self._notify.print_warn('Requests are failed!')

        return urwid.Columns( 
            [ ('fixed',25,
               GenreListBox(self._palette).build(self.genre_callback,
                                               genre)),
              stations.build()
              ]
            )

    def __update_content(self,genre):
        self.view.set_body(self.get_content(genre))

    def genre_callback(self,button):
        genre = button.get_label()
        if not self.__thread:
            self.__thread = SimpleThread()
        self.__thread.start(self.__update_content,genre)
        self._notify.print_warn('Station-list of %s Downloading...' % genre)

class FavoriteStation(Application):
    def __init__(self,*args):
        super(self.__class__,self).__init__(args)

    def __update_content(self):
        self.view.set_body(self.get_content())

    def get_content(self):
        stations = FavoriteStationListBox(self.__update_content,
                                          self.ui,
                                          self._notify,
                                          self._palette,
                                          self._dialog)

        FavoriteListDataBase().get_stations(stations.add_station)
        return stations.build()

class StationSearch(Application):
    def __init__(self,*args):
        super(self.__class__,self).__init__(args)
        self.__thread = None
        self.__keyword = ''
        self.__entry = config.request_limit
        self.__bitrate = config.request_bitrate
        self.__codec = self._translate_codec(
            filter(lambda item:item[1] == config.request_codec,
                   shoutbrowser.mime_type.iteritems())[0][0])
    def __del__(self):
        if self.__thread:
            self.__thread.terminate()

    def _on_bitrate_toggled(self,radio,state,rate):
        if state:
            self.__bitrate = rate

    def _on_codec_toggled(self,radio,state,codec):
        if state:
            self.__codec = codec

    def __update_content(self):
        self.view.set_body(self.get_content(self.__keyword,
                                            self.__bitrate,
                                            self.__codec,
                                            self.__entry_box.value())
                           )
    def _verify_search_form(self):
        is_valid = False
        if not self.__keyword:
            self._notify.print_error('Please set keyword at least')
        elif not self.__entry:
            self._notify.print_error('Please set entry more than 0')
        else:
            is_valid = True
        return is_valid

    def _on_search(self,button):
        self.__entry = self.__entry_box.value()
        self.__keyword = self.__keyword_box.get_text()[0]
        if not self._verify_search_form():
            return

        if not self.__thread:
            self.__thread = SimpleThread()
        self.__thread.start(self.__update_content)
        self._notify.print_warn('Stations Searching...')

    def _create_field_label(self,text):
        return urwid.Text(text+':')

    def _create_radio(self,group,label,value,callback,state=False):
        return urwid.RadioButton(group,
                                 label,
                                 state,
                                 callback,
                                 value)

    def _build_keyword_box(self):
        self.__keyword_box = urwid.Edit(edit_text=self.__keyword)
        yield self._create_field_label('keyword')
        yield self._decorate_textbase_box(self.__keyword_box)

    def _build_limit_box(self):
        self.__entry_box = urwid.IntEdit(default=self.__entry)
        yield self._create_field_label('entry')
        yield self._decorate_textbase_box(self.__entry_box)

    def _decorate_textbase_box(self,widget):
        return urwid.AttrWrap(widget,
                              self._palette.TEXTBOX)

    def _toggle_state(self,group,value):
        for radio in group:
            if radio.user_data == value:
                radio.set_state(True,False)
            else:
                radio.set_state(False,False)

    def _build_bitrate_group(self):
        bitrates = config.request_bitrate_range
        group = []
        event = self._on_bitrate_toggled
        yield self._create_radio(group, 'All', 'All',event)
        for rate in bitrates:
            yield self._create_radio(group, rate+'kb', rate,event)
        self._toggle_state(group,self.__bitrate)

    def _build_bitrate_box(self):
        yield self._create_field_label('bitrate')
        yield urwid.GridFlow(
            [x for x in self._build_bitrate_group()]
            ,9,1,1,'left')

    def _translate_codec(self,codec):
        return codec.split('/')[1]

    def _build_codec_group(self):
        group = []
        event = self._on_codec_toggled
        for codec,label in shoutbrowser.mime_type.iteritems():
            codec = self._translate_codec(codec)
            yield self._create_radio(group,label,codec,event)
        self._toggle_state(group,self.__codec)

    def _build_codec_box(self):
        yield self._create_field_label('format')
        yield urwid.GridFlow(
            [x for x in self._build_codec_group()]
            ,9,1,1,'left')

    def _build_search_button(self):
        yield urwid.Padding(
            urwid.Button('search',self._on_search),
            'center',10)

    def _build_search_form(self):
        body = urwid.ListBox([
                widget for widget in itertools.chain(
                    self._build_keyword_box(),
                    self._build_bitrate_box(),
                    self._build_codec_box(),
                    self._build_limit_box(),
                    self._build_search_button()
                    )])

        return urwid.Frame(urwid.LineBox(body),
                           header=urwid.Text('Search Form'))

    def get_content(self,*args):
        stations = SearchedStationListBox(self.ui,
                                          self._notify,
                                          self._palette,
                                          self._dialog)
        screen_height = self.ui.get_cols_rows()[1]
        if args:
            keyword,bitrate,codec,limit = args
            shoutbrowser.search_stations(stations.add_station,
                                         keyword,bitrate,codec,limit)
        self._notify.clean()
        return urwid.Columns([
                ('fixed',25,self._build_search_form())
                ,stations.build()
                ])

class SimpleThread:
    __worker_thread = None
    def start(self,target_func,*func_args):
        self.terminate()
        self.__worker_thread = threading.Thread(target=target_func,
                                                args=func_args)
        self.__worker_thread.start()

    def terminate(self):
        worker = self.__worker_thread
        if worker and worker.isAlive():
            self.__worker_thread.join()

class DialogBroker(object):
    def __init__(self,bottom_widget,palette):
        self.__bottom = bottom_widget
        self.__palette = palette
        self.__dialog = None
        self.__prev_size = None
        self.__top = None

    def connect(self,widget,size):
        if self.__top:
            raise Exception
        self.__top = widget,size

    def disconnect(self):
        del self.__top ; del self.__dialog
        self.__top = None ; self.__dialog = None

    def propagate_key(self,size,key):
        if self.__dialog:
            self.__dialog.keypress(size,key)

    @property
    def has_connection(self):
        return None != self.__dialog

    def __create_dialog(self,size):
        widget,widget_size = self.__top

        #calcurate position to locate top-widget as center
        left,top = [ (size[i] - widget_size[i])/2 for i in range(2)]
        wrapped_widget = urwid.Filler(
            urwid.AttrWrap(widget,
                           self.__palette.DIALOG)
            )

        self.__dialog = urwid.Overlay(wrapped_widget,
                                      self.__bottom,
                                      ('fixed left',left),
                                      widget_size[0],
                                      ('fixed top',top),
                                      widget_size[1])
    def get_canvas(self,size):
        canvas = None
        if self.__top:
            if self.__prev_size != size:
                self.__create_dialog(size)
            canvas = self.__dialog.render(size,True)
        return canvas

class AbstractDialog(object):
    def __init__(self,broker):
        self.__broker = broker

    def build(self,content,size):
        self.__broker.connect(content,size)

    def close(self):
        self.__broker.disconnect()

class PlayerDialog(AbstractDialog):
    def __init__(self,broker,notify):
        self.__super = super(self.__class__,self)
        self.__super.__init__( broker )
        self.notify = notify
        self.__thread = None
        self.__proc = None
        self.__temp = None

    def _get_playlist(self,id):
        playlist = shoutbrowser.get_playlist(id)
        if not config.is_player_support_playlist:
            parser = shoutbrowser.PlayListParser(playlist)
            if not config.is_player_support_activity_check:
                self.notify.print_warn("Station's Activity Asking...")
                return parser.get_active_item()
            else:
                return parser.iteritems().next()
        else:
            self._write_playlist_to_tempfile(playlist)
            return self.__temp.name

    def _write_playlist_to_tempfile(self,playlist):
        self.__temp = open(
            os.path.join( tempfile.gettempdir(),
                          '%s.%s' % (tempfile.gettempprefix(),id)),
            'w')
        self.__temp.write(playlist.read())
        if not self.__temp.closed:
            self.__temp.close()
            
    def _run_player(self,id):
        location = self._get_playlist(id)
        command = [param if param !='%s' else location 
                   for param in config.player_command]
        self.notify.print_warn("Execute Player now!")
        self.__proc = subprocess.Popen(command,
                                       stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE)
        self.__proc.wait()
        err = self.__proc.stderr.read()
        if err:
            self.notify.print_error(err)
    def _run_worker(self,id):
        self.notify.print_warn('Playlist Downloading...')
        if not self.__thread:
            self.__thread = SimpleThread()
        self.__thread.start(self._run_player,id)

    def build(self,name,id):
        terminate_button = urwid.Padding(
            urwid.Button('Terminate',self._close),
            'center',('relative',22)
            )
        content = urwid.Pile([urwid.Text('Player calling...'),
                              urwid.Divider('-'),
                              urwid.Text(name),
                              terminate_button
                              ])
        self.__super.build(content,(60,6))
        self._run_worker(id)

    def _cleanup_worker(self):
        if self.__thread:
            if self.__proc.returncode == None:
                os.kill(self.__proc.pid,signal.SIGTERM)
            self.__thread.terminate()

    def _cleanup_tempfile(self):
        if self.__temp:
            if not self.__temp.closed:
                self.__temp.close()
            os.remove(self.__temp.name)

    def _close(self,button):
        self.notify.print_warn('Player Terminating...')
        self._cleanup_worker()
        self._cleanup_tempfile()
        self.notify.print_warn('Player Terminated!')
        self.__super.close()

class HelpDialog(AbstractDialog):
    __about_message =\
'''tuxshout version %s
(c)2009 s.yamada
Release under the LGPL
tuxshout is SHOUTcast Radio tuner''' % version
    __help_message =\
'''To run below method, type shortcut-key (i.e. B-key for Browser)
Help     : display this messages
Browser  : supplys genre-based searching
Favorite : supplys listing of registered stations
Search   : supplys searching with detail infomations
Exit     : exit program'''
    __method_message =\
'''if station selected,below shortcut-key available
some shortcut are available at below mode
Browser=B Search=S Favorite=F
p [B|S|F]: play the station
enter    : same as a-key
a [B|S]  : add the station to favorite list
r [F]    : remove the station from favorite list
delete   : same as r-key'''

    __config_message =\
'''configurable parameter is store in the config.py
please edit it to adapt to you environment'''
    def __init__(self,broker):
        self.__super = super(self.__class__,self)
        self.__super.__init__( broker )

    def build(self):
        content = urwid.Pile([urwid.Text(self.__about_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__help_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__method_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__config_message),
                              urwid.Padding(
                    urwid.Button('Close',self._close),
                    'center',
                    ('relative',15)
                    )
                              ])
        self.__super.build(content,(65,24))

    def _close(self,button):
        self.__super.close()

class TextShout:
    _browse_key ='b'
    _search_key ='s'
    _favorite_key ='f'
    _help_key = '?'
    _exit_key ='x'
    _notify = None
    _palette = None
    __worker_thread = None
    def __init__(self):
        self.ui = Screen()
        self._palette = ColorPalette(self.ui)
        msg = Notification(self._palette)
        view = urwid.Frame( None )
        msg.set_output(view.set_footer)
        self._notify = msg ; self.view = view
        self.__dialog = DialogBroker(self.view,self._palette)
        self._switch_application(GenreBrowser)

    def update_menu(self,active=_browse_key):
        def create_menu_label(label_item):
            key = label_item[0]
            readable_key = key.upper()
            base_text = '%s: %s' % (readable_key,label_item[1])
            menu_label = urwid.Text(base_text)
            attr = self._palette.MENU
            if active == key:
                attr = self._palette.MENU_REVERSE
            return urwid.AttrWrap(menu_label,attr)
        menu_label_list = (
            (self._browse_key,'Browse'),
            (self._favorite_key,'Favorite'),
            (self._search_key,'Search'),
            (self._help_key,'Help'),
            (self._exit_key,'Exit')
            )
        menu = urwid.AttrWrap(
            urwid.GridFlow([
                    create_menu_label(item)
                    for item in sorted(menu_label_list)],
                           12,1,1,'left'),
            self._palette.MENU)
        self.view.set_header(menu)
 
    def set_status(self,text,state):
        info = urwid.AttrWrap(
            urwid.Text(text),state)
        self.view.set_footer(info)

    def main(self):
        space = (65,24)
        if space > self.ui.get_cols_rows():
            print '''
tuxshout can not run at this nallow screen!
please alocate more wideer screen than 
width :%s
height:%s''' % space
            return
        self.ui.run_wrapper( self.run )

    def run(self):
        self.update_menu()
        size = self.ui.get_cols_rows()
        while 1:
            self.draw_screen(size,self.__dialog.get_canvas(size))
            keys = self.ui.get_input()
            if self._exit_key in keys:
                break
            for k in keys:
                if k == "window resize":
                    size = self.ui.get_cols_rows()
                    continue

                #key-action of dialog prioir than other widgets
                if self.__dialog.has_connection:
                    self.__dialog.propagate_key(size,k)
                    continue

                #key-action of any other widget prioir than menu
                if not self.view.keypress( size, k ):
                    continue
                elif k == self._browse_key:
                    self.update_menu(k)
                    self._switch_application(GenreBrowser)
                elif k == self._search_key:
                    self.update_menu(k)
                    self._switch_application(StationSearch)
                elif k == self._favorite_key:
                    self.update_menu(k)
                    self._switch_application(FavoriteStation)
                elif k == self._help_key:
                    HelpDialog(self.__dialog).build()

    def _switch_application(self,application):
        self.view.set_body(
            application(self.ui,
                        self.view,
                        self._notify,
                        self._palette,
                        self.__dialog).get_content()
            )
        self._notify.clean()

    def draw_screen(self, size,canvas=None):
        if not canvas:
            canvas = self.view.render( size, focus=True )
        self.ui.draw_screen( size, canvas )

if __name__ == '__main__':
    TextShout().main()
