import xml.sax
from xml.sax.handler import ContentHandler
from xml.sax.saxutils import XMLFilterBase
import xmlutil,httputil

class StationListHandler(ContentHandler):
    def __init__(self,callback):
        self._callback = callback
        self.__info = {}
        self.__active_node = None
        self.error = None

    def get_log(self):
        return self._log

    def startElement(self,name,attrs):
        if name != 'entry' or name != 'directory':
            self.__active_node = name
        else:
            self.__info.clear()
            self.__active_node = None

    def characters(self,content):
        if self.__active_node:
            text = content.strip()
            if text:
                self.__info[self.__active_node] = text

    def endElement(self, name):
        if name == 'entry':
            info = (self.__info.get('server_type',''),
                    self.__info.get('bitrate',''),
                    self.__info.get('genre',''),
                    self.__info.get('current_song',''))

            self._callback(self.__info['server_name'],
                           self.__info['listen_url'],
                           info=info)

class StationFilter(XMLFilterBase):
    def __init__(self,resource,condition,limit):
        XMLFilterBase.__init__(self,resource)
        self.__filter = condition
        self.__limit = limit
        self.__active_node = None
        self.__info = {}
        self.output = None

    def startElement(self,name,attrs):
        if self.__limit == 0:return
        if name != 'entry' or name != 'directory':
            self.__active_node = name
        else:
            if name == 'directory':
                self.output.startElement(name,attrs)
            self.__info.clear()
            self.__active_node = None

    def characters(self,content):
        if self.__active_node:
            text = content.strip()
            if text:
                self.__info[self.__active_node] = text

    def endElement(self, name):
        if self.__limit == 0:return
        if name == 'entry':
            if not self.__filter.has_match(self.__info):return
            self.output.startElement('entry',{})
            self.__write_info('server_name')
            self.__write_info('listen_url')
            self.__write_info('server_type')
            self.__write_info('bitrate')
            self.__write_info('genre')
            self.__write_info('current_song')
            self.output.endElement('entry')
            self.__limit -= 1
        elif name == 'directory':
            self.output.endElement('directory')

    def __write_info(self,name):
        self.output.startElement(name,{})
        self.output.characters(self.__info.get(name,''))
        self.output.endElement(name)
        
class StationFilterCondition:
    def __init__(self,genre=None,bitrate=None,codec=None,keyword=None):
        self.genre = genre
        self.bitrate = bitrate
        self.codec = codec
        self.keyword = keyword

    def has_match(self,attr):
        matched = False
        if not self.has_match_codec(attr):return matched

        if (self.is_match(attr,'genre',self.genre) or
            self.is_match(attr,'bitrate',self.bitrate)):
            matched = True
        elif self.has_match_keyword(attr):
            matched = True
        return matched

    def has_match_codec(self,attr):
        value = attr.get('server_type','')
        return value in self.codec

    def has_match_keyword(self,attr):
        matched = False
        if self.keyword:
            for value in attr.values():
                if self._find_value(value,self.keyword):
                    matched = True
                    break
        return matched

    def is_match(self,attr,name,value):
        return self._find_value(attr.get(name,''),value)
        
    def _find_value(self,target,condition):
        if target and condition:
            target = target.lower()
            condition = condition.lower()
            return target.find(condition) > -1
        else:
            return False

def download_station_list():
    return httputil.get_opener().open('http://dir.xiph.org/yp.xml')

def parse_station_list(filterobj,callback):
    filter,resource = filterobj
    filter.output = StationListHandler(callback)
    filter.parse(resource)
    resource.close()
    del filterobj
    del resource

def _query_stations(condition,limit):
    resource = download_station_list()
    parser = xml.sax.make_parser()
    filter = StationFilter(parser,condition,limit)
    return filter,resource

def get_stations(genre,limit):
    codec = get_codec_mimes()
    condition = StationFilterCondition(genre=genre,codec=codec)
    return _query_stations(condition,limit)

def search_stations(search_text,bitrate,codec,limit):
    condition = StationFilterCondition(bitrate=bitrate,
                                       codec=codec,
                                       keyword=search_text)
    return _query_stations(condition,limit)

def get_codec_mimes():
    return ['audio/mpeg',
            'audio/aacp',
            'audio/aac',
            'application/ogg',
            'audio/flac']

def _treate_bitrate(bitrate):
    if bitrate.isdigit():
        if len(bitrate) > 3:
            bitrate = bitrate[0:3]
        bitrate +='kb'
    return bitrate

def translate_info(*args):
    args = args[0]
    mime,bitrate,genre = args[0:3]
    format = mime.split('/')[1].upper()
    info = '[%s %s] Genre: %s' % (format,
                                  _treate_bitrate(bitrate),
                                  genre)
    if len(args) > 3:
        info += '\nPlaying: %s' % args[3]
    return info

def is_support_playlist():
    return False

def get_playlist(url):
    #icecast dose not handle station's id,just handle station url
    return url
