/**********************************************************************
 * pk.h                                                     August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef PK_H
#define PK_H

#ifdef __KERNEL__
#include <linux/types.h>
#else
#include "compat.h"
#endif

#define PK_IDLEN\
  64 /* 64 bits */

#define PK_NO_OPTION      0x00000000
#define PK_PKCS1_SIGN     0x00000001
#define PK_PKCS1_ENCRYPT  0x00000002
#define PK_ALGO_RSA       0x00000010
#define PK_ALGO_ELGAMAL   0x00000020

#define PK_OK             0
#define PK_BAD_ALGO       1
#define PK_KEYGEN_FAILED  2
#define PK_INVALID_KEY    3
#define PK_INVALID_INPUT  4
#define PK_UNWRAP_FAILED  5

const char* pk_errstr(int err);
unsigned  pk_option_rev(const char *str);

int pk_init(size_t bitLen, unsigned options);


void pk_deinit(void);


int pk_keygen(u8 **pubKey, size_t *pubKeyLen,
                     u8 **priKey, size_t *priKeyLen,
                     size_t bitLen, unsigned options);

int pk_getKeyID(u8 *id,
                       const u8 *pubKey, size_t pubKeyLen,
                       unsigned options);

int pk_encrypt(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *pubKey, size_t pubKeyLen,
                      unsigned options);

int pk_decrypt(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *priKey, size_t priKeyLen,
                      unsigned options);

int pk_verify(const u8 *payload, size_t payloadLen,
                     const u8 *input, size_t inputLen,
                     const u8 *pubKey, size_t pubKeyLen,
                     unsigned options);

int pk_sign(u8 *output, size_t *outputLen,
                   const u8 *input, size_t inputLen,
                   const u8 *priKey, size_t priKeyLen,
                   unsigned options);

#include "unsx.h"
#include "elgamal.h"
#include "rsa.h"

#define PK_ENC_LENGTH(BITS,OPTIONS)\
 ( (OPTIONS & PK_ALGO_RSA) ? RSA_ENC_LENGTH(BITS,OPTIONS)\
  :(OPTIONS & PK_ALGO_ELGAMAL) ? ELGAMAL_ENC_LENGTH(BITS,OPTIONS)\
   :0\
 )

#define PK_SIG_LENGTH(BITS,OPTIONS)\
 ( (OPTIONS & PK_ALGO_RSA) ? RSA_SIG_LENGTH(BITS,OPTIONS)\
  :(OPTIONS & PK_ALGO_ELGAMAL) ? ELGAMAL_SIG_LENGTH(BITS,OPTIONS)\
   :0\
 )


static inline void unload_unsx(u8 *dest, unsx *src, size_t len)
{
	unsx t;

	for (; len; len--) {
		t = *src++;
		*dest++ = (t >> 24) & 0xff;
		*dest++ = (t >> 16) & 0xff;
		*dest++ = (t >> 8)  & 0xff;
		*dest++ = t         & 0xff;
	}
}

static inline void load_unsx(unsx *dest, const u8 *src, size_t len)
{
	unsx t;

	for (; len; len--) {
		t  = *src++ << 24;
		t |= *src++ << 16;
		t |= *src++ << 8;
		t |= *src++;
		*dest++ = t;
	}
}


static inline unsx *alloc_load_unsx(const u8 *src, size_t len)
{
	unsx *dest;

	dest = (unsx*) kmalloc(len*sizeof(unsx), GFP_KERNEL);
	if(!dest)
		return(NULL);

	load_unsx(dest, src, len);

	return(dest);
}



/* Load and unload bytes in reverse order. This is needed for
 * PKCS#1 encoding.
 * See RFC3477 Sections 4.1 and 4.2 */

static inline void unload_unsx_rev(u8 *dest, unsx *src, size_t len)
{
	unsx t;

	src += len - 1;
	for (; len; len--) {
		t = *src--;
		*dest++ = (t >> 24) & 0xff;
		*dest++ = (t >> 16) & 0xff;
		*dest++ = (t >> 8)  & 0xff;
		*dest++ = t         & 0xff;
	}
}

static inline void load_unsx_rev(unsx *dest, const u8 *src, size_t len)
{
	unsx t;

	src += (len * sizeof(unsx)) - 1;
	for (; len; len--) {
		t  = *src--;
		t |= *src-- << 8;
		t |= *src-- << 16;
		t |= *src-- << 24;
		*dest++ = t;
	}
}

#endif /* PK_H */
