/**********************************************************************
 * from_kernle.c                                            August 2005
 *
 * KSSLD(key_tool): An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>



#include <openssl/ssl.h>
#include <openssl/evp.h>
#include <vanessa_logger.h>

#include "log.h"
#include "kssl.h"
#include "key_tool.h"
#include "cipher.h"
#include "asym_method.h"

static kssl_key_type_t
parse_key_type(unsigned char *buf, size_t *len);

static char *
key_type_str(kssl_key_type_t key_type);

static EVP_PKEY *
parse_key(unsigned char *buf, size_t *len, kssl_key_type_t type);

#ifdef WITH_DH_DSA_SUPPORT
static DH *
parse_dh_param(unsigned char *buf, size_t *len);
#endif

static int
parse_cert(unsigned char *in_buf, size_t *in_buf_len, 
		unsigned char **out_buf, size_t *out_buf_len);

static int
write_cert_to_file(FILE *fp, unsigned char *buf, size_t len);

static int 
key_tool_from_kernel_key(int cmd, const char *vserver, 
		const char *key_filename, const char *cert_filename);

static int 
key_tool_from_kernel_real(const char *vserver);

static int 
key_tool_from_kernel_ciphers(const char *vserver);

int
key_tool_from_kernel_daemons(void);

static int 
key_tool_from_kernel_mode(const char *vserver);

static int 
key_tool_from_kernel_asym_methods(const char *vserver);


int 
key_tool_from_kernel(int cmd, const char *arg_1, const char *arg_2,
		const char *arg_3)
{
	switch(cmd) {
		case CMD_RSA:
		case CMD_DSA:
		case CMD_DH:
			return key_tool_from_kernel_key(cmd, arg_1, arg_2,
					arg_3);
		case CMD_REAL:
			return key_tool_from_kernel_real(arg_1);
		case CMD_CIPHERS:
			return key_tool_from_kernel_ciphers(arg_1);
		case CMD_AVAILABLE_CIPHERS:
			return cipher_available_printf();
		case CMD_MODE:
			return key_tool_from_kernel_mode(arg_1);
		case CMD_ASYM_METHODS:
			return key_tool_from_kernel_asym_methods(arg_1);
		case CMD_AVAILABLE_ASYM_METHODS:
			return asym_method_available_printf();
		case CMD_DAEMONS:
			return key_tool_from_kernel_daemons();
	}
	
	VANESSA_LOGGER_DEBUG("Unknown command\n");
	return -1;
}


int 
key_tool_from_kernel_key(int cmd, const char *vserver, 
		const char *key_filename, const char *cert_filename)
{
	FILE *cert_fp = NULL;
	FILE *key_fp = NULL;
	unsigned char *cert = NULL;
	size_t cert_len = 0;
	EVP_PKEY *key = NULL;
	DH *dh_param = NULL;
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint32_t in_len_cpy;
	uint32_t in_offset = 0;
	unsigned char *in_buf = NULL;
	kssl_key_type_t key_type;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_key: "
				"key_tool_write_head");
		return -1;
	}

	/* Read Key and Certificate from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CERT_KEY_LEN, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_CERT_KEY_LEN");
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	memcpy(&in_len, &ctl[1], sizeof(uint32_t));
	in_len_cpy = in_len = ntohl(in_len);

	if (in_len == (sizeof(kssl_ctl_t) + sizeof(uint32_t))) {
		VANESSA_LOGGER_ERR("No key in kernel");
		goto leave;
	}

	in_buf = (char *)calloc(1, in_len);
	if (!in_buf) {
		VANESSA_LOGGER_DEBUG_ERRNO("calloc");
		goto leave;
	}

	/* Write header */
	memset(ctl, 0, sizeof(kssl_ctl_t));
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_key: "
				"key_tool_write_head");
		return -1;
	}
	memcpy(in_buf, &ctl[0], sizeof(kssl_ctl_t));

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CERT_KEY, in_buf, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_CERT_KEY");
		VANESSA_LOGGER_ERR("Error getting data from kernel");
		goto leave;
	}

	if (in_len != in_len_cpy) {
		VANESSA_LOGGER_DEBUG_UNSAFE("wrong length %u != %u",
				in_len, in_len_cpy);
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	in_offset = sizeof(kssl_ctl_t);

	/* Parse Key Type */
	in_len = in_len_cpy - in_offset;
	key_type = parse_key_type(in_buf+in_offset, &in_len);
	in_offset += in_len;

	switch (cmd) {
		case CMD_RSA:
			if (key_type != kssl_key_type_rsa) {
				VANESSA_LOGGER_ERR_UNSAFE("RSA key requested "
						"but %s present in the kernel",
						key_type_str(key_type));
				goto leave;
			}
			break;
#ifdef WITH_DH_DSA_SUPPORT
		case CMD_DSA:
			if (key_type != kssl_key_type_dsa) {
				VANESSA_LOGGER_ERR_UNSAFE("DSA key requested "
						"but %s present in the kernel",
						key_type_str(key_type));
				goto leave;
			}
			break;
		case CMD_DH:
			if (key_type != kssl_key_type_dsa) {
				VANESSA_LOGGER_ERR_UNSAFE("DH parameters "
						"requested but %s present in "
						"the kernel",
						key_type_str(key_type));
				goto leave;
			}
			break;
#endif /* WITH_DH_DSA_SUPPORT */
	}

	/* Parse Key */
	in_len = in_len_cpy - in_offset;
	switch (key_type) {
		case kssl_key_type_rsa:
#ifdef WITH_DH_DSA_SUPPORT
		case kssl_key_type_dsa:
#endif /* WITH_DH_DSA_SUPPORT */
			if (!cert_filename) {
				VANESSA_LOGGER_ERR("DSA or RSA key but no "
						"certificate file specified");
				goto leave;
			}
			key = parse_key(in_buf+in_offset, &in_len, key_type);
			if (!key) {
				VANESSA_LOGGER_DEBUG("parse_key");
				VANESSA_LOGGER_ERR("Error parsing key");
				goto leave;
			}
			break;
#ifdef WITH_DH_DSA_SUPPORT
		case kssl_key_type_dh:
			if (cert_filename) {
				VANESSA_LOGGER_ERR("Certificate file for "
						"DH parameter ignored");
				cert_filename = NULL;
				goto leave;
			}
			dh_param = parse_dh_param(in_buf+in_offset, &in_len);
			if(!dh_param) {
				VANESSA_LOGGER_DEBUG("parse_dh_param");
				VANESSA_LOGGER_ERR("Error parsing key");
				goto leave;
			}
			break;
#else /* WITH_DH_DSA_SUPPORT */
		case kssl_key_type_dh:
#endif /* WITH_DH_DSA_SUPPORT */
		case kssl_key_type_none:
		case kssl_key_type_unknown:
		default:
			VANESSA_LOGGER_DEBUG("parse_key_type");
			VANESSA_LOGGER_ERR("unknown/unsuported key type");
			goto leave;
	}
	in_offset += in_len;

	/* Parse Certificate */
	if (cert_filename) {
		in_len = in_len_cpy - in_offset;
		if (parse_cert(in_buf+in_offset, &in_len, &cert, 
					&cert_len) < 0) {
			VANESSA_LOGGER_DEBUG("parse_cert");
			VANESSA_LOGGER_ERR("Error parsing certificate");
			goto leave;
		}
		in_offset += in_len;
	}

	/* Write Key */
	key_fp = fopen(key_filename, "w");
	if (!key_fp) {
		VANESSA_LOGGER_DEBUG_ERRNO("fopen");
		VANESSA_LOGGER_ERR_UNSAFE("Error opening key file: "
				"\"%s\"", key_filename);
		goto leave;
	}
 
	if (cert_filename) {
		if(!PEM_write_PrivateKey(key_fp, key, NULL, NULL, 0, 0, 
					NULL)) {
			KSSLD_KEY_TOOL_DEBUG_SSL_ERR("PEM_write_PrivateKey");
			VANESSA_LOGGER_ERR("Error writing key file");
			goto leave;
		}
	}
	else {
		if(!PEM_write_DHparams(key_fp, dh_param)) {
			KSSLD_KEY_TOOL_DEBUG_SSL_ERR("PEM_write_DHparams");
			VANESSA_LOGGER_ERR("Error loading parameters file");
			goto leave;
		}
	}

	fclose(key_fp);
	key_fp = NULL;

	/* Write Certificate */

	if (cert_filename) {
		/* Open, read and verify certificate syntax */
		cert_fp = fopen(cert_filename, "w");
		if (!cert_fp) {
			VANESSA_LOGGER_DEBUG_ERRNO("fopen");
			VANESSA_LOGGER_ERR_UNSAFE("Error opening cert file: "
					"\"%s\"", cert_filename);
			goto leave;
		}

		if (write_cert_to_file(cert_fp, cert, cert_len) < 0) {
			VANESSA_LOGGER_DEBUG("write_cert_to_file");
			VANESSA_LOGGER_ERR("Error loading cert file");
			goto leave;
		}
	}

	status = 0;
leave:
	if (in_buf)
		free(in_buf);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	if (key_fp)
		tmp_status = fclose(key_fp);
	if (cert_fp)
		tmp_status = fclose(cert_fp);
	if (key)
		EVP_PKEY_free(key);
	if (dh_param)
		DH_free(dh_param);
	
	return status;
}


int 
key_tool_from_kernel_real(const char *vserver)
{
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint8_t *buf;

	struct in_addr addr;
	uint16_t port;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_real: "
				"key_tool_write_head");
		return -1;
	}

	/* Real IP Address and Port from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_R_IP_PORT, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_R_IP_PORT");
		VANESSA_LOGGER_ERR("Error getting real ip and port "
				"from kernel");
		return -1;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t) + sizeof(uint16_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t)
				+ sizeof(uint16_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		return -1;
	}

	buf = (uint8_t *)&ctl[1];
	memcpy(&(addr.s_addr), buf, sizeof(uint32_t));
	memcpy(&port, buf + sizeof(uint32_t), sizeof(uint16_t));

        printf("%s %u\n", inet_ntoa(addr), ntohs(port));

	close(sock_fd);
	return 0;
}


int 
key_tool_from_kernel_ciphers(const char *vserver)
{
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint32_t in_len_cpy;
	unsigned char *in_buf = NULL;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_ciphers: "
				"key_tool_write_head");
		return -1;
	}

	/* Ciphers from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CIPHERS_LEN, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_CIPHERS_LEN");
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	memcpy(&in_len, &ctl[1], sizeof(uint32_t));
	in_len_cpy = in_len = ntohl(in_len);

	if (in_len == sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_ERR("No ciphers in kernel");
		goto leave;
	}

	in_buf = (char *)calloc(1, in_len);
	if (!in_buf) {
		VANESSA_LOGGER_DEBUG_ERRNO("calloc");
		goto leave;
	}

	/* Write header */
	memset(ctl, 0, sizeof(kssl_ctl_t));
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_ciphers: "
				"key_tool_write_head");
		return -1;
	}
	memcpy(in_buf, &ctl[0], sizeof(kssl_ctl_t));

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CIPHERS, in_buf, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_CIPHERS");
		VANESSA_LOGGER_ERR("Error getting data from kernel");
		goto leave;
	}

	if (in_len != in_len_cpy) {
		VANESSA_LOGGER_DEBUG_UNSAFE("wrong length %u != %u",
				in_len, in_len_cpy);
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	tmp_status = cipher_list_printf(
			(cipher_suite_t *)(in_buf+sizeof(kssl_ctl_t)), 
			(in_len - sizeof(kssl_ctl_t)) / sizeof(cipher_suite_t));


	status = 0;
leave:
	if (in_buf)
		free(in_buf);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	
	return status;
}


int 
key_tool_from_kernel_daemons(void)
{
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint32_t in_len_cpy;
	unsigned char *in_buf = NULL;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, NULL) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_virtuals: "
				"key_tool_write_head");
		return -1;
	}

	/* Virtuals from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_DAEMONS_LEN, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_DAEMONS_LEN");
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	memcpy(&in_len, &ctl[1], sizeof(uint32_t));
	in_len_cpy = in_len = ntohl(in_len);

	if (in_len == sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_INFO("No virtuals in the kernel");
		goto ok;
	}

	in_buf = (char *)calloc(1, in_len);
	if (!in_buf) {
		VANESSA_LOGGER_DEBUG_ERRNO("calloc");
		goto leave;
	}

	/* Write header */
	memset(ctl, 0, sizeof(kssl_ctl_t));
	if (key_tool_write_head(ctl, NULL) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_virtuals: "
				"key_tool_write_head");
		return -1;
	}
	memcpy(in_buf, &ctl[0], sizeof(kssl_ctl_t));

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_DAEMONS, in_buf, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_DAEMONS");
		VANESSA_LOGGER_ERR("Error getting data from kernel");
		goto leave;
	}

	if (in_len != in_len_cpy) {
		VANESSA_LOGGER_DEBUG_UNSAFE("wrong length %u != %u",
				in_len, in_len_cpy);
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}


	{
		u8 *p;

		p = in_buf + sizeof(kssl_ctl_t);
		in_len -= sizeof(kssl_ctl_t);

		while (in_len >= sizeof(u32) + sizeof(u16)) {
			struct in_addr addr;
			addr.s_addr = *(u32 *)p;
			printf("%s:%d\n", inet_ntoa(addr),
					ntohs(*(u16 *)(p+sizeof(u32))));
			p += sizeof(u32) + sizeof(u16);
			in_len -= sizeof(u32) + sizeof(u16);
		}
	}

ok:
	status = 0;
leave:
	if (in_buf)
		free(in_buf);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	
	return status;
}


int 
key_tool_from_kernel_mode(const char *vserver)
{
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint8_t *buf;

	kssl_daemon_mode_t mode;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_mode: "
				"key_tool_write_head");
		return -1;
	}

	/* Mode from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_DAEMON_MODE, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: KSSL_CTL_DAEMON_MODE");
		VANESSA_LOGGER_ERR("Error getting mode from kernel");
		return -1;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		return -1;
	}

	buf = (uint8_t *)&ctl[1];
	memcpy(&mode, buf, sizeof(uint32_t));
	mode = ntohl(mode);

	switch (mode) {
		case kssl_daemon_mode_running:
			printf("%s\n", KSSL_DAEMON_MODE_RUNNING);
			break;
		case kssl_daemon_mode_stopped:
			printf("%s\n", KSSL_DAEMON_MODE_STOPPED);
			break;
		case kssl_daemon_mode_quiescent:
			printf("%s\n", KSSL_DAEMON_MODE_QUIESCENT);
			break;
		case kssl_daemon_mode_none:
		case kssl_daemon_mode_unknown:
			printf("unknown\n");
			break;
	}

	close(sock_fd);
	return 0;
}


int 
key_tool_from_kernel_asym_methods(const char *vserver)
{
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl[2];
	int sock_fd = -1;
	uint32_t in_len;
	uint32_t in_len_cpy;
	unsigned char *in_buf = NULL;

	/* Write header */
	in_len = sizeof(kssl_ctl_t) * 2;
	memset(ctl, 0, in_len);
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_asym_methods: "
				"key_tool_write_head");
		return -1;
	}

	/* Ciphers from Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_ASYM_METHODS_LEN, &ctl, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: "
				"KSSL_CTL_ASYM_METHODS_LEN");
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	if (in_len < sizeof(kssl_ctl_t) + sizeof(uint32_t)) {
		VANESSA_LOGGER_DEBUG_UNSAFE("length too short %u < %u",
				in_len, sizeof(kssl_ctl_t) + sizeof(uint32_t));
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	memcpy(&in_len, &ctl[1], sizeof(uint32_t));
	in_len_cpy = in_len = ntohl(in_len);

	if (in_len == sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_ERR("No asym methods in kernel");
		goto leave;
	}

	in_buf = (char *)calloc(1, in_len);
	if (!in_buf) {
		VANESSA_LOGGER_DEBUG_ERRNO("calloc");
		goto leave;
	}

	/* Write header */
	memset(ctl, 0, sizeof(kssl_ctl_t));
	if (key_tool_write_head(ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_from_kernel_asym_methods: "
				"key_tool_write_head");
		return -1;
	}
	memcpy(in_buf, &ctl[0], sizeof(kssl_ctl_t));

	if (getsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_ASYM_METHODS, in_buf, 
				&in_len) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("getsockopt: "
				"KSSL_CTL_ASYM_METHODS");
		VANESSA_LOGGER_ERR("Error getting data from kernel");
		goto leave;
	}

	if (in_len != in_len_cpy) {
		VANESSA_LOGGER_DEBUG_UNSAFE("wrong length %u != %u",
				in_len, in_len_cpy);
		VANESSA_LOGGER_ERR("Error getting length from kernel");
		goto leave;
	}

	tmp_status = asym_method_list_printf(
			(kssl_asym_method_t *)(in_buf+sizeof(kssl_ctl_t)), 
			(in_len - sizeof(kssl_ctl_t)) / 
			sizeof(kssl_asym_method_t));

	status = 0;
leave:
	if (in_buf)
		free(in_buf);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	
	return status;
}


static int
get_buf_from_lbuf(unsigned char *lbuf, size_t *lbuf_len, 
		unsigned char **buf, size_t *buf_len)
{
        int bytes = 0;
        uint32_t tmp_len;

        if (*lbuf_len < sizeof(uint32_t))
                return -1;

        memcpy(&tmp_len, lbuf, sizeof(uint32_t));
        bytes += sizeof(uint32_t);
        tmp_len = ntohl(tmp_len);

        if (tmp_len > (size_t)(~0U>>1) || tmp_len > INT_MAX ||
                        tmp_len + sizeof(uint32_t) > *lbuf_len)  {
                VANESSA_LOGGER_DEBUG_UNSAFE("size it too large: %u\n", 
				tmp_len);
                return -1;
        }

	*lbuf_len = tmp_len + sizeof(uint32_t);
	*buf_len = tmp_len;
	
        if (!tmp_len)
		*buf = NULL;
	else
		*buf = lbuf + bytes;

        return 0;
}


static BIGNUM *
cpy_alloc_bn_from_lbuf(unsigned char *lbuf, size_t *len, BIGNUM *bn)
{
	unsigned char *tmp_buf;
	size_t tmp_len;

	if (get_buf_from_lbuf(lbuf, len, &tmp_buf, &tmp_len) < 0) {
		VANESSA_LOGGER_DEBUG("get_buf_from_lbuf");
		return NULL;
	}
	
	bn = BN_bin2bn(tmp_buf, tmp_len, bn);
	if (!bn) {
		VANESSA_LOGGER_DEBUG("BN_bin2bn");
		return NULL;
	}

	return bn;
}


#define PARSE_BN(bn)                                                          \
do {                                                                          \
	size_t tmp_len;                                                       \
	tmp_len = *len - bytes;                                               \
	bn = cpy_alloc_bn_from_lbuf(buf + bytes, &tmp_len, bn);               \
	if (!bn) {                                                            \
		VANESSA_LOGGER_DEBUG("cpy_alloc_bn_from_lbuf");               \
		goto error;                                                   \
	}                                                                     \
	bytes += tmp_len;                                                     \
} while(0)


static int
parse_rsa_key(unsigned char *buf, size_t *len, EVP_PKEY *key)
{
	RSA *rsa_key = NULL;
	size_t bytes = 0;

	rsa_key = RSA_new();
	if (!rsa_key) {
		KSSLD_KEY_TOOL_DEBUG_SSL_ERR("RSA_new");
		goto error;
	}

	PARSE_BN(rsa_key->n);
	PARSE_BN(rsa_key->e);
	PARSE_BN(rsa_key->d);
	PARSE_BN(rsa_key->p);
	PARSE_BN(rsa_key->q);
	PARSE_BN(rsa_key->dmp1);
	PARSE_BN(rsa_key->dmq1);
	PARSE_BN(rsa_key->iqmp);

	if (!EVP_PKEY_set1_RSA(key, rsa_key)) {
		VANESSA_LOGGER_DEBUG("EVP_PKEY_set1_RSA");
		goto error;
	}
	*len = bytes;
	return 0;

error:
	if (rsa_key)
		RSA_free(rsa_key);
	return -1;
}

#ifdef WITH_DH_DSA_SUPPORT
static int 
parse_dsa_key(unsigned char *buf, size_t *len, EVP_PKEY *key)
{
	DSA *dsa_key = NULL;
	size_t bytes = 0;

	dsa_key = DSA_new();
	if (!dsa_key) {
		VANESSA_LOGGER_DEBUG("DSA_new");
		return -1;
	}

	PARSE_BN(dsa_key->p);
	PARSE_BN(dsa_key->q);
	PARSE_BN(dsa_key->g);
	PARSE_BN(dsa_key->priv_key);
	PARSE_BN(dsa_key->pub_key);

	if (!EVP_PKEY_set1_DSA(key, dsa_key)) {
		VANESSA_LOGGER_DEBUG("EVP_PKEY_set1_DSA");
		goto error;
	}
	*len = bytes;
	return 0;

error:
	if (dsa_key)
		DSA_free(dsa_key);
	return -1;
}


static DH *
parse_dh_param(unsigned char *buf, size_t *len)
{
	DH *dh_param = NULL;
	size_t bytes = 0;

	dh_param = DH_new();
	if (!dh_param) {
		VANESSA_LOGGER_DEBUG("DH_new");
		goto error;
	}

	PARSE_BN(dh_param->p);
	PARSE_BN(dh_param->g);

	*len = bytes;
	return dh_param;

error:
	if (dh_param)
		DH_free(dh_param);
	return NULL;
}
#endif /* WITH_DH_DSA_SUPPORT */


static EVP_PKEY *
parse_key(unsigned char *buf, size_t *len, kssl_key_type_t type)
{
	EVP_PKEY *key = NULL;

	key = EVP_PKEY_new();
	if (!key) {
		VANESSA_LOGGER_DEBUG("parse_key");
		return NULL;
	}

	switch (type) {
		case kssl_key_type_rsa:
			if (parse_rsa_key(buf, len, key) < 0) {
				VANESSA_LOGGER_DEBUG("parse_rsa_key");
				goto error;
			}
			break;
#ifdef WITH_DH_DSA_SUPPORT
		case kssl_key_type_dsa:
			if (parse_dsa_key(buf, len, key) < 0) {
				VANESSA_LOGGER_DEBUG("parse_rsa_key");
				goto error;
			}
			break;
#else /* WITH_DH_DSA_SUPPORT */
		case kssl_key_type_dsa:
#endif /* WITH_DH_DSA_SUPPORT */
		case kssl_key_type_dh:
		case kssl_key_type_unknown:
		case kssl_key_type_none:
			VANESSA_LOGGER_DEBUG("Unsuported key type\n");
			goto error;
	}

	return key;

error:
	if (key)
		EVP_PKEY_free(key);
	return NULL;
}


static int
parse_cert(unsigned char *in_buf, size_t *in_buf_len, 
		unsigned char **out_buf, size_t *out_buf_len)
{
	return get_buf_from_lbuf(in_buf, in_buf_len, out_buf, out_buf_len);

}


static int
write_cert_to_file(FILE *fp, unsigned char *buf, size_t len)
{
	int status = -1;
	BIO *out = NULL;
	BIO *b64 = NULL;

	fprintf(fp, "-----BEGIN CERTIFICATE-----\n");
	fflush(fp);

	b64 = BIO_new(BIO_f_base64());
	if (!b64) {
		KSSLD_KEY_TOOL_DEBUG_SSL_ERR("BIO_f_base64");
		goto leave;
	}

	out = BIO_new_fp(fp, BIO_NOCLOSE);
	if (!out) {
		VANESSA_LOGGER_DEBUG("BIO_new_fp");
		goto leave;
	}

	out = BIO_push(b64, out);

	if (BIO_write(out, buf, len) != len) {
		KSSLD_KEY_TOOL_DEBUG_SSL_ERR("BIO_write");
		goto leave;
	}
	BIO_flush(out);

	fprintf(fp, "-----END CERTIFICATE-----\n");
	fflush(fp);

	status = 0;
leave:
	if (out)
		if (!BIO_free(out))
			VANESSA_LOGGER_DEBUG("BIO_free: out");

	return status;
}


static kssl_key_type_t
parse_key_type(unsigned char *buf, size_t *len)
{
	unsigned char *label_buf = NULL;
	size_t label_buf_len;

	if (get_buf_from_lbuf(buf, len, &label_buf, &label_buf_len) < 0)
		return kssl_key_type_none;

	if (!label_buf_len)
		return kssl_key_type_none;

	if (!strncmp(KSSL_KEY_TYPE_RSA, label_buf, label_buf_len))
		return kssl_key_type_rsa;
#ifdef WITH_DH_DSA_SUPPORT
	else if (!strncmp(KSSL_KEY_TYPE_DSA, label_buf, label_buf_len))
		return kssl_key_type_dsa;
	else if (!strncmp(KSSL_KEY_TYPE_DH, label_buf, label_buf_len))
		return kssl_key_type_dh;
#endif /* WITH_DH_DSA_SUPPORT */

	return kssl_key_type_unknown;
}



static char *
key_type_str(kssl_key_type_t key_type)
{
	switch(key_type) {
		case kssl_key_type_none:
			return "none";
		case kssl_key_type_rsa:
			return KSSL_KEY_TYPE_RSA;
#ifdef WITH_DH_DSA_SUPPORT
		case kssl_key_type_dsa:
			return KSSL_KEY_TYPE_DSA;
		case kssl_key_type_dh:
			return KSSL_KEY_TYPE_DH;
#endif /* WITH_DH_DSA_SUPPORT */
		case kssl_key_type_unknown:
		default:
			return "unknown";
	}

	return "unknown";
}

