/**********************************************************************
 * key_tool.c                                               August 2005
 *
 * KSSLD(key_tool): An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <vanessa_logger.h>

#include "key_tool.h"
#include "log.h"

vanessa_logger_t *vl = NULL;

static void usage(int exit_status);

int 
main(int argc, char **argv) 
{
	int cmd = 0;
	int action = 0;
	char *arg_3 = NULL;
	char *arg_4 = NULL;
	char *arg_5 = NULL;

	if (argc < 3)
		usage(1);

	if (!strcasecmp(ACTION_SET_STR, argv[1]))
		action = ACTION_SET;
	else if (!strcasecmp(ACTION_GET_STR, argv[1]))
		action = ACTION_GET;
	else
		usage(1);


	if (!strcasecmp(CMD_RSA_STR, argv[2])) {
		if(argc != 6)
			usage(1);
		cmd = CMD_RSA;
	}
#ifdef WITH_DH_DSA_SUPPORT
	else if (!strcasecmp(CMD_DSA_STR, argv[2])) {
		if(argc != 6)
			usage(1);
		cmd = CMD_DSA;
	}
	else if (!strcasecmp(CMD_DH_STR, argv[2])) {
		if(argc != 6)
			usage(1);
		cmd = CMD_DH;
	}
#endif
	else if (!strcasecmp(CMD_REAL_STR, argv[2])) {
		if(argc != ((action == ACTION_SET) ? 5 : 4))
			usage(1);
		cmd = CMD_REAL;
	}
	else if (!strcasecmp(CMD_CIPHERS_STR, argv[2])) {
		if(argc != ((action == ACTION_SET) ? 5 : 4))
			usage(1);
		cmd = CMD_CIPHERS;
	}
	else if (!strcasecmp(CMD_AVAILABLE_CIPHERS_STR, argv[2])) {
		if (action != ACTION_GET || argc != 3)
			usage(1);
		cmd = CMD_AVAILABLE_CIPHERS;
	}
	else if (!strcasecmp(CMD_MODE_STR, argv[2])) {
		if(argc != ((action == ACTION_SET) ? 5 : 4))
			usage(1);
		cmd = CMD_MODE;
	}
	else if (!strcasecmp(CMD_ASYM_METHODS_STR, argv[2])) {
		if(argc != ((action == ACTION_SET) ? 5 : 4))
			usage(1);
		cmd = CMD_ASYM_METHODS;
	}
	else if (!strcasecmp(CMD_AVAILABLE_ASYM_METHODS_STR, argv[2])) {
		if (action != ACTION_GET || argc != 3)
			usage(1);
		cmd = CMD_AVAILABLE_ASYM_METHODS;
	}
	else if (!strcasecmp(CMD_DEL_DAEMON_STR, argv[2])) {
		if (action != ACTION_SET || argc != 4)
			usage(1);
		cmd = CMD_DEL_DAEMON;
	}
	else if (!strcasecmp(CMD_ADD_DAEMON_STR, argv[2])) {
		if (action != ACTION_SET || argc != 4)
			usage(1);
		cmd = CMD_ADD_DAEMON;
	}
	else if (!strcasecmp(CMD_DAEMONS_STR, argv[2])) {
		if (action != ACTION_GET || argc != 3)
			usage(1);
		cmd = CMD_DAEMONS;
	}
	else if (!strcasecmp(CMD_FLUSH_STR, argv[2])) {
		if (action != ACTION_SET || argc != 3)
			usage(1);
		cmd = CMD_FLUSH;
	}
	else {
		usage(1);
	}

	if(argc > 3)
		arg_3 = argv[3];
	if(argc > 4)
		arg_4 = argv[4];
	if(argc > 5)
		arg_5 = argv[5];

	/* Initialise logger */
	vl = vanessa_logger_openlog_filehandle(stderr, LOG_IDENT, 
			LOG_DEBUG, VANESSA_LOGGER_F_NO_IDENT_PID);
	if (!vl) {
		fprintf(stderr, "Could not open logger\n");
		return 1;
	}
	vanessa_logger_set(vl);

	if (action == ACTION_SET) {
		if (key_tool_to_kernel(cmd, arg_3, arg_4, arg_5) < 0) {
			VANESSA_LOGGER_DEBUG("to_kernel");
			return 1;
		}
	}
	else {
		if (key_tool_from_kernel(cmd, arg_3, arg_4, arg_5) < 0) {
			VANESSA_LOGGER_DEBUG("from_kernel");
			return 1;
		}
	}

	return 0;
}


static void
usage(int exit_status)
{
	fprintf(exit_status ? stderr : stdout,
		"kssl_key_tool v%s Copyright (C) 2005 NTT COMWARE Corporation\n"
		"\n"
		"kssl_key_tool is a tool for configuring certificates and "
		"keys for kssld\n"
		"\n"
		"Usage:\n"
		"  kssl_key_tool get daemons\n"
		"  kssl_key_tool set flush\n"
		"  kssl_key_tool set add_daemon VIRTUAL_SERVER\n"
		"  kssl_key_tool set del_daemon VIRTUAL_SERVER\n"
		"  kssl_key_tool get|set rsa VIRTUAL_SERVER "
			"RSA_KEY_FILE RSA_CERTIFICATE_FILE\n"
#ifdef WITH_DH_DSA_SUPPORT
		"  kssl_key_tool get|set dsa VIRTUAL_SERVER "
			"DSA_KEY_FILE DSA_CERTIFICATE_FILE\n"
		"  kssl_key_tool get|set dh VIRTUAL_SERVER "
			"DH_PARAMETERS_FILE\n"
#endif /* WITH_DH_DSA_SUPPORT */
		"  kssl_key_tool set real VIRTUAL_SERVER REAL_SERVER\n"
		"  kssl_key_tool get real VIRTUAL_SERVER\n"
		"  kssl_key_tool set ciphers VIRTUAL_SERVER "
			"CIPHER[,CIPHER...]\n"
		"  kssl_key_tool get ciphers VIRTUAL_SERVER\n"
		"  kssl_key_tool get available_ciphers\n"
		"  kssl_key_tool set mode VIRTUAL_SERVER start|stop|quiescent\n"
		"  kssl_key_tool get mode VIRTUAL_SERVER\n"
		"  kssl_key_tool set asym_methods VIRTUAL_SERVER "
			"METHOD[,METHOD...]\n"
		"  kssl_key_tool get asym_methods VIRTUAL_SERVER\n"
		"  kssl_key_tool get available_asym_methods\n"
		"\n"
		" REAL_SERVER and VIRTUAL_SERVER are of the following form:\n"
		"   IP_ADDRESS|HOSTNAME:PORT_NUMBER|PORT_NAME\n"
		"\n"
		"Files must be in PEM format\n"
		"\n",
		VERSION);

	exit(exit_status);
}

int
key_tool_parse_server(const char *str, struct sockaddr_in *addr)
{

	char *ip;
	char *port;

	ip = strdup(str);
	if (!ip) {
		return -1;
	}
	port = strchr(ip, ':');
	if (!port) {
		free(ip);
		return -1;
	}
	*port++ = '\0';

	addr->sin_addr.s_addr = 0;
	addr->sin_port = 0;

	if(vanessa_socket_host_port_sockaddr_in(ip, port, addr, 0) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_parse_server: "
				"vanessa_socket_host_port_sockaddr_in");
		free(ip);
		return -1;
	}

	free(ip);
	return 0;
}


int 
key_tool_write_head(kssl_ctl_t *ctl, const char *vserver)
{
	struct sockaddr_in vaddr;

	ctl->version = htonl(KSSL_CTL_VERSION);

	if (!vserver) {
		ctl->vaddr = 0;
		ctl->vport = 0;
		return 0;
	}

	if (key_tool_parse_server(vserver, &vaddr) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_set_head: "
				"key_tool_parse_server");
		return -1;
	}

     	ctl->vaddr = vaddr.sin_addr.s_addr;
	ctl->vport = vaddr.sin_port;

	return 0;
}

