/**********************************************************************
 * app_data.c                                               August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/kernel.h>
#include <linux/net.h>
#include <linux/version.h>

#include <net/tcp.h>

#include "record.h"
#include "message.h"
#include "conn.h"
#include "kssl_alloc.h"
#include "log.h"


int 
kssl_application_data_send_vec(kssl_record_t *cr, size_t offset, size_t len)
{
	int status = -ENOMEM;
	size_t vecno = 0;
	size_t tail_size;
	size_t new_iov_len;
	struct iovec *new_iov = NULL;
	struct sk_buff **new_skbv = NULL;

	KSSL_DEBUG(12, "kssl_application_data_send_vec: %p "
			"offset=%u len=%u\n", cr, offset, len);

	tail_size = kssl_record_tail_len(cr, len + TLS_HEAD_NLEN);

	new_iov_len = 1 + cr->iov_len + (tail_size ? 1 : 0);
	new_iov = (struct iovec *)kssl_kmalloc(new_iov_len * 
			sizeof(struct iovec), GFP_KERNEL);
	if (!new_iov) {
		KSSL_DEBUG(6, "kssl_application_data_send_vec: kssl_kmalloc 1\n");
		goto mem_error;
	}
	memset(new_iov, 0, new_iov_len * sizeof(struct iovec));

	new_skbv = (struct sk_buff **)kssl_kmalloc(new_iov_len * 
			sizeof(struct sk_buff *), GFP_KERNEL);
	if (!new_skbv) {
		KSSL_DEBUG(6, "kssl_application_data_send_vec: kssl_kmalloc 2\n");
		goto mem_error;
	}
	memset(new_skbv, 0, new_iov_len * sizeof(struct sk_buff *));

	new_iov->iov_len = TLS_HEAD_NLEN;
	new_iov->iov_base = (u8 *)kssl_kmalloc(TLS_HEAD_NLEN, GFP_KERNEL);
	if (!new_iov->iov_base) {
		KSSL_DEBUG(6, "kssl_application_data_send_vec: kssl_kmalloc 3\n");
		goto mem_error;
	}

	vecno = kssl_record_vec_cpy_ptr(cr, new_iov+1, new_skbv+1, len, offset);

	if(tail_size) {
		(new_iov + vecno + 1)->iov_len = tail_size;
		(new_iov + vecno + 1)->iov_base = (u8 *)kssl_kmalloc(
			   	tail_size, GFP_KERNEL);
		if (!(new_iov + vecno + 1)) {
			KSSL_DEBUG(6, "kssl_application_data_send_vec: kssl_kmalloc 3\n");
			goto mem_error;
		}
		cr->iov_len = vecno + 2;
	}
	else {
		cr->iov_len = vecno + 1;
	}


	kssl_kfree(cr->iov);
	cr->iov = new_iov;
	kssl_kfree(cr->skbv);
	cr->skbv = new_skbv;
	cr->iov_alloc = new_iov_len;
	cr->content_len = len + TLS_HEAD_NLEN;
	cr->total_len = len + TLS_HEAD_NLEN + tail_size;
	cr->offset = 0;
	
	kssl_record_head_set(cr, ct_application_data, cr->content_len);

	status = kssl_record_send(cr);
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_application_data_send_vec: kssl_record_send\n");
		return status;
	}

	return cr->total_len;

mem_error:
	if (new_iov) {
		if (new_iov->iov_base)
			kssl_kfree(new_iov->iov_base);
		if (tail_size && vecno && !(new_iov + vecno + 1)->iov_base)
			kssl_kfree((new_iov + vecno + 1)->iov_base);
		kssl_kfree(new_iov);
	}
	if (new_skbv)
		kssl_kfree(new_skbv);

	return -ENOMEM;
}


int 
kssl_application_data_send_from_pt(kssl_record_t *cr, size_t len, 
		size_t offset, int reuse) 
{
	int status = 0;
	u32 old_conn_flag;

	KSSL_DEBUG(12, "kssl_application_data_send_from_pt: enter\n");

	old_conn_flag = cr->conn_flag;
	cr->conn_flag = KSSL_CONN_SSL;

	status = kssl_application_data_send_vec(cr, offset, len);
	if (status < 0) {
		cr->conn_flag = old_conn_flag;
	}

	KSSL_NOTICE(3, "ISSL013: Send(ssl): send decrypted data to realserver\n");

	return status;
}


int 
kssl_application_data_process(kssl_record_t *cr, alert_t *alert) 
{
	KSSL_DEBUG(12, "kssl_application_data_process enter: "
			"len=%d offset=%d\n",
			cr->content_len - cr->offset, cr->offset);

	if (cr->content_len < cr->offset) {
		kssl_record_destroy(cr);
		return 0;
	}

	return kssl_record_pt_send(cr, cr->content_len - cr->offset, 0);
}

