/**********************************************************************
 * kssl_alloc.c                                             August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifdef KSSL_ALLOC_DEBUG

#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/slab.h>

#include "kssl_alloc.h"
#include "log.h"

typedef struct {
	void *ptr;
	size_t len;
	const char *function;
	const char *file;
	size_t line;
} kssl_alloc_t;

#define KSSL_ALLOC_MAX 8192
static size_t kssl_alloc_count;
static kssl_alloc_t kssl_alloc_table[KSSL_ALLOC_MAX];


void kssl_alloc_cleanup(void)
{
	size_t i;

	for (i = 0; i < KSSL_ALLOC_MAX; i++) {
		if (!kssl_alloc_table[i].ptr)
			continue;
		kfree(kssl_alloc_table[i].ptr);
		kssl_alloc_table[i].ptr = NULL;
	}
}


void kssl_alloc_dump(void)
{
	size_t i;

	for (i = 0; i < KSSL_ALLOC_MAX; i++) {
		if (!kssl_alloc_table[i].ptr)
			continue;
		KSSL_DEBUG(3, "%5u: 0x%p %6u %s(%s:%u)\n", i,
				kssl_alloc_table[i].ptr,
				kssl_alloc_table[i].len,
				kssl_alloc_table[i].function,
				kssl_alloc_table[i].file,
				kssl_alloc_table[i].line);
		if (! (i % 128))
			if (!in_softirq())
				cond_resched();

	}
}


void *
__kssl_kmalloc(size_t len, int flags,
		const char *function, const char *file, size_t line)
{
	size_t slot;
	void *ptr;

	if (kssl_alloc_count >= KSSL_ALLOC_MAX) {
		KSSL_DEBUG(6, "%s(%s:%u): kssl_kmalloc: no free slots\n",
				function, file, line);
		kssl_alloc_dump();
		return NULL;
	}

	if (!kssl_alloc_count)
		memset(kssl_alloc_table, 0, 
				sizeof(kssl_alloc_t) * KSSL_ALLOC_MAX);

	for (slot = 0; slot < KSSL_ALLOC_MAX; slot++) {
		if(!kssl_alloc_table[slot].ptr)
			break;
	}
	if (slot == KSSL_ALLOC_MAX) {
		KSSL_DEBUG(1, "BUG! %s(%s:%u): kssl_kmalloc: "
				"could not find a slot\n",
				function, file, line);
		return NULL;
	}

	ptr = kmalloc(len, flags);
	
	KSSL_DEBUG(12, "%s(%s:%u): kssl_kmalloc p=%p len=%u\n",
			function, file, line, ptr, len);

	if (ptr) {
		kssl_alloc_table[slot].ptr = ptr;
		kssl_alloc_table[slot].len = len;
		kssl_alloc_table[slot].function = function;
		kssl_alloc_table[slot].file = file;
		kssl_alloc_table[slot].line = line;
		kssl_alloc_count++;
	}

	return ptr;
}


void 
__kssl_kfree(void *ptr, const char *function, const char *file, size_t line)
{
	size_t slot;

	KSSL_DEBUG(12, "%s(%s:%u): kssl_kfree p=%p\n",
			function, file, line, ptr); 

	if (!kssl_alloc_count) {
		KSSL_DEBUG(1, "BUG! %s(%s:%u): kssl_kfree(%p): \n"
				"no slots used\n", function, file, line, ptr);
		return;
	}

	for (slot = 0; slot < KSSL_ALLOC_MAX; slot++) {
		if(kssl_alloc_table[slot].ptr == ptr)
			break;
	}
	if (slot == KSSL_ALLOC_MAX) {
		KSSL_DEBUG(1, "BUG! %s(%s:%u): kssl_kfree(%p): "
				"could find slot\n",
				function, file, line, ptr);
		return;
	}

	kssl_alloc_table[slot].ptr = NULL;
	kssl_alloc_count--;

	kfree(ptr);
}

#endif /* KSSL_ALLOC_DEBUG */
