/**********************************************************************
 * socket.c                                                 August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/kernel.h>
#include <linux/net.h>
#include <linux/version.h>

#include <net/sock.h>
#include <net/tcp.h>
#include <linux/un.h>

#include "socket.h"
#include "log.h"

int kssl_socket_inet_open_listen(const u32 addr, const u16 port, 
		struct socket **sock)
{
	int status;
	struct sockaddr_in inaddr;

	status = sock_create(PF_INET, SOCK_STREAM, IPPROTO_TCP, sock);
	if (status < 0)
		return status;

	inaddr.sin_family = AF_INET;
	inaddr.sin_addr.s_addr = htonl(addr);
	inaddr.sin_port = htons(port);

	(*sock)->sk->reuse   = 1;

	status = (*sock)->ops->bind(*sock, (struct sockaddr*)&inaddr, 
			sizeof(inaddr));
	if(status < 0)
		goto error;

	status = (*sock)->ops->listen(*sock, 7);
	if(status < 0)
		goto error;

	return 0;
error:
	kssl_socket_close(*sock);
	*sock = NULL;
	return status;
}

int kssl_socket_inet_open_connect(const u32 addr, const u16 port, 
		struct socket **sock)
{
	int status;
	struct sockaddr_in inaddr;

	status = sock_create(PF_INET, SOCK_STREAM, IPPROTO_TCP, sock);
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_socket_open_connect: "
				"sock_create\n");
		return status;
	}

	inaddr.sin_family = AF_INET;
	inaddr.sin_addr.s_addr = htonl(addr);
	inaddr.sin_port = htons(port);

	/*
	status = (*sock)->ops->bind(*sock, (struct sockaddr*)&inaddr, 
			sizeof(inaddr));
	if(status < 0) {
		KSSL_DEBUG(6, "kssl_socket_open_connect: bind\n");
		goto error;
	}
	*/

	status = (*sock)->ops->connect(*sock, (struct sockaddr*)&inaddr, 
			sizeof(inaddr), 0);
	if(status < 0) {
		KSSL_DEBUG(6, "kssl_socket_open_connect: connect\n");
		goto error;
	}

	return 0;

error:
	kssl_socket_close(*sock);
	*sock = NULL;
	return status;

}

int
kssl_socket_unix_open_connect(const char *peername, struct socket **sock)
{
	int status;
	struct sockaddr_un unaddr;

	*sock = NULL;

	if (strlen(peername) > UNIX_PATH_MAX-1) {
		KSSL_DEBUG(6, "un_open_connect: peername is too long\n");
		return -EINVAL;
	}

	unaddr.sun_family = AF_UNIX;
	strcpy(unaddr.sun_path, peername);

	status = sock_create(AF_UNIX, SOCK_STREAM, 0, sock);
	if (status < 0) {
		KSSL_DEBUG(6, "un_open_connect: sock_create\n");
		return status;
	}

	status = (*sock)->ops->connect(*sock, (struct sockaddr*)&unaddr,
			sizeof(unaddr), 0);
	if (status < 0) {
		KSSL_DEBUG(9, "un_open_connect: connect: %d\n", status);
		kssl_socket_close(*sock);
		*sock = NULL;
		return status;
	}

	return 0;
}

void kssl_socket_close(struct socket *sock)
{
	if (!sock)
		return;

	sock_release(sock);
}

int kssl_socket_accept(struct socket *sock, struct socket **new)
{
	int status;

	if (!sock)
		return -EINVAL;

	if (!sock->sk->tp_pinfo.af_tcp.accept_queue)
		return(0);

	*new = sock_alloc();
	if (!*new)
		return -ENOMEM;

	(*new)->type = sock->type;
	(*new)->ops = sock->ops;

	status = sock->ops->accept(sock, *new, O_NONBLOCK);
	if (status < 0) {
		sock_release(*new);
		return status;
	}

	if ((*new)->sk->state == TCP_CLOSE) {
		sock_release(*new);
		return -ECONNABORTED;
	}

	return 1;
}


