/**********************************************************************
 * sched.c                                                  August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

static struct l7vs_scheduler *l7vs_sched_load(char *name);
static void l7vs_sched_unload(struct l7vs_scheduler *sched);
static struct l7vs_scheduler *l7vs_sched_lookup(char *name);
static gint l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name);

static GList *l7vs_sched_list = NULL;

struct l7vs_scheduler *
l7vs_sched_get(char *name)
{
        struct l7vs_scheduler *sched;

        sched = l7vs_sched_lookup(name);
        if (sched == NULL) {
                sched = l7vs_sched_load(name);
                if (sched == NULL) {
                        VANESSA_LOGGER_ERR("Scheduler module not found"
                                           " (maybe module problem)");
                        return sched;
                }
                sched->refcnt = 0;
        }

        sched->refcnt++;
        return sched;
}

void
l7vs_sched_put(struct l7vs_scheduler *sched)
{
        if (--sched->refcnt <= 0) {
                l7vs_module_remove(&l7vs_sched_list, sched);
                l7vs_sched_unload(sched);
        }
}

void
l7vs_sched_bind(struct l7vs_scheduler *sched,
                struct l7vs_service *svc)
{
        svc->scheduler = sched;
        if (sched->bind != NULL) {
                sched->bind(svc);
        }
}

void
l7vs_sched_unbind(struct l7vs_scheduler *sched,
                  struct l7vs_service *svc)
{
        svc->scheduler = NULL;
        if (sched->unbind != NULL) {
                sched->unbind(svc);
        }
}

static struct l7vs_scheduler *
l7vs_sched_load(char *name)
{
        struct l7vs_scheduler *sched;

        sched = (struct l7vs_scheduler *)l7vs_module_load(name, "sched");
        if (sched != NULL) {
                l7vs_module_register(&l7vs_sched_list, sched);
        }

        return sched;
}

static void
l7vs_sched_unload(struct l7vs_scheduler *sched)
{
        void *h;

        h = sched->handle;
        sched->fini();
        l7vs_module_unload(h);
}

static struct l7vs_scheduler *
l7vs_sched_lookup(char *name)
{
        return (struct l7vs_scheduler *)
                l7vs_module_lookup(l7vs_sched_list, name,
                                   (GCompareFunc)l7vs_sched_cmp);
}

static gint
l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name)
{
        return strcmp(sched->modname, name);
}
