/*!
 * @file 	iomux.c
 * @brief	It is a code in which sets of iomux are managed. 
 * @brief	An actual set is operated only by this code. 
 * @brief	Moreover, epoll is operated by this code. 
 * @brief	L7VSD: Linux Virtual Server for Layer7 Load Balancing
 *
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <sys/time.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"
#include "l7vs_conn.h"
#include "l7vs_iomuxlist.h"

#ifndef  MAXEVENTS
#define	 MAXEVENTS	8192		//! epoll max event size
#endif

static GHashTable*		iomux_hash = NULL;			//! fd hashtable
static struct l7vs_iomux	iomux_array[MAXEVENTS] = {};		//! iomux array
static GList*			iomux_avail_list = NULL;		//! iomux available list
static unsigned int		iomux_avail_list_length = MAXEVENTS;	//! number of iomux available list
static int			eventpoll = -1;				//! epoll fd
static struct epoll_event	event_array[MAXEVENTS];			//! event list

//inner functions
static gboolean	removeall( gpointer, gpointer, gpointer );

/*!
 * initialize iomux_hash
 *
 * @param[in]	void
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
int
l7vs_iomux_init()
{
	int i = 0;
	struct l7vs_iomux temp_iom = {-1, NULL, iomux_create, NULL};

	if (-1 !=  eventpoll) {
		VANESSA_LOGGER_ERR_UNSAFE("error / epoll already exist at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	//create epoll
	eventpoll = epoll_create(MAXEVENTS);
	if (0 > eventpoll) {
		VANESSA_LOGGER_ERR_UNSAFE("error / epoll create error at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	//new hashtable
	if (!iomux_hash) {
		iomux_hash = g_hash_table_new(&g_int_hash, &g_int_equal);
	}

	//init iomux_array and avail_list
	if (!iomux_avail_list) {
		for (i = 0; i < MAXEVENTS; ++i) {
			memcpy(iomux_array + i, &temp_iom, sizeof(struct l7vs_iomux));
			iomux_avail_list = g_list_append(iomux_avail_list, &iomux_array[i]);
		}
	}

	return 1;
}

/*!
 * finalize iomux_hash
 *
 * @param[in]	void
 * @return	void
 */
void
l7vs_iomux_fini()
{
	if (-1 == eventpoll) {
		VANESSA_LOGGER_ERR_UNSAFE("error / epoll is not initialized at:%s:%d", __FILE__, __LINE__);
		return;
	}

	//g_hash_table_forach( iomux_hash, closefd_func, NULL );
	l7vs_iomux_removeall();

	//g_hashtable destroy
	if (iomux_hash) {
		g_hash_table_destroy(iomux_hash);
		iomux_hash = NULL;
	}

	//g_list free
	if (iomux_avail_list) {
		g_list_free(iomux_avail_list);
		iomux_avail_list = NULL;
	}

	//epoll close
	close(eventpoll);
	eventpoll = -1;
	
}

/*!
 * get available iom from avail_list
 *
 * @param[in]	void
 * @return	available iom
 */
struct l7vs_iomux*
l7vs_iomux_get_from_avail_list()
{
	if (0 == iomux_avail_list_length) {
		VANESSA_LOGGER_ERR("warning / no iomux available");
		return NULL;
	}

	struct l7vs_iomux *iom = (struct l7vs_iomux *)(g_list_first(iomux_avail_list))->data;
	iomux_avail_list = g_list_remove(iomux_avail_list, (gpointer)iom);
	--iomux_avail_list_length;
	return iom;
}

/*!
 * put disuse iom to avail_list
 *
 * @param[in]	iom	return iom
 * @return	void
 */
void
l7vs_iomux_put_to_avail_list(struct l7vs_iomux *iom)
{
	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}

	iom->fd = -1;
	iom->callback = NULL;
	iom->status = iomux_create;
	iom->data = NULL;

	iomux_avail_list = g_list_append(iomux_avail_list, (gpointer)iom);
	++iomux_avail_list_length;
}

/*!
 * iomux is added to hashtbale
 *
 * @param[in]	*iom	iompointer
 * @param[in]	action	iom epoll action
 * @return	void
 */
void 
l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action)
{
	int ret = 0;
	struct epoll_event event;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	if (-1 == iom->fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fd is not specified at:%s:%d", __FILE__, __LINE__);
		return;
	}

	event.data.ptr = iom;
	switch( action ){
		case iom_read:
			event.events = EPOLLET | EPOLLIN | EPOLLHUP;
			break;
		case iom_write:
			event.events = EPOLLET | EPOLLOUT | EPOLLHUP;
			break;
		default:
			//error !
			VANESSA_LOGGER_ERR_UNSAFE("error / invalid action at:%s:%d" __FILE__, __LINE__);
			return;
	}

	ret = epoll_ctl( eventpoll, EPOLL_CTL_ADD, iom->fd, &event );
	g_hash_table_insert(iomux_hash, &(iom->fd), iom );
}

/*!
 * iomux is remove from hashtable
 *
 * @param[in]	*iom	iompointer
 * @return	void
 */
void
l7vs_iomux_remove(struct l7vs_iomux* iom)
{
	int ret = 0;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	if (-1 == iom->fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fd is not specified at:%s:%d", __FILE__, __LINE__);
		return;
	}
	
	ret = epoll_ctl(eventpoll, EPOLL_CTL_DEL, iom->fd, NULL);
	g_hash_table_remove(iomux_hash, &(iom->fd));
}

/*!
 * iomux modify epoll events
 *
 * @param[in]	*iom	iom pointer
 * @param[in]	action	iom action mode
 * @return	void
 */
void
l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action)
{
	int ret = 0;
	struct epoll_event	event;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	if (-1 == iom->fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fd is not specified at:%s:%d", __FILE__, __LINE__);
		return;
	}

	event.data.ptr = iom;
	switch (action) {
		case	iom_read:
			event.events = EPOLLET | EPOLLIN | EPOLLHUP;
			break;
		case	iom_write:
			event.events = EPOLLET | EPOLLOUT | EPOLLHUP;
			break;
		case	iom_hup:
			event.events = EPOLLET | EPOLLHUP;
			break;
		default:
			//error !
			VANESSA_LOGGER_ERR_UNSAFE("error / invalid action at:%s:%d", __FILE__, __LINE__);
			return;
	}
	ret = epoll_ctl(eventpoll, EPOLL_CTL_MOD, iom->fd, &event);
}

/*!
 * close all filedescriptor
 *
 * removeall() function is called l7vs_iomux_delall()
 * GHashTable is don't all contenior;
 * but function callback set all contenor;
 */
static gboolean
removeall(gpointer key, gpointer value, gpointer userdata)
{
	struct l7vs_iomux *iom = (struct l7vs_iomux *)value;
	epoll_ctl(eventpoll, EPOLL_CTL_DEL, iom->fd, NULL);
	return TRUE;
}

/*!
 * all contnior delete hash_table
 *
 * @return	void*
 */ 
void
l7vs_iomux_removeall()
{
	g_hash_table_foreach_remove(iomux_hash, removeall, NULL);
}

/*!
 * polling function for epoll.
 *
 * @param[in]	*timo	waittime;
 * @return	error	
 */
int
l7vs_iomux_poll(struct timeval *timo)
{
	int	fdnum;
	int	i;
	int	ret;
	struct l7vs_iomux	*iom;
	struct l7vs_conn	*conn;

	fdnum = epoll_wait(eventpoll, event_array, MAXEVENTS, 0);
	for (i = 0; i < fdnum; ++i) {
		if (event_array[i].data.ptr) {
			iom = (struct l7vs_iomux *)event_array[i].data.ptr;

			if (iom->status != iomux_create) {
				if (event_array[i].events & (EPOLLHUP | EPOLLERR)) {
					// disconnect when sending
					conn = (struct l7vs_conn *)iom->data;
					// destroy conn
					l7vs_conn_destroy(conn);

				} else if (event_array[i].events & (EPOLLIN | EPOLLOUT)) {
					if (iom->status != iomux_disabled) {
						if (iom->callback) {
							// if there is mismatch between status and events,
							// read event will regist after, (when send was finished) 
							// therefore, through callback
							if (!(event_array[i].events & EPOLLIN &&
							      (iomux_conn_sending == iom->status ||
							       iomux_conn_sending_busy == iom->status))) {
								// execute callback
								ret = iom->callback(iom);
							}
						} else {
							VANESSA_LOGGER_ERR_UNSAFE("error / callback not bind at:%s:%d", __FILE__, __LINE__);
						}
					}
				} else {
					VANESSA_LOGGER_ERR_UNSAFE("error / unkown event:%x at:%s:%d", event_array[i].events, __FILE__, __LINE__);
				}
			} else {
				VANESSA_LOGGER_DEBUG_UNSAFE("error / invalid status:iomux_create at:%s:%d", __FILE__, __LINE__);
			}
		} else {
			VANESSA_LOGGER_ERR_UNSAFE("error / epolldata.data.ptr is null at:%s:%d", __FILE__, __LINE__);
		}
	}

	return 1;

}
