/*! @file
 *  @brief Sync structure header
 *
 *  @author NTT COMWARE
 *  @date 2007-05-18
 *  @version 0.1
 */
#ifndef L7VS_SYNC_H
#define L7VS_SYNC_H

//! Maximum size of transfer raw data
#define SEND_DATA_SIZE 500
//! Maximum size of input/output raw data
#define DATA_SIZE 480


//! Transfer data between active server and standby server.
struct sync_send_data
{
	unsigned long long serial;  //!< Serial number for checking data time.
	unsigned short size ;       //!< Data size for checking packet loss.
	char word[SEND_DATA_SIZE];  //!< Raw data
};

//! Input/Output data from/to external program. This data will send/receive between active server and standby server.
struct sync_data
{
	unsigned short size;        //!< Data size for checking packet loss.
	unsigned char type;         //!< Data type for distinction.
	char word[DATA_SIZE];       //!< Raw data
};

//! Operation mode
enum sync_operation{
	SYNC_SEND,                  //!< Send mode
	SYNC_RECV                   //!< Receive mode
};

//! Error type
enum sync_err_t{
	SYNC_OK,                    //!< Success.
	SYNC_INITIALIZE_ALREADY,    //!< It had already been initialized.
	SYNC_INITIALIZE_YET,        //!< It had not been initialized yet.
	SYNC_SOCK_NG,               //!< socket() NG
	SYNC_BIND_NG,               //!< bind() NG
	SYNC_SEND_NG,               //!< sendto() NG
	SYNC_RECV_NG,               //!< recvfrom() NG
	SYNC_SIZE_OVER,             //!< Size is over.
	SYNC_CLOSE_NG,              //!< close() NG
	SYNC_NIC_NG,                //!< Binding network device NG
	SYNC_NIC_NULL,              //!< Points to NIC is NULL
	SYNC_IP_NG,                 //!< IP address format is wrong
	SYNC_IP_NULL,               //!< Points to IP address is NULL
	SYNC_SERVNAME_NG,           //!< Service name is wrong
	SYNC_SERVNAME_NULL,         //!< Points to service name is NULL
	SYNC_PORT_ZERO,             //!< Port number is zero
	SYNC_UNKNOWN_MODE,          //!< Operation mode is not SYNC_SEND or SYNC_RECV
	SYNC_FD_NULL,               //!< Points to file descriptor is NULL
	SYNC_DATA_NULL,             //!< Points to input data is NULL
	SYNC_NOT_SEND_MODE,         //!< Initialized by NOT send mode
	SYNC_NOT_RECV_MODE,         //!< Initialized by NOT receive mode
};


//! sync initialize data
struct l7vs_sync_init_data {
	const char *ip_addr;
	const char *servname;
	const char *nic;
	enum sync_operation mode;
	int fd;
	size_t size;
};


/*! Initialize network socket
 *
 *  @param ip_addr Destination IP address (IPv4 or IPv6)
 *  @param servname Destination service name or port number
 *  @param nic Network device (ex. eth0, bond0, hme0, etc.)
 *  @param mode Operation mode (SYNC_SEND : send mode / SYNC_RECV : receive mode)
 *  @param fd File descriptor.
 *
 *  @return Initialize results. (SYNC_OK/SYNC_INITIALIZE_ALREADY/SYNC_SOCK_NG/SYNC_BIND_NG)
 *  @retval SYNC_OK Everything is ok
 *  @retval SYNC_INITIALIZE_ALREADY Network socket had already been initialized.
 *  @retval SYNC_SOCK_NG socket() NG
 *  @retval SYNC_BIND_NG bind() NG
 */	
extern enum sync_err_t sync_sock_init(const char *ip_addr, const char *servname, const char *nic,
			const enum sync_operation mode, int *fd);


/*! Send transfer data to standby server
 *
 *  @param data Points to input data from external program. This will be send to standby server.
 *
 *  @return Send results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_SEND_NG sendto() NG
 */
extern enum sync_err_t sync_send_to(const struct sync_data *send_data);


/*! Receive transfer data from active server
 *
 *  @param  recv_data Points to output data from external program.
 *
 *  @return Send results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_NOT_RECV_MODE Initialize flag is not receive flag.
 *  @retval SYNC_DATA_NULL Points to receive data is NULL.
 *  @retval SYNC_RECV_NG recvfrom() NG
 *  @retval SYNC_SIZE_NG Size of received data is different from send size.
 */ 
extern enum sync_err_t sync_recv_from(struct sync_data *recv_data);


/*! Finalize network socket
 *
 *  @return Finalize results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_CLOSE_NG close() NG
 */
extern enum sync_err_t sync_sock_fini(void);

#endif /* L7VS_SYNC_H */
