/*!
 * @file lsock.c
 * @bref listen socket control.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_lsock.h"
#include "l7vs_module.h"
#include "l7vs_conn.h"
#include "l7vs_service.h"
#include "l7vs_sched.h"

/* static functions */
static int	l7vs_lsock_accept(struct l7vs_lsock *lsock);
static void	l7vs_lsock_table_add(struct l7vs_lsock *lsock);
static void	l7vs_lsock_table_remove(struct l7vs_lsock *lsock);
static gint	l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b);
static int	l7vs_lsock_callback(struct l7vs_iomux *iom );

//! l7vs_lsock list 
static GList *l7vs_lsock_list;

/*!
 * inner function of fini.
 * all element free.
 */
void	freeAllList( gpointer data, gpointer userdata ){
	struct l7vs_lsock*	lsock = (struct l7vs_lsock*)data;
	free( lsock );
}

/*!
 * initialize functions
 * @param[in]	void
 * @return	everydays zero.
 */
int
l7vs_lsock_init(void)
{       
        if(l7vs_lsock_list == NULL )
        {       
                return 0;
        }
        g_list_foreach( l7vs_lsock_list, freeAllList, NULL );
        g_list_free( l7vs_lsock_list );
        l7vs_lsock_list = NULL;
        return 0;
}

/*!
 * finalize function.
 *  free list element memory.
 *  drop list memory.
 * @param[in]	void
 * @return	void
 */
void
l7vs_lsock_fini(void)
{
	g_list_foreach( l7vs_lsock_list, freeAllList, NULL );
	g_list_free( l7vs_lsock_list );
	l7vs_lsock_list = NULL;
}
/*!
 * serch socket
 * if target soket have link list then return having socket.
 * else create new socket to return.
 * @param[in]	sin	socket address struct( if ipv6 may be change this struct ) 
 * @param[in]	proto	TCS/UDP select.
 * @param[in]	backlog	
 */
struct l7vs_lsock *
l7vs_lsock_get(struct sockaddr_in *sin, uint8_t proto, int backlog)
{
        struct l7vs_lsock *lsock;
        int stype;
        int ret;

	if(sin == NULL )
	{
		return NULL;
	}

        lsock = l7vs_lsock_table_lookup(sin, proto);
        if (lsock != NULL) {
                lsock->refcnt++;
                return lsock;
        }
                                                         
        switch (proto) {
        case IPPROTO_TCP:
                stype = SOCK_STREAM;
                break;
        case IPPROTO_UDP:
                stype = SOCK_DGRAM;
                break;
        default:
                VANESSA_LOGGER_ERR_UNSAFE("Protocol number should be"
                                          " TCP or UDP (%d)",
                                          proto);
                return NULL;
        }

        lsock = (struct l7vs_lsock *)calloc(1, sizeof(*lsock));
        if (lsock == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate lsock");
                return lsock;
        }

	lsock->iom = l7vs_iomux_get_from_avail_list();
        if (!lsock->iom) {
                VANESSA_LOGGER_ERR("can not get lsock_iomux");
                return NULL;
        }

        VANESSA_LOGGER_DEBUG_UNSAFE("creating lsock %p", lsock);
        lsock->proto = proto;
        lsock->iom->fd = socket(PF_INET, stype, proto);  //create a socket to get connection request from the client
        if (lsock->iom->fd < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                free(lsock);
                return NULL;
        }

        ret = bind(lsock->iom->fd, (struct sockaddr *)sin, sizeof(*sin)); //binding the socket for incoming client request
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not bind socket: %s",
                                          strerror(errno));
                close(lsock->iom->fd);
                free(lsock);
                return NULL;
        }
        lsock->addr = *sin;

        ret = listen(lsock->iom->fd, backlog); //listening for client requests
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not listen: %s\n",
                                          strerror(errno));
                close(lsock->iom->fd);
                free(lsock);
                return NULL;
        }

        lsock->refcnt = 1;

        lsock->iom->callback = l7vs_lsock_callback;
        lsock->iom->data = lsock;
	lsock->iom->status = iomux_lsock_connect_waiting;
        l7vs_lsock_table_add(lsock);  //Add socket in the list. It may be used for maintaining session (Am not sure)
//        l7vs_iomux_add(&lsock->iom, L7VS_IOMUX_READ);

	l7vs_iomux_add( lsock->iom, iom_read );
        return lsock;
}

/*!
 * lisning socket remove list and iomuxlist.
 * @param[in]	lsock	removing lisning socket
 * @return	void
 */

void
l7vs_lsock_put(struct l7vs_lsock *lsock)
{
	if(lsock == NULL)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / lsock is null at:%s:%d", __FILE__, __LINE__);
		return;
	}

	lsock->iom->status = iomux_lsock_released;

        if (--lsock->refcnt > 0)
	{
		lsock->iom->status = iomux_lsock_connect_waiting;
                return;
	}

	lsock->iom->status = iomux_lsock_destroyed;

        VANESSA_LOGGER_DEBUG_UNSAFE("removing lsock %p", lsock);
        l7vs_iomux_remove(lsock->iom);
        l7vs_lsock_table_remove(lsock);

	if(lsock->iom->fd > 0)
	{
        	close(lsock->iom->fd);
	}
	l7vs_iomux_put_to_avail_list(lsock->iom);
        free(lsock);
}

/*!
 * lsock_list append new lsock ponter
 * @param[in]	lsock	to insert table lsock
 * @return	void
 */
static void
l7vs_lsock_table_add(struct l7vs_lsock *lsock)
{
        l7vs_lsock_list = g_list_append(l7vs_lsock_list, (gpointer)lsock);
}

/*!
 * lsock_list remove lsock (don't free element)
 * @param[in]	to remove lsock
 * @return	void
 */
static void
l7vs_lsock_table_remove(struct l7vs_lsock *lsock)
{
        l7vs_lsock_list = g_list_remove(l7vs_lsock_list, (gpointer)lsock);
}


/*!
 * serch table for lsock.
 * @param[in]	*sin	socketaddr_in struct pointer (ex. to change then IPv6 target )
 * @param[in]	*proto	port no
 * @return	*l7vs_lsock
 */
struct l7vs_lsock *
l7vs_lsock_table_lookup(struct sockaddr_in *sin, uint8_t proto)
{
        struct l7vs_lsock tmpl;
        GList *l;

	if(sin == NULL)
	{
		return NULL;
	}

        tmpl.addr = *sin;
        tmpl.proto = proto;

        l = g_list_find_custom(l7vs_lsock_list, (gpointer)&tmpl,
                               l7vs_lsock_addr_cmp);
        if (l == NULL)
                return NULL;
        return (struct l7vs_lsock *)l->data;  //for checking up and finding the socket in the list for a particular client socket(not sure)
}

/*!
 * using foreach function from Compare
 * @param	a l7vs_lsock pointer
 * @param	b l7vs_lsock pointer
 * @return	compare pattern 
 */
static gint
l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b)
{
        struct l7vs_lsock *la = (struct l7vs_lsock *)a;
        struct l7vs_lsock *lb = (struct l7vs_lsock *)b;


        if (la->addr.sin_addr.s_addr != lb->addr.sin_addr.s_addr)
                return la->addr.sin_addr.s_addr - lb->addr.sin_addr.s_addr;
        if (la->addr.sin_port != lb->addr.sin_port)
                return la->addr.sin_port - lb->addr.sin_port;
        if (la->proto != lb->proto)
                return la->proto - lb->proto;

        return 0;
}

int
l7vs_lsock_select_service(struct l7vs_lsock *lsock, 
                          struct l7vs_conn *conn,
                          char *buf,
                          size_t len,
                          struct l7vs_service **srv_ret,
                          struct l7vs_dest **dest_ret,
                          int *tcps_ret)
{
        GList *l;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        int ret, val;
        int tcps;
	struct l7vs_service *srv_tmp = NULL;	//! temporary for srv
	struct l7vs_dest *dest_tmp = NULL;	//! temporary for dest
	int tcps_tmp = 1;			//! temporary for tcps
	size_t len_tmp = 0;			//! temporary for len
	int sorry_save = 0;			//! sorry dest save flag
	struct l7vs_service *srv_tmp2 = NULL;	//! temporary for srv when match_cldata NG
	struct l7vs_dest *dest_tmp2 = NULL;	//! temporary for dest when match_cldata NG
	int tcps_tmp2 = 1;			//! temporary for tcps when match_cldata NG
	size_t len_tmp2 = 0;			//! temporary for len when match_cldata NG
	int sorry_save2 = 0;			//! sorry dest save flag when match_cldata NG

        val = -1;
        l = lsock->srv_list;
        while (l != NULL) {
                srv = (struct l7vs_service *)l->data;
                dest = NULL;
                tcps = 1;
                ret = srv->pm->match_cldata(srv, conn, buf, &len, &dest, &tcps);
                if (ret == 0) {
			// check sorry status if ret == 0
			// sorry_check argument 1 means check connection count
			if (l7vs_sched_sorry_check(srv, 1)) {
				if (!sorry_save) {
					// save sorry dest to xxx_tmp only first time
					srv_tmp = srv;
					dest_tmp = dest;
					tcps_tmp = tcps;
					len_tmp = len;
					sorry_save = 1;
				}
			} else {
				// check cldata length if sorry_check != 1 (not sorry status)
				if (len > conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE) {
					VANESSA_LOGGER_ERR("bufsize too long modified by protomod ");
				} else {
					// service and destination is decided
					*srv_ret = srv;
					*dest_ret = dest;
					*tcps_ret = tcps;
					conn->cldata_len = len;
					sorry_save = 0;
					sorry_save2 = 0;
					val = 1;
					break;
				}
			}
		}
		// when match_cldata NG
		if (!sorry_save2) {
			// save sorry dest to xxx_tmp2 only first time
			srv_tmp2 = srv;
			dest_tmp2 = dest;
			tcps_tmp2 = tcps;
			len_tmp2 = len;
			sorry_save2 = 1;
		}
		// get next srv from srv_list
                l = g_list_next(l);
        }

	if (sorry_save) {
		// saved sorry dest is exist
		// set sorry-server destination
		//*dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp, NULL, 1);
		*dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp, NULL, 0);
		if (!*dest_ret) {
			VANESSA_LOGGER_ERR("sorry-server dest is NULL");
			return -1;
		}
		*srv_ret = srv_tmp;
		*tcps_ret = tcps_tmp;
		conn->cldata_len = len_tmp;
		conn->sorry_conn_flag = 1;
		val = 1;
	} else if (sorry_save2) {
		// saved sorry dest2 is exist
		// set sorry-server destination
		//*dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp2, NULL, 1);
		*dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp2, NULL, 0);
		if (!*dest_ret) {
			VANESSA_LOGGER_ERR("sorry-server dest is NULL");
			return -1;
		}
		*srv_ret = srv_tmp2;
		*tcps_ret = tcps_tmp2;
		conn->cldata_len = len_tmp2;
		conn->sorry_conn_flag = 1;
		val = 1;
	}
	// return val=1 service and destination is decided (to real-server or sorry-server)
	// return val=-1 service and destination not decided (match_cldata result all NG)
        return val;
}

/*
 * service registration
 */
void
l7vs_lsock_add_service(struct l7vs_lsock *lsock,
                       struct l7vs_service *srv)
{
        lsock->srv_list = g_list_append(lsock->srv_list, srv);
}

void
l7vs_lsock_remove_service(struct l7vs_lsock *lsock,
                          struct l7vs_service *srv)
{
        lsock->srv_list = g_list_remove(lsock->srv_list, srv);
}

/*!
 * socket acception function.
 * create connection
 * @param[in]	socket option
 * @return	success = 0 / false = -1
 */
static int
l7vs_lsock_accept(struct l7vs_lsock *lsock)
{
	if(lsock == NULL)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / lsock is null at:%s:%d", __FILE__, __LINE__);
		return -1;
	}
	if (iomux_lsock_accepted != lsock->iom->status)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", lsock->iom->status, __FILE__, __LINE__);
		return -1;
	}

        if (!l7vs_conn_create(lsock->iom->fd, lsock))
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / conn create failed at:%s:%d", __FILE__, __LINE__);
		lsock->iom->status = iomux_lsock_conn_create_error;
		return -1;
	}

	lsock->iom->status = iomux_lsock_conn_created;
	return 0;
}

static int
l7vs_lsock_callback(struct l7vs_iomux *iom )
{
	int ret;
	if(iom == NULL)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return -1;
	}
	if (iomux_lsock_connect_waiting != iom->status)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return -1;
	}

	iom->status = iomux_lsock_accepted;
        ret = l7vs_lsock_accept((struct l7vs_lsock *)iom->data); //for accepting data from clients
	if (-1 == ret)
	{
		VANESSA_LOGGER_ERR_UNSAFE("error / lsock accept failed at:%s:%d", __FILE__, __LINE__);
		iom->status = iomux_lsock_connect_waiting;
		return -1;
	}

	iom->status = iomux_lsock_connect_waiting;
	l7vs_iomux_mod( iom, iom_read );

	return 0; 
}
