/*
 * @file  config.c
 * @brief the configulation interface component
 * @brief it receives configulation command from l7vsadm,
 * @brief execute coresponding action and return result
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"

/* Added headers for sending messages to l7vsagent */
#include <stdio.h>
#include <errno.h>
#include <sys/msg.h>
#include <string.h>
#include <signal.h>

#include <netdb.h>
#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_service.h"
#include "l7vs_sync.h"
#include "l7vs_sched.h"
#include "sync.h"

static int	l7vs_config_callback( struct l7vs_iomux* );
static void	l7vs_config_dispatch( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_list_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_list_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_add_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_del_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_edit_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_add_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_del_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_edit_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_flush_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_sync_init(union l7vs_config_req *arg, struct sockaddr_un *addr);
static void	l7vs_config_sync_fini(union l7vs_config_req *arg, struct sockaddr_un *addr);
static int	l7vs_config_send_response( struct sockaddr_un*, struct iovec*, int );

char	servicebuf[NI_MAXHOST + NI_MAXSERV + 2];
const char l7vs_config_sockname[] = L7VS_CONFIG_SOCKNAME;

static struct l7vs_iomux *l7vs_config_iomux;

int
l7vs_config_init(void)
{
        int s;
        int ret;
        int opt;
        mode_t mode;
        struct sockaddr_un addr;
        socklen_t len;

	if (l7vs_config_iomux) {
                VANESSA_LOGGER_ERR("config is already initialized.");
		return -1;
	}

        if (sizeof(addr.sun_path) < sizeof(l7vs_config_sockname)) {
                VANESSA_LOGGER_ERR("Internal error."
                                   " The config socket name is too large.");
                return -1;
        }

        s = socket(PF_LOCAL, SOCK_DGRAM, 0);
        if (s < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                return s;
        }

        opt = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("setsockopt SO_PASSCRED: %s",
                                          strerror(errno));
                close(s);
                return ret;
        }

        memset(&addr, 0, sizeof(addr));
        addr.sun_family = AF_LOCAL;
        strcpy(addr.sun_path, l7vs_config_sockname);
        len = SUN_LEN(&addr);

        mode = umask(0077);
        if (mode < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("umask: %s", strerror(errno));
                close(s);
                return -1;
        }

        ret = bind(s, (struct sockaddr *)&addr, len);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("bind: %s", strerror(errno));
                if (errno == EADDRINUSE) {
                        VANESSA_LOGGER_ERR("Config socket already exists. "
                                           "You could remove it if"
                                           " the l7vs daemon is not running");
                }
                close(s);
                umask(mode);
                return ret;
        }
        umask(mode);

	l7vs_config_iomux = l7vs_iomux_get_from_avail_list();
	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR("can not get config_iomux");
		return -1;
	}
	
        l7vs_config_iomux->fd = s;
        l7vs_config_iomux->data = NULL;
        l7vs_config_iomux->callback = l7vs_config_callback;
        l7vs_config_iomux->status = iomux_config_command_waiting;
        //l7vs_iomux_add(&l7vs_config_iomux, L7VS_IOMUX_READ);
	l7vs_iomux_add( l7vs_config_iomux, iom_read );

        return 0;
}

void
l7vs_config_fini(void)
{
	if (l7vs_config_iomux) {
		l7vs_config_iomux->status = iomux_config_destroyed;
		l7vs_iomux_remove(l7vs_config_iomux);
		if (-1 != l7vs_config_iomux->fd) {
		        close(l7vs_config_iomux->fd);
		}
		l7vs_iomux_put_to_avail_list(l7vs_config_iomux);
		l7vs_config_iomux = NULL;
	}
        unlink(l7vs_config_sockname);
}

static int
l7vs_config_callback(struct l7vs_iomux *iom )
{
        struct msghdr msg;
        struct sockaddr_un addr;
        struct iovec iov;
        struct ucred *cred;
        struct cmsghdr *cmsg;
        unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
        union l7vs_config_req req;
        int ret;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return -1;
	}
	if (iomux_config_command_waiting != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return -1;
	}

        cmsg = (struct cmsghdr *)cbuf;
        cmsg->cmsg_len = sizeof(cbuf);

        memset(&addr, 0, sizeof(addr));
        memset(&msg, 0, sizeof(msg));
        msg.msg_name = &addr;
        msg.msg_namelen = sizeof(addr);
        msg.msg_iov = &iov;
        msg.msg_iovlen = 1;
        msg.msg_control = cmsg;
        msg.msg_controllen = sizeof(cbuf);

        iov.iov_base = &req;
        iov.iov_len = sizeof(req);

        ret = recvmsg(iom->fd, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("recvmsg on config socket: %s",
                                          strerror(errno));
                return -1;
        }

        if (msg.msg_flags & (MSG_TRUNC|MSG_CTRUNC)) {
                VANESSA_LOGGER_ERR("Unknown message on config socket"
                                   " (message too large)");
                VANESSA_LOGGER_DEBUG_UNSAFE("msg_flags=%x", msg.msg_flags);
                return -1;
        }

        if (! (cmsg->cmsg_level == SOL_SOCKET &&
               cmsg->cmsg_type == SCM_CREDENTIALS)) {
                VANESSA_LOGGER_ERR("Could not receive a remote credential");
                return -1;
        }

        /*
         * actually credential check is unnecessary,
         * at least for Linux 2.4.20.
         */
        cred = (struct ucred *)CMSG_DATA(cmsg);
        if (cred->uid != 0) {
                VANESSA_LOGGER_ERR("Request from unprivileged user");
                return -1;
        }

        l7vs_config_dispatch(&req, &addr);

	l7vs_config_iomux->status = iomux_config_command_waiting;

        return 0;
}

struct l7vs_config_table {
        enum l7vs_config_command cmd;
        void (*func)(union l7vs_config_req *req, struct sockaddr_un *addr);
};

struct l7vs_config_table l7vs_config_table[] = {
        {L7VS_CONFIG_LIST_VS,   l7vs_config_list_vs},
        {L7VS_CONFIG_LIST_RS,   l7vs_config_list_rs},
        {L7VS_CONFIG_ADD_VS,    l7vs_config_add_vs},
        {L7VS_CONFIG_DEL_VS,    l7vs_config_del_vs},
        {L7VS_CONFIG_EDIT_VS,   l7vs_config_edit_vs},
        {L7VS_CONFIG_ADD_RS,    l7vs_config_add_rs},
        {L7VS_CONFIG_DEL_RS,    l7vs_config_del_rs},
        {L7VS_CONFIG_EDIT_RS,   l7vs_config_edit_rs},
        {L7VS_CONFIG_FLUSH_VS,  l7vs_config_flush_vs},
	{L7VS_CONFIG_SYNC_INIT, l7vs_config_sync_init},
	{L7VS_CONFIG_SYNC_FINI, l7vs_config_sync_fini},
        {L7VS_CONFIG_NONE, NULL}
};

static void
l7vs_config_dispatch(union l7vs_config_req *req, struct sockaddr_un *addr)
{
        struct l7vs_config_table *t;
        struct iovec iov;
        struct l7vs_config_rsp_unknown_cmd rsp;

        for (t = l7vs_config_table; t->cmd != L7VS_CONFIG_NONE; t++) {
                if (t->cmd == req->cmd) {
                        break;
                }
        }

        if (t->cmd == L7VS_CONFIG_NONE) {
                VANESSA_LOGGER_ERR("Unknown command from config socket");
                rsp.cmd = req->cmd;
                rsp.code = L7VS_CONFIG_ERR_INVALID_COMMAND;
                iov.iov_base = &rsp;
                iov.iov_len = sizeof(rsp);
                l7vs_config_send_response(addr, &iov, 1);
                return;
        }

        t->func(req, addr);
}

static void
l7vs_config_list_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_vs *req = &arg->list_vs;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_service_arg_multi *sas;
        struct iovec iov[2];
        int iovlen;
        int len;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_list_vs_received;

        sas = NULL;
        rsp.cmd = req->cmd;
        rsp.code = 0;
        rsp.num = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        len = l7vs_service_list_service_arg(&sas, &rsp.num);
        if (len < 0) {
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        if (rsp.num == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = sas;
                iov[1].iov_len = len;
                iovlen = 2;
        }

	l7vs_config_iomux->status = iomux_config_command_list_vs_succeeded;

        l7vs_config_send_response(addr, iov, iovlen);
        if (sas != NULL) {
                free(sas);
        }
        return;
}

static void
l7vs_config_list_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_rs *req = &arg->list_rs;
        struct l7vs_config_rsp_list_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service  *srv=NULL;
        struct l7vs_dest_arg *das=NULL;
        struct iovec iov[2];
        int iovlen;
        int ret;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_list_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);

        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

	l7vs_config_iomux->status = iomux_config_command_list_rs_vs_found;

        ret = l7vs_service_list_dest_arg(srv, &das);
        if (ret < 0) {
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        rsp.num = ret;
        if (ret == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = das;
                iov[1].iov_len = ret * sizeof(*das);
                iovlen = 2;
        }

	l7vs_config_iomux->status = iomux_config_command_list_rs_succeeded;

        l7vs_config_send_response(addr, iov, iovlen);
        if (das != NULL) {
                free(das);
		das = NULL;
        }
}

static void
l7vs_config_add_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct iovec iov;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_add_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv != NULL) {
		l7vs_config_iomux->status = iomux_config_command_vs_exist_error;
                VANESSA_LOGGER_ERR("ADD_VS: service already exists");
                rsp.code = L7VS_CONFIG_ERR_VS_EXISTS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_vs_vs_not_found;

        srv = l7vs_service_create(sarg, &rsp.code);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_vs_create_error;
                VANESSA_LOGGER_ERR("ADD_VS: failed to create a service");
		goto out;
        } else {
		l7vs_config_iomux->status = iomux_config_command_add_vs_vs_created;
		VANESSA_LOGGER_INFO("ADD_VS: IFRM001: created a service ");
	}

	l7vs_config_iomux->status = iomux_config_command_add_vs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_del_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
	struct l7vs_dest *dest;
        struct iovec iov;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_del_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                VANESSA_LOGGER_DEBUG("DEL_VS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_vs_vs_found;

	while (g_list_length(srv->dest_list) > 0) {
		dest = (struct l7vs_dest *)((g_list_first(srv->dest_list))->data);
		srv->dest_list = g_list_remove(srv->dest_list, dest);
		l7vs_dest_destroy(dest);
	}

	l7vs_config_iomux->status = iomux_config_command_del_vs_dest_destroyed;

        l7vs_service_destroy(srv);

	VANESSA_LOGGER_INFO("DEL_VS: IFRM003: released virtual service");
	l7vs_config_iomux->status = iomux_config_command_del_vs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_edit_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_scheduler *sched, *sched_old;
        struct iovec iov;
	int	retval;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_edit_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                VANESSA_LOGGER_DEBUG("EDIT_VS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_vs_vs_found;

	retval = l7vs_service_set_QoS_Threashold( srv, sarg );
	if( 0 > retval ){
		l7vs_config_iomux->status = iomux_config_command_set_qos_error;
                VANESSA_LOGGER_DEBUG("EDIT_VS: QoS set failure");
                rsp.code = L7VS_CONFIG_ERR_QOSSET;
                goto out;
	}

	retval = l7vs_service_set_SorryServer_Values( srv, sarg );
	if( 0 > retval ){
		l7vs_config_iomux->status = iomux_config_command_set_sorry_server_error;
                VANESSA_LOGGER_DEBUG("EDIT_VS: SorryServer set failure");
                rsp.code = L7VS_CONFIG_ERR_SORRYSET;
                goto out;
	}

        sched = (struct l7vs_scheduler* )l7vs_sched_get(sarg->srv_arg.schedmod);
        if (sched == NULL) {
		l7vs_config_iomux->status = iomux_config_command_nosched_error;
                VANESSA_LOGGER_DEBUG("EDIT_VS: unknown scheduler");
                rsp.code = L7VS_CONFIG_ERR_NOSCHED;
                goto out;
        } else {
		VANESSA_LOGGER_INFO("EDIT_VS: IFRM002: changed virtual service");
	}

	l7vs_config_iomux->status = iomux_config_command_edit_vs_sched_got;

        sched_old = srv->scheduler;
        if (sched != sched_old) {
                l7vs_sched_unbind(sched_old, srv);
                l7vs_sched_put(sched_old);
                l7vs_sched_bind(sched, srv);
        }

	l7vs_config_iomux->status = iomux_config_command_edit_vs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_add_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct iovec iov;
        int ret;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_add_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                VANESSA_LOGGER_DEBUG("ADD_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_vs_found;

        if (l7vs_service_lookup_dest(srv, &req->darg.addr) != NULL) {
		l7vs_config_iomux->status = iomux_config_command_rs_exist_error;
                VANESSA_LOGGER_DEBUG("ADD_RS: real server already exists");
                rsp.code = L7VS_CONFIG_ERR_RS_EXISTS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_rs_not_found;

        ret = l7vs_service_add_dest(srv, &req->darg);
        if (ret < 0) {
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
		goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_del_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;
        int ret;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_del_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                VANESSA_LOGGER_DEBUG("DEL_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_vs_found;

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
		l7vs_config_iomux->status = iomux_config_command_nors_error;
                VANESSA_LOGGER_DEBUG("DEL_VS: destination nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_rs_found;

        ret = l7vs_service_remove_dest(srv, &req->darg);
        if (ret < 0) {
                /* XXX actually impossible */
                rsp.code = L7VS_CONFIG_ERR_NORS;
		goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_edit_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_edit_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_rs_vs_found;

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_RS: destination nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_rs_rs_found;

        dest->weight = req->darg.weight;

	l7vs_config_iomux->status = iomux_config_command_edit_rs_succeeded;

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_flush_vs(union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_req_flush_vs *req = &arg->flush_vs;
        struct l7vs_config_rsp_flush_vs rsp;
        struct iovec iov;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_flush_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        l7vs_service_flush_all();

	l7vs_config_iomux->status = iomux_config_command_flush_vs_succeeded;

        l7vs_config_send_response(addr, &iov, 1);
}

static int
l7vs_config_send_response(struct sockaddr_un *addr,
                          struct iovec *iov,
                          int iovlen)
{
        struct  msghdr msg;
        int ret;

	if (!l7vs_config_iomux)	{
                VANESSA_LOGGER_ERR("config_iomux is null");
		return -1;
	}

        memset(&msg, 0, sizeof(msg));
        msg.msg_name = addr;
        msg.msg_namelen = sizeof(*addr);
        msg.msg_iov = iov;
        msg.msg_iovlen = iovlen;

        ret = sendmsg(l7vs_config_iomux->fd, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("sendmsg on config socket: %s",
                                          strerror(errno));
        }
	l7vs_config_iomux->status = iomux_config_command_response_sent;

        return ret;
}

/*!
 * Configure sync initialize.
 * @param[in]	*arg	config request pointer
 * @param[in]	*addr	socket address pointer
 * @return	void
 */
static void
l7vs_config_sync_init(union l7vs_config_req *arg,
			struct sockaddr_un *addr)
{
	struct  l7vs_config_req_sync_init *req = &arg->sync_init;
	struct  l7vs_config_rsp_sync_init rsp;
	struct  iovec iov;
	int     ret;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_sync_initialize_received;

	rsp.cmd = req->cmd;
	rsp.code = 0;

	// sync_initialize
	ret = l7vs_sync_init(&req->data);
	if (ret == 1) {
		VANESSA_LOGGER_ERR("sync initialize error");
	}

	iov.iov_base = &rsp;
	iov.iov_len = sizeof(rsp);

	l7vs_config_iomux->status = iomux_config_command_sync_initialize_succeeded;

	// send response
	ret = l7vs_config_send_response(addr, &iov, 1);
	if (ret == -1) {
		VANESSA_LOGGER_ERR("error");
	}
}

/*!
 * Configure sync finalize.
 * @param[in]	*arg	config request pointer
 * @param[in]	*addr	socket address pointer
 * @return	void
 */
static void
l7vs_config_sync_fini(union l7vs_config_req *arg,
			struct sockaddr_un *addr)
{
	struct  l7vs_config_req_sync_fini *req = &arg->sync_fini;
	struct  l7vs_config_rsp_sync_fini rsp;
	struct  iovec iov;
	int     ret;

	if (!l7vs_config_iomux) {
		VANESSA_LOGGER_ERR_UNSAFE("error / config iom is null at:%s:%d", __FILE__, __LINE__);
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_sync_finalize_received;

	rsp.cmd = req->cmd;
	rsp.code = 0;

	// sync finalize
	ret = l7vs_sync_fini();
	if (ret == 1) {
		VANESSA_LOGGER_ERR("sync finalize error");
	}

	iov.iov_base = &rsp;
	iov.iov_len = sizeof(rsp);

	l7vs_config_iomux->status = iomux_config_command_sync_finalize_succeeded;

	// send response
	ret = l7vs_config_send_response(addr, &iov, 1);
	if (ret == -1) {
		VANESSA_LOGGER_ERR("error");
	}
}
