/*
 * @file  protomod.c
 * @brief the framework module of protocol module 
 * @brief it proceeds common function of protocol module
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs_module.h"
#include "l7vs_dest.h"
#include "l7vs_service.h"
#include "l7vs_sched.h"
#include "l7vs_conn.h"

static struct l7vs_protomod *l7vs_protomod_load(char *modname);
static void l7vs_protomod_unload(struct l7vs_protomod *pmod);
static gint l7vs_protomod_cmp(struct l7vs_protomod *pmod, char *name);

static int protomod_initialize(struct l7vs_service *srv, struct l7vs_conn *conn ,char *buf ,size_t len , struct l7vs_dest **dest);
static int protomod_finalize(struct l7vs_service *srv, struct l7vs_conn *conn ,char *buf , size_t len , struct l7vs_dest **dest, int resched);
 
static GList *l7vs_protomod_list = NULL;

struct l7vs_protomod *
l7vs_protomod_get(char *name)
{
	struct l7vs_protomod *pmod;

	if (name == NULL) {
		VANESSA_LOGGER_ERR("name is null");
		return NULL;
	}
	pmod = l7vs_protomod_lookup(name);
	if (pmod == NULL) {
		pmod = l7vs_protomod_load(name);
		if (pmod == NULL) {
			VANESSA_LOGGER_ERR("Protocol module not found"
					   " (maybe module problem)");
			return pmod;
		}
		pmod->refcnt = 0;
	}

	pmod->refcnt++;
	return pmod;
}

void
l7vs_protomod_put(struct l7vs_protomod *pmod)
{
	if (pmod == NULL) {
		VANESSA_LOGGER_ERR("pmod is null");
		return;
	}
	if (--pmod->refcnt <= 0) {
		l7vs_module_remove(&l7vs_protomod_list, pmod);
		l7vs_protomod_unload(pmod);
	}
}

static struct l7vs_protomod *
l7vs_protomod_load(char *modname)
{
	struct l7vs_protomod *pmod;

	if (modname == NULL) {
		VANESSA_LOGGER_ERR("modname is null");
		return NULL;
	}

	pmod = (struct l7vs_protomod *)l7vs_module_load(modname, "protomod");

	if (pmod != NULL) {
		/*[ added (2005/05/31) */
		pmod->initialize = protomod_initialize;
		pmod->finalize = protomod_finalize;
		/*]*/
		l7vs_module_register(&l7vs_protomod_list, pmod);
	}

	return pmod;
}

static void 
l7vs_protomod_unload(struct l7vs_protomod *pmod)
{
	void *h;

	if (pmod == NULL) {
		VANESSA_LOGGER_ERR("pmod is null");
		return;
	}

	h = pmod->handle;
	if (pmod->fini != NULL) {
		pmod->fini();
	}
	l7vs_module_unload(h);
}

struct l7vs_protomod *
l7vs_protomod_lookup(char *modname)
{
	if (modname == NULL) {
		VANESSA_LOGGER_ERR("modname is null");
		return NULL;
	}

	return (struct l7vs_protomod *)
		l7vs_module_lookup(l7vs_protomod_list, modname,
				   (GCompareFunc)l7vs_protomod_cmp);
}

static gint
l7vs_protomod_cmp(struct l7vs_protomod *pmod, char *name)
{
	if (pmod == NULL || name == NULL) {
		VANESSA_LOGGER_ERR("pmod or name is null");
		return -1;
	}
	return strcmp(pmod->modname, name);
}

static int
protomod_initialize(struct l7vs_service *srv, struct l7vs_conn *conn, char *buf, size_t len, struct l7vs_dest **dest)
{
	if (dest == NULL) {
		VANESSA_LOGGER_ERR("dest list is null");
		return -1;
	}
	 *dest = NULL;
	 return 0;
}

static int
protomod_finalize(struct l7vs_service *srv, struct l7vs_conn *conn, char *buf, size_t len, struct l7vs_dest **dest, int resched)
{
	int ret = 0;
	GList*	active_list = NULL;
	GList*   d = NULL;

	//create active_list
	for( d = g_list_first( srv->dest_list ); d != NULL; d = g_list_next(d) ){
		struct l7vs_dest* target = d->data;
		if( target->weight > 0 ) active_list = g_list_append( active_list, target );
	}

	if( !*dest ){
		*dest = srv->scheduler->schedule( srv, conn );
		if( !*dest ){
			VANESSA_LOGGER_ERR( "realserver nonexistence" );
			ret = -1;
			goto PROTOMOD_FINALIZE_OUT;
		}
	}
	else{
		//check realserver
		for( d = g_list_first( active_list ); d != NULL; d = g_list_next(d) ){
			struct l7vs_dest* active_d = d->data;
			if( memcmp( &(active_d->addr.sin_addr.s_addr ),
				    &((*dest)->addr.sin_addr.s_addr ),
				    sizeof( struct in_addr ) )
			    && active_d->addr.sin_port == (*dest)->addr.sin_port ){
			*dest = active_d;
			ret = 0;
			goto PROTOMOD_FINALIZE_OUT;
			}
		}
		// check schedule
		if( !resched ){
			VANESSA_LOGGER_ERR( "realserver nonexistence" );
			ret = -1;
			goto PROTOMOD_FINALIZE_OUT;
		}
		else{
			*dest = srv->scheduler->schedule( srv, conn );
			if( !*dest ){
				VANESSA_LOGGER_ERR( "realserver nonexistence" );
				ret = -1;
				goto PROTOMOD_FINALIZE_OUT;
			}
			ret = 0;
		}
	}
PROTOMOD_FINALIZE_OUT:
	g_list_free( active_list );
	return ret;
}
