/*
 * @file  sslproxymain.cpp
 * @brief the main module of SSLproxy
 * @brief SSLproxy module is terminal as for the SSL communication,
 * @brief and it operates as reverse Proxy.
 *
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************
 *
 * Distributed under the Boost Software Licence, Version 1.0
 * http://www.boost.org/LICENSE_1_0.txt
 *
 **********************************************************************/

#include "sslproxy.h"
#include "sslproxyserver.h"

using namespace l7vs;

//! Prototype of functions.
static int convVerifyOption(std::string opt_string);
static long int convSslOption(std::string opt_string);
static int getParameters(std::string config_filename);
static int splitEndpoint(std::string endpoint_str, std::string& host, std::string& port);
static void sig_exit_handler(int sig);
static int set_sighandler(int sig, void (*handler)(int));
static int set_sighandlers(void);
static void usage(void);

//! Target_id.
std::string target_id;

//! SSLproxy parameters.
std::string recv_endpoint;
std::string target_endpoint;
int num_thread;
int timeout_sec;
std::string ca_dir;
std::string ca_file;
std::string cert_chain_dir;
std::string cert_chain_file;
std::string private_key_dir;
std::string private_key_file;
boost::asio::ssl::context::file_format private_key_filetype;
std::string private_key_passwd_from;
std::string private_key_passwd_dir;
std::string private_key_passwd_file;
int verify_options;
int verify_cert_depth;
long int ssl_options;
bool tmp_dh_use;
std::string tmp_dh_dir;
std::string tmp_dh_file;
std::string cipher_list;
std::string conn_log_flag;

/*!
 * Convert verify option string to intger(#define).
 *
 * @param[in]	opt_string	option string
 * @retval	ret	option value
 * @retval	-1	no match
 */
static int convVerifyOption(std::string opt_string)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 1,
			"in_function : static int convVerifyOption(std::string opt_string) : "
			"opt_string = %s",
			opt_string.c_str());
	}
	/*------ DEBUG LOG END ------*/

	int ret = -1;
	/*!
	 * /usr/include/openssl/ssl.h
	 * #define SSL_VERIFY_NONE                 0x00
	 * #define SSL_VERIFY_PEER                 0x01
	 * #define SSL_VERIFY_FAIL_IF_NO_PEER_CERT 0x02
	 * #define SSL_VERIFY_CLIENT_ONCE          0x04
	 */
	if (opt_string == "SSL_VERIFY_NONE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 2,
				"function : static int convVerifyOption(std::string opt_string) : "
				"SSL_VERIFY_NONE");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_VERIFY_NONE;
	}
	if (opt_string == "SSL_VERIFY_PEER") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 3,
				"function : static int convVerifyOption(std::string opt_string) : "
				"SSL_VERIFY_PEER");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_VERIFY_PEER;
	}
	if (opt_string == "SSL_VERIFY_FAIL_IF_NO_PEER_CERT") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 4,
				"function : static int convVerifyOption(std::string opt_string) : "
				"SSL_VERIFY_FAIL_IF_NO_PEER_CERT");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
	}
	if (opt_string == "SSL_VERIFY_CLIENT_ONCE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 5,
				"function : static int convVerifyOption(std::string opt_string) : "
				"SSL_VERIFY_CLIENT_ONCE");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_VERIFY_CLIENT_ONCE;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 6,
			"out_function : static int convVerifyOption(std::string opt_string) : "
			"return_value = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/
	// if ret == -1 then No match.
	return ret;
}

/*!
 * Convert SSL option string to intger(#define).
 *
 * @param[in]	opt_string	option string
 * @retval	ret	option value
 * @retval	-1	no match
 */
static long int convSslOption(std::string opt_string)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 7,
			"in_function : static long int convSslOption(std::string opt_string) : "
			"opt_string = %s",
			opt_string.c_str());
	}
	/*------ DEBUG LOG END ------*/

	long int ret = -1;
	/*!
	 * /usr/include/openssl/ssl.h
	 * #define SSL_OP_MICROSOFT_SESS_ID_BUG                    0x00000001L
	 * #define SSL_OP_NETSCAPE_CHALLENGE_BUG                   0x00000002L
	 * #define SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG         0x00000008L
	 * #define SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG              0x00000010L
	 * #define SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER               0x00000020L
	 * #define SSL_OP_MSIE_SSLV2_RSA_PADDING                   0x00000040L
	 * #define SSL_OP_SSLEAY_080_CLIENT_DH_BUG                 0x00000080L
	 * #define SSL_OP_TLS_D5_BUG                               0x00000100L
	 * #define SSL_OP_TLS_BLOCK_PADDING_BUG                    0x00000200L
	 * #define SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS              0x00000800L
	 * #define SSL_OP_ALL                                      0x00000FF7L
	 * #define SSL_OP_NO_QUERY_MTU                             0x00001000L
	 * #define SSL_OP_COOKIE_EXCHANGE                          0x00002000L
	 * #define SSL_OP_NO_SESSION_RESUMPTION_ON_RENEGOTIATION   0x00010000L
	 * #define SSL_OP_SINGLE_ECDH_USE                          0x00080000L
	 * #define SSL_OP_SINGLE_DH_USE                            0x00100000L
	 * #define SSL_OP_EPHEMERAL_RSA                            0x00200000L
	 * #define SSL_OP_CIPHER_SERVER_PREFERENCE                 0x00400000L
	 * #define SSL_OP_TLS_ROLLBACK_BUG                         0x00800000L
	 * #define SSL_OP_NO_SSLv2                                 0x01000000L
	 * #define SSL_OP_NO_SSLv3                                 0x02000000L
	 * #define SSL_OP_NO_TLSv1                                 0x04000000L
	 * #define SSL_OP_PKCS1_CHECK_1                            0x08000000L
	 * #define SSL_OP_PKCS1_CHECK_2                            0x10000000L
	 * #define SSL_OP_NETSCAPE_CA_DN_BUG                       0x20000000L
	 * #define SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG          0x40000000L
	 */
	if (opt_string == "SSL_OP_MICROSOFT_SESS_ID_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 8,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_MICROSOFT_SESS_ID_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_MICROSOFT_SESS_ID_BUG;
	}
	if (opt_string == "SSL_OP_NETSCAPE_CHALLENGE_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 9,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NETSCAPE_CHALLENGE_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NETSCAPE_CHALLENGE_BUG;
	}
	if (opt_string == "SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 10,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG;
	}
	if (opt_string == "SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 11,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG;
	}
	if (opt_string == "SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 12,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER;
	}
	if (opt_string == "SSL_OP_MSIE_SSLV2_RSA_PADDING") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 13,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_MSIE_SSLV2_RSA_PADDING");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_MSIE_SSLV2_RSA_PADDING;
	}
	if (opt_string == "SSL_OP_SSLEAY_080_CLIENT_DH_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 14,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_SSLEAY_080_CLIENT_DH_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_SSLEAY_080_CLIENT_DH_BUG;
	}
	if (opt_string == "SSL_OP_TLS_D5_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 15,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_TLS_D5_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_TLS_D5_BUG;
	}
	if (opt_string == "SSL_OP_TLS_BLOCK_PADDING_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 16,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_TLS_BLOCK_PADDING_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_TLS_BLOCK_PADDING_BUG;
	}
	if (opt_string == "SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 17,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
	}
	if (opt_string == "SSL_OP_ALL") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 18,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_ALL");
		}
		/*------ DEBUG LOG END ------*/
		// boost::asio::ssl::context::default_workarounds
		ret = SSL_OP_ALL;
	}
	if (opt_string == "SSL_OP_NO_QUERY_MTU") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 19,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NO_QUERY_MTU");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NO_QUERY_MTU;
	}
	if (opt_string == "SSL_OP_COOKIE_EXCHANGE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 20,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_COOKIE_EXCHANGE");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_COOKIE_EXCHANGE;
	}
	if (opt_string == "SSL_OP_NO_SESSION_RESUMPTION_ON_RENEGOTIATION") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 21,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NO_SESSION_RESUMPTION_ON_RENEGOTIATION");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NO_SESSION_RESUMPTION_ON_RENEGOTIATION;
	}
	if (opt_string == "SSL_OP_SINGLE_ECDH_USE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 22,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_SINGLE_ECDH_USE");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_SINGLE_ECDH_USE;
	}
	if (opt_string == "SSL_OP_SINGLE_DH_USE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 23,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_SINGLE_DH_USE");
		}
		/*------ DEBUG LOG END ------*/
		// boost::asio::ssl::context::single_dh_use
		ret = SSL_OP_SINGLE_DH_USE;
	}
	if (opt_string == "SSL_OP_EPHEMERAL_RSA") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 24,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_EPHEMERAL_RSA");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_EPHEMERAL_RSA;
	}
	if (opt_string == "SSL_OP_CIPHER_SERVER_PREFERENCE") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 25,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_CIPHER_SERVER_PREFERENCE");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_CIPHER_SERVER_PREFERENCE;
	}
	if (opt_string == "SSL_OP_TLS_ROLLBACK_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 26,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_TLS_ROLLBACK_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_TLS_ROLLBACK_BUG;
	}
	if (opt_string == "SSL_OP_NO_SSLv2") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 27,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NO_SSLv2");
		}
		/*------ DEBUG LOG END ------*/
		// boost::asio::ssl::context::no_sslv2
		ret = SSL_OP_NO_SSLv2;
	}
	if (opt_string == "SSL_OP_NO_SSLv3") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 28,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NO_SSLv3");
		}
		/*------ DEBUG LOG END ------*/
		// boost::asio::ssl::context::no_sslv3
		ret = SSL_OP_NO_SSLv3;
	}
	if (opt_string == "SSL_OP_NO_TLSv1") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 29,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NO_TLSv1");
		}
		/*------ DEBUG LOG END ------*/
		// boost::asio::ssl::context::no_tlsv1
		ret = SSL_OP_NO_TLSv1;
	}
	if (opt_string == "SSL_OP_PKCS1_CHECK_1") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 30,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_PKCS1_CHECK_1");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_PKCS1_CHECK_1;
	}
	if (opt_string == "SSL_OP_PKCS1_CHECK_2") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 31,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_PKCS1_CHECK_2");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_PKCS1_CHECK_2;
	}
	if (opt_string == "SSL_OP_NETSCAPE_CA_DN_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 32,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NETSCAPE_CA_DN_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NETSCAPE_CA_DN_BUG;
	}
	if (opt_string == "SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG") {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 33,
				"function : static int convSslOption(std::string opt_string) : "
				"SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG");
		}
		/*------ DEBUG LOG END ------*/
		ret = SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 34,
			"out_function : static long int convSslOption(std::string opt_string) : "
			"return_value = %ld",
			ret);
	}
	/*------ DEBUG LOG END ------*/
	// if ret == -1 then No match.
	return ret;
}

/*!
 * Get SSLproxy parameter from parameter file.
 *
 * @param[in]	config_filename	config filename string
 * @retval	0	success
 * @retval	-1	failed
 */
static int getParameters(std::string config_filename)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 35,
			"in_function : static int getParameters(std::string config_filename) : "
			"config_filename = %s",
			config_filename.c_str());
	}
	/*------ DEBUG LOG END ------*/

	int ret = 0;
	
	try {
		// Read configuration file.
		if (parameter_reread_file(PARAM_COMP_ALL, config_filename) == -1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 1, 
					     "Config file read error.");
			throw -1;
		}

		// Get parameter "recv_endpoint".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "recv_endpoint")) {
			recv_endpoint = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										"recv_endpoint");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 36,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get recv_endpoint OK.");
			}
			/*------ DEBUG LOG END ------*/
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 2, 
					     "Cannot get recv_endpoint parameter.");
			throw -1;
		}

		// Get parameter "target_endpoint".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "target_endpoint")) {
			target_endpoint = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										  "target_endpoint");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 37,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get target_endpoint OK.");
			}
			/*------ DEBUG LOG END ------*/
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 3, 
					     "Cannot get target_endpoint parameter.");
			throw -1;
		}

		// Get parameter "num_thread".
		if (Parameter::getInstance().isIntExist(PARAM_COMP_SSLPROXY, "num_thread")) {
			num_thread = Parameter::getInstance().getIntValue(PARAM_COMP_SSLPROXY, 
									  "num_thread");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 38,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get num_thread OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (num_thread <= 0 || num_thread > INT_MAX) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 4, 
						     "Invalid num_thread parameter value.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 1, 
					    "num_thread parameter not found. "
					    "Use default value.");
			num_thread = DEFAULT_NUM_THREAD;
		}

		// Get parameter "timeout_sec".
		if (Parameter::getInstance().isIntExist(PARAM_COMP_SSLPROXY, "timeout_sec")) {
			timeout_sec = Parameter::getInstance().getIntValue(PARAM_COMP_SSLPROXY, 
									   "timeout_sec");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 39,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get timeout_sec OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (timeout_sec <= 0 || timeout_sec > INT_MAX) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 5, 
						     "Invalid timeout_sec parameter value.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 2, 
					    "timeout_sec parameter not found. "
					    "Use default value.");
			timeout_sec = DEFAULT_TIMEOUT_SEC;
		}

		// Get parameter "ca_dir".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "ca_dir")) {
			ca_dir = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
									 "ca_dir");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 40,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get ca_dir OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (ca_dir == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 3, 
						    "ca_dir parameter is nothing. "
						    "Use default value.");
				ca_dir = DEFAULT_CA_DIR;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 4, 
					    "ca_dir parameter not found. "
					    "Use default value.");
			ca_dir = DEFAULT_CA_DIR;
		}

		// Get parameter "ca_file".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "ca_file")) {
			ca_file = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
									  "ca_file");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 41,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get ca_file OK.");
			}
			/*------ DEBUG LOG END ------*/
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 6, 
					     "Cannot get ca_file parameter.");
			throw -1;
		}

		// Get parameter "cert_chain_dir".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "cert_chain_dir")) {
			cert_chain_dir = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										 "cert_chain_dir");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 42,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get cert_chain_dir OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (cert_chain_dir == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 5, 
						    "cert_chain_dir parameter is nothing. "
						    "Use default value.");
				cert_chain_dir = DEFAULT_CERT_CHAIN_DIR;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 6, 
					    "cert_chain_dir parameter not found. "
					    "Use default value.");
			cert_chain_dir = DEFAULT_CERT_CHAIN_DIR;
		}

		// Get parameter "cert_chain_file".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "cert_chain_file")) {
			cert_chain_file = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										  "cert_chain_file");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 43,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get cert_chain_file OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (cert_chain_file == "") {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 7, 
						     "cert_chain_file parameter is nothing.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 8, 
					     "Cannot get cert_chain_file parameter.");
			throw -1;
		}

		// Get parameter "private_key_dir".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_dir")) {
			private_key_dir = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										  "private_key_dir");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 44,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get private_key_dir OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (private_key_dir == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 7, 
						    "private_key_dir parameter is nothing. "
						    "Use default value.");
				private_key_dir = DEFAULT_PRIVATE_KEY_DIR;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 8, 
					    "private_key_dir parameter not found. Use default value.");
			private_key_dir = DEFAULT_PRIVATE_KEY_DIR;
		}

		// Get parameter "private_key_file".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_file")) {
			private_key_file = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										   "private_key_file");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 45,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get private_key_file OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (private_key_file == "") {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 9, 
						     "private_key_file parameter is nothing.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 10, 
					     "Cannot get private_key_file parameter.");
			throw -1;
		}

		// Get parameter "private_key_filetype".
		// Convert string to enum.
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_filetype")) {
			std::string private_key_filetype_string = 
				Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
									"private_key_filetype");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 46,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get private_key_filetype_string OK.");
			}
			/*------ DEBUG LOG END ------*/
			/*!
			 * /usr/include/openssl/ssl.h
			 * #define SSL_FILETYPE_PEM        X509_FILETYPE_PEM  ->1
			 * #define SSL_FILETYPE_ASN1       X509_FILETYPE_ASN1 ->2
			 */
			if (private_key_filetype_string == "SSL_FILETYPE_PEM") {
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 47,
						"function : static int getParameters("
						"std::string config_filename) : "
						"set SSL_FILETYPE_PEM.");
				}
				/*------ DEBUG LOG END ------*/
				private_key_filetype = boost::asio::ssl::context::pem;
			} else if (private_key_filetype_string == "SSL_FILETYPE_ASN1") {
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 48,
						"function : static int getParameters("
						"std::string config_filename) : "
						"set SSL_FILETYPE_ASN1.");
				}
				/*------ DEBUG LOG END ------*/
				private_key_filetype = boost::asio::ssl::context::asn1;
			} else {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 11, 
						     "private_key_filetype convert error. [%s]", 
						     private_key_filetype_string.c_str());
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 9, 
					    "private_key_filetype parameter not found. "
					    "Use default value.");
			private_key_filetype = DEFAULT_PRIVATE_KEY_FILETYPE;
		}

		// Get parameter "private_key_passwd_from".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_passwd_from")) {
			private_key_passwd_from = 
				Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
									"private_key_passwd_from");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 49,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get private_key_passwd_from OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (private_key_passwd_from == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 10, 
						    "private_key_passwd_from parameter is nothing. "
						    "Use default value.");
				private_key_passwd_from = DEFAULT_PRIVATE_KEY_PASSWD_FROM;
			} else if (private_key_passwd_from != "console" && 
				   private_key_passwd_from != "file") {
					LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 12, 
							     "Invalid private_key_passwd_from parameter value.");
					throw -1;
			}
		} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 11, 
						    "private_key_passwd_from parameter not found. "
						    "Use default value.");
				private_key_passwd_from = DEFAULT_PRIVATE_KEY_PASSWD_FROM;
		}

		if (private_key_passwd_from == "file") {
			// Get parameter "private_key_passwd_dir".
			if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_passwd_dir")) {
				private_key_passwd_dir = 
					Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										"private_key_passwd_dir");
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 50,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get private_key_passwd_dir OK.");
				}
				/*------ DEBUG LOG END ------*/
				if (private_key_passwd_dir == "") {
					LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 12, 
							    "private_key_passwd_dir parameter is nothing. "
							    "Use default value.");
					private_key_passwd_dir = DEFAULT_PRIVATE_KEY_DIR;
				}
			} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 13, 
						    "private_key_passwd_dir parameter not found. "
						    "Use default value.");
				private_key_passwd_dir = DEFAULT_PRIVATE_KEY_PASSWD_DIR;
			}

			// Get parameter "private_key_passwd_file".
			if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "private_key_passwd_file")) {
				private_key_passwd_file = 
					Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										"private_key_passwd_file");
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 51,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get private_key_passwd_file OK.");
				}
				/*------ DEBUG LOG END ------*/
				if (private_key_passwd_file == "") {
					LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 13, 
							     "private_key_passwd_file parameter is nothing.");
					throw -1;
				}
			} else {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 14, 
						     "Cannot get private_key_passwd_file parameter.");
				throw -1;
			}
		}

		// Get parameter "verify_options".
		// Get map data and Convert string to integer and Make bit data.
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "verify_options")) {
			std::multimap<std::string, std::string> vopMap;
			Parameter::getInstance().getStringMapValue(PARAM_COMP_SSLPROXY, 
								   "verify_options", 
								   vopMap);
			if (vopMap.size() != 0) {
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 52,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get verify_options map OK.");
				}
				/*------ DEBUG LOG END ------*/
				for(std::multimap<std::string, std::string>::iterator stritr = vopMap.begin(); 
				    stritr != vopMap.end(); 
				    ++stritr ) {
					/*-------- DEBUG LOG --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 53,
							"function : "
							"static int getParameters(std::string config_filename) : "
							"verify option map key = %s, value = %s",
							(stritr->first).c_str(),
							(stritr->second).c_str());
					}
					/*------ DEBUG LOG END ------*/
					int retvop = convVerifyOption(stritr->second);
					if (retvop == -1) {
						LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 15, 
								     "verify_options convert error. [%s]", 
								     stritr->second.c_str());
						throw -1;
					}
					verify_options = (verify_options | retvop);
				}
			} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 14, 
						    "verify_options parameter is nothing. "
						    "Use default value.");
				verify_options = DEFAULT_VERIFY_OPTIONS;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 15, 
					    "verify_options parameter not found. "
					    "Use default value.");
			verify_options = DEFAULT_VERIFY_OPTIONS;
		}

		// Get parameter "verify_cert_depth".
		if (Parameter::getInstance().isIntExist(PARAM_COMP_SSLPROXY, "verify_cert_depth")) {
			verify_cert_depth = Parameter::getInstance().getIntValue(PARAM_COMP_SSLPROXY, 
										 "verify_cert_depth");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 54,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get verify_cert_depth OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (verify_cert_depth < 0 || verify_cert_depth > INT_MAX) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 16, 
						     "Invalid verify_cert_depth parameter value.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 16, 
					    "verify_cert_depth parameter not found. "
					    "Use default value.");
			verify_cert_depth = DEFAULT_VERIFY_CERT_DEPTH;
		}

		// Get parameter "ssl_options".
		// Get map data and Convert string to integer and Make bit data.
		// and Check dh parameter file use or not.
		tmp_dh_use = false;
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "ssl_options")) {
			std::multimap<std::string, std::string> sopMap;
			Parameter::getInstance().getStringMapValue(PARAM_COMP_SSLPROXY, 
								   "ssl_options", 
								   sopMap);
			if (sopMap.size() != 0) {
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 55,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get ssl_options map OK.");
				}
				/*------ DEBUG LOG END ------*/
				for(std::multimap<std::string, std::string>::iterator stritr = sopMap.begin(); 
				    stritr != sopMap.end(); 
				    ++stritr ) {
					/*-------- DEBUG LOG --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 56,
							"function : "
							"static int getParameters(std::string config_filename) : "
							"SSL option map key = %s, value = %s",
							(stritr->first).c_str(),
							(stritr->second).c_str());
					}
					/*------ DEBUG LOG END ------*/
					long int retsop = convSslOption(stritr->second);
					if (retsop == -1) {
						LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 17, 
								     "ssl_options convert error. [%s]", 
								     stritr->second.c_str());
						throw -1;
					}
					if (retsop == SSL_OP_SINGLE_DH_USE) {
						/*-------- DEBUG LOG --------*/
						if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 57,
								"function : "
								"static int getParameters(std::string config_filename) : "
								"tmp_dh_use is true.");
						}
						/*------ DEBUG LOG END ------*/
						tmp_dh_use = true;
					}
					ssl_options = (ssl_options | retsop);
				}
			} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 17, 
						    "ssl_options parameter is nothing. "
						    "Use default value.");
				ssl_options = DEFAULT_SSL_OPTIONS;
				tmp_dh_use = true;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 18, 
					    "ssl_options parameter not found. "
					    "Use default value.");
			ssl_options = DEFAULT_SSL_OPTIONS;
			tmp_dh_use = true;
		}

		if (tmp_dh_use == true) {
			// Get parameter "tmp_dh_dir".
			if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "tmp_dh_dir")) {
				tmp_dh_dir = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										     "tmp_dh_dir");
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 58,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get tmp_dh_dir OK.");
				}
				/*------ DEBUG LOG END ------*/
				if (tmp_dh_dir == "") {
					LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 19, 
							    "tmp_dh_dir parameter is nothing. "
							    "Use default value.");
					tmp_dh_dir = DEFAULT_TMP_DH_DIR;
				}
			} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 20, 
						    "tmp_dh_dir parameter not found. "
						    "Use default value.");
				tmp_dh_dir = DEFAULT_TMP_DH_DIR;
			}

			// Get parameter "tmp_dh_file".
			if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "tmp_dh_file")) {
				tmp_dh_file = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
										      "tmp_dh_file");
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 59,
						"function : static int getParameters("
						"std::string config_filename) : "
						"get tmp_dh_file OK.");
				}
				/*------ DEBUG LOG END ------*/
				if (tmp_dh_file == "") {
					LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 18, 
							     "tmp_dh_file parameter is nothing.");
					throw -1;
				}
			} else {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 19, 
						     "Cannot get tmp_dh_file parameter.");
				throw -1;
			}
		}

		// Get parameter "cipher_list".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_SSLPROXY, "cipher_list")) {
			cipher_list = Parameter::getInstance().getStringValue(PARAM_COMP_SSLPROXY, 
									      "cipher_list");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 60,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get cipher_list OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (cipher_list == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 21, 
						    "cipher_list parameter is nothing. "
						    "Use default value.");
				cipher_list = DEFAULT_CIPHER_LIST;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 22, 
					    "cipher_list parameter not found. "
					    "Use default value.");
			cipher_list = DEFAULT_CIPHER_LIST;
		}

		// Get parameter "conn_log_flag".
		if (Parameter::getInstance().isStringExist(PARAM_COMP_LOGGER, "conn_log_flag")) {
			conn_log_flag = Parameter::getInstance().getStringValue(PARAM_COMP_LOGGER, 
										"conn_log_flag");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 61,
					"function : static int getParameters("
					"std::string config_filename) : "
					"get conn_log_flag OK.");
			}
			/*------ DEBUG LOG END ------*/
			if (conn_log_flag == "") {
				LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 23, 
						    "conn_log_flag parameter is nothing. "
						    "Use default value.");
				conn_log_flag = DEFAULT_CONN_LOG_FLAG;
			} else if (conn_log_flag != "on" && conn_log_flag != "off") {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 20, 
						     "Invalid conn_log_flag parameter value.");
				throw -1;
			}
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_COMMON, 24, 
					    "conn_log_flag parameter not found. "
					    "Use default value.");
			conn_log_flag = DEFAULT_CONN_LOG_FLAG;
		}
	} catch (int e) {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 62,
				"function : "
				"static int getParameters(std::string config_filename) : "
				"Catch exception e = %d.",
				e);
		}
		/*------ DEBUG LOG END ------*/
		ret = -1;
	}
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		std::ostringstream oss;
		oss << 
		"  recv_endpoint = "		<< recv_endpoint		<<
		", target_endpoint = "		<< target_endpoint		<<
		", num_thread = "		<< num_thread			<<
		", timout_sec = "		<< timeout_sec			<<
		", ca_dir = "			<< ca_dir			<<
		", ca_file = "			<< ca_file			<<
		", cert_chain_dir = "		<< cert_chain_dir		<<
		", cert_chain_file = "		<< cert_chain_file		<<
		", private_key_dir = "		<< private_key_dir		<<
		", private_key_file = "		<< private_key_file		<<
		", private_key_filetype = "	<< private_key_filetype		<<
		", private_key_passwd_from = "	<< private_key_passwd_from	<<
		", private_key_passwd_dir = "	<< private_key_passwd_dir	<<
		", private_key_passwd_file = "	<< private_key_passwd_file	<<
		", verify_options = "		<< verify_options		<<
		", verify_cert_depth = "	<< verify_cert_depth		<<
		", ssl_options = "		<< ssl_options			<<
		", tmp_dh_dir = "		<< tmp_dh_dir			<<
		", tmp_dh_file = "		<< tmp_dh_file			<<
		", cipher_list = "		<< cipher_list			<<
		", conn_log_flag = "		<< conn_log_flag;
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 63,
			"out_function : "
			"static int getParameters(std::string config_filename) : "
			"return value = %d : "
			"parameter value %s",
			ret,
			oss.str().c_str());
	}
	/*------ DEBUG LOG END ------*/
	return ret;
}

/*!
 * Split endpoint data to host and port string.
 *
 * @param[in]	endpoint_str	endpoint string
 * @param[out]	host	host string
 * @param[out]	port	port string
 * @retval	0	success
 * @retval	-1	failed
 */
static int splitEndpoint(std::string endpoint_str, std::string& host, std::string& port)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 64,
			"in_function : "
			"static int splitEndpoint"
			"(std::string endpoint_str, std::string& host, std::string& port) : "
			"endpoint_str = %s",
			endpoint_str.c_str());
	}
	/*------ DEBUG LOG END ------*/

	int ret = 0;
	try {
		int mark = endpoint_str.find_first_of(":");
		if (mark == -1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 21, "Delimitation not found.");
			throw -1;
		}
		if (mark <= 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 22, "Host string not found.");
			throw -1;
		}
		host = endpoint_str.substr(0, mark);
		port = endpoint_str.substr(mark + 1);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 65,
				"function : "
				"static int splitEndpoint"
				"(std::string endpoint_str, std::string& host, std::string& port) : "
				"set host and port string");
		}
		/*------ DEBUG LOG END ------*/
		if (port.length() <= 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 23, "Port string not found.");
			throw -1;
		}
	} catch (int e) {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 66,
				"function : "
				"static int splitEndpoint"
				"(std::string endpoint_str, std::string& host, std::string& port) : "
				"Catch exception e = %d.",
				e);
		}
		/*------ DEBUG LOG END ------*/
		ret = -1;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 67,
			"out_function : "
			"static int splitEndpoint"
			"(std::string endpoint_str, std::string& host, std::string& port) : "
			"return value ret = %d : "
			"host = %s, "
			"port = %s",
			ret,
			host.c_str(),
			port.c_str());
	}
	/*------ DEBUG LOG END ------*/
	return ret;
}

/*!
 * exit signal handler
 *
 * @param[in]	sig	signal
 * @return	void
 */
static void sig_exit_handler(int sig)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 68,
			"in_function : static void sig_exit_handler(int sig) "
			"sig = %d",
			sig);
	}
	/*------ DEBUG LOG END ------*/
	int retexit;

	if (sig == SIGTERM) {
		LOGGER_PUT_LOG_INFO(LOG_CAT_SSLPROXY_COMMON, 1, "SIGTERM received.");
		retexit = EXIT_SUCCESS;
	} else {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 24, "Signal received. signal=%d.", sig);
		retexit = EXIT_FAILURE;
	}

	fflush(stdout);
	fsync(1);

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 69,
			"out_function : static void sig_exit_handler(int sig) : "
			"exit_value = %d", 
			retexit);
	}
	/*------ DEBUG LOG END ------*/
	_exit(retexit);
}

/*!
 * setup signal handler
 *
 * @param[in]	sig	signal
 * @param[in]	handler	signal handler
 * @retval	 0 succeed
 * @retval	-1 failed
 */
static int set_sighandler(int sig, void (*handler)(int))
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 70,
			"in_function : static int set_sighandler(int sig, void (*handler)(int)) "
			"sig = %d: "
			"handler = %p",
			sig, handler);
	}
	/*------ DEBUG LOG END ------*/

	struct sigaction act;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 71,
			"sigaction : sig = %d : act = NULL",
			sig);
	}
	/*------ DEBUG LOG END ------*/
	ret = sigaction(sig, NULL, &act);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 25, "sigaction on signal %d failed", sig);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 72,
				"out_function : static int set_sighandler(int sig, void (*handler)(int)) "
				"return_value : %d",
				ret);
		}
		/*------ DEBUG LOG END ------*/
		return ret;
	}
	act.sa_flags &= ~SA_RESETHAND;
	act.sa_handler = handler;

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 73,
			"sigaction : sig=%d: act.sa_flags=%d, act.sa_handler=%p",
			sig, act.sa_flags, act.sa_handler);
	}
	/*------ DEBUG LOG END ------*/
	ret = sigaction(sig, &act, NULL);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 26, "sigaction on signal %d failed", sig);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 74,
				"out_function : static int set_sighandler(int sig, void (*handler)(int)) "
				"return_value : %d",
				ret);
		}
		/*------ DEBUG LOG END ------*/
		return ret;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 75,
			"out_function : static int set_sighandler(int sig, void (*handler)(int)) "
			"return_value : 0");
	}
	/*------ DEBUG LOG END ------*/
	return 0;
}

/*!
 * setup all signal handlers
 *
 * @param[in]	void
 * @retval	 0 succeed
 * @retval	-1 failed
 */
static int set_sighandlers(void)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 76,
			"in_function : static int set_sighandlers(void) ");
	}
	/*------ DEBUG LOG END ------*/
	int ret;

#define SET_SIGHANDLER(sig, handler)								\
	do {											\
		ret = set_sighandler((sig), (handler));						\
		if (ret < 0) {									\
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {	\
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 77,		\
					"out_function : static int set_sighandlers(void) "	\
					"return_value : %d",					\
					ret);							\
			}									\
			return ret;								\
		}										\
	} while (0)

	SET_SIGHANDLER(SIGHUP,  sig_exit_handler);
	SET_SIGHANDLER(SIGINT,  sig_exit_handler);
	SET_SIGHANDLER(SIGQUIT, sig_exit_handler);
	SET_SIGHANDLER(SIGTERM, sig_exit_handler);
	SET_SIGHANDLER(SIGUSR1, SIG_IGN);
	SET_SIGHANDLER(SIGUSR2, SIG_IGN);
	SET_SIGHANDLER(SIGALRM, SIG_IGN);
	SET_SIGHANDLER(SIGCHLD, SIG_IGN);

#undef SET_SIGHANDLER

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 78,
			"out_function : static int set_sighandlers(void) "
			"return_value : 0");
	}
	/*------ DEBUG LOG END ------*/
	return 0;
}

/*!
 * Show usage.
 *
 */
static void usage(void)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 79,
			"in_function : static void usage(void)");
	}
	/*------ DEBUG LOG END ------*/

	std::cerr << "Usage   : " << "sslproxy <target_id> <config_filename>" << std::endl;
	std::cerr << "Example : " << "sslproxy target_1 /etc/l7vs/sslproxy/sslproxy.target_1.cf" << std::endl;

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 80,
			"out_function : static void usage(void) "
			"return_value = void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * SSLproxy main function.
 * 
 * @param[in]   argc    number of argument
 * @param[in]   *argv[] array of argument string
 */
void sslproxy_main(int argc, char* argv[])
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		std::ostringstream oss;
		oss << "argc = " <<  argc;
		for (int i = 0; i < argc; i++) {
			oss << ", arg[" << i << "] = " << argv[i];
		}
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 81,
			"in_function : void sslproxy_main(int argc, char* argv[]) : "
			"%s", 
			oss.str().c_str());
	}
	/*------ DEBUG LOG END ------*/

	int result = EXIT_SUCCESS;
	int log_msg_id = 0;
	std::string function_str = "";

	try {
		// Argument check.
		if (argc != 3) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 27, "Argument argc is illegal.");
			std::cerr << "Argument argc is illegal." << std::endl;
			usage();
			throw EXIT_FAILURE;
		}

		// Check argument 1. (target_id)
		if (strlen(argv[1]) > MAX_TARGET_ID_SIZE) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 28, "Target_id is too long.");
			std::cerr << "Target_id is too long." << std::endl;
			throw EXIT_FAILURE;
		}
		target_id = argv[1];

		// Check argument 2. (config file)
		if (argv[2][0] != '/') {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 29, 
				"Need specify config file in full path. %s", target_id.c_str());
			std::cerr << "Need specify config file in full path. " << target_id << std::endl;
			throw EXIT_FAILURE;
		}
		FILE  *fp;
		if ((fp = fopen(argv[2], "r")) == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 30, 
				"Config file cannot open. %s", target_id.c_str());
			std::cerr << "Config file cannot open. " << target_id << std::endl;
			throw EXIT_FAILURE;
		}
		fclose(fp);
		std::string config_filename = argv[2];

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 82,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Argument check END. "
				"target_id = %s, "
				"config_filename = %s",
				target_id.c_str(), 
				config_filename.c_str());
		}
		/*------ DEBUG LOG END ------*/

		// Get config parameters.
		if (getParameters(config_filename) == -1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 31, 
				"Get parameter error. %s", target_id.c_str());
			std::cerr << "Get parameter error. " << target_id << std::endl;
			throw EXIT_FAILURE;
		}

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 83,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Get parameter END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set signal handlers.
		if (set_sighandlers() < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 32, 
				"Set signal handler error. %s", target_id.c_str());
			std::cerr << "Set signal handler error. " << target_id << std::endl;
			throw EXIT_FAILURE;
		}

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 84,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Set signal END.");
		}
		/*------ DEBUG LOG END ------*/

		// Check target endpoint id duplication.
		// Read all sslproxy process and target id is extracted.
		if ((fp = popen(TARGET_ID_CHECK_STRING, "r")) == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 33, 
				"popen error. %s", target_id.c_str());
			std::cerr << "popen error. " << target_id << std::endl;
			throw EXIT_FAILURE;
		}
		char key[MAX_TARGET_ID_SIZE], buf[MAX_TARGET_ID_SIZE];
		int match_cnt = 0;
		while (fgets(buf, MAX_TARGET_ID_SIZE, fp) != NULL) {
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				std::string buf_str = buf;
				buf_str.erase(buf_str.size() - 1, 1);
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 85,
					"function : void sslproxy_main(int argc, char* argv[]) : "
					"Target id of executing process = %s.",
					buf_str.c_str());
			}
			/*------ DEBUG LOG END ------*/
			sprintf(key, "%s\n", target_id.c_str());
			if (strcmp(key, buf) == 0) {
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 86,
						"function : void sslproxy_main(int argc, char* argv[]) : "
						"Same target id found.");
				}
				/*------ DEBUG LOG END ------*/
				// Count same target id process.
				match_cnt++;
			}
			// Duplication error when the one the same besides oneself exists.
			if (match_cnt > 1) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 34, 
					"sslproxy process is already exist. %s", target_id.c_str());
				std::cerr << "sslproxy process is already exist. " << target_id << std::endl;
				pclose(fp);
				throw EXIT_FAILURE;
			}
		}
		pclose(fp);

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 87,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Check target id END.");
		}
		/*------ DEBUG LOG END ------*/

		// SSLproxy server initialize and setting.
		// IO event dispatcher class.
		boost::asio::io_service ioservice;

		// resolver class.
		log_msg_id = 39;
		function_str = "resolver()";
		boost::asio::ip::tcp::resolver resolver(ioservice);

		std::string hoststr;
		std::string portstr;

		// Target endpoint -> hoststr:portstr
		if (splitEndpoint(target_endpoint, hoststr, portstr) == -1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 35, 
				"Invalid target_endpoint. [%s]", target_id.c_str());
			std::cerr << "Invalid target_endpoint. [" << target_id << "]" << std::endl;
			throw EXIT_FAILURE;
		}

		// query class for target.
		log_msg_id = 40;
		function_str = "target_query()";
		boost::asio::ip::tcp::resolver::query target_query(boost::asio::ip::tcp::v4(), hoststr, portstr);

		// query result for target. (iterator)
		log_msg_id = 41;
		function_str = "Target resolver.resolve()";
		boost::asio::ip::tcp::resolver::iterator target_itr = resolver.resolve(target_query);

		// query result for target. (endpoint_type)
		boost::asio::ip::tcp::resolver::endpoint_type target_entry = *target_itr;
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 88,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Check target_endpoint END. "
				"%s:%d", 
				target_entry.address().to_string().c_str(),
				target_entry.port());
		}
		/*------ DEBUG LOG END ------*/

		// Recv endpoint -> hoststr:portstr
		if (splitEndpoint(recv_endpoint, hoststr, portstr) == -1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 36, 
				"Invalid recv_endpoint. [%s]", target_id.c_str());
			std::cerr << "Invalid recv_endpoint. [" << target_id << "]" << std::endl;
			throw EXIT_FAILURE;
		}

		// query class for recv.
		log_msg_id = 42;
		function_str = "recv_query()";
		boost::asio::ip::tcp::resolver::query recv_query(boost::asio::ip::tcp::v4(), hoststr, portstr);

		// query result for recv. (iterator)
		log_msg_id = 43;
		function_str = "Recv resolver.resolve()";
		boost::asio::ip::tcp::resolver::iterator recv_itr = resolver.resolve(recv_query);

		// query result for recv. (endpoint_type)
		boost::asio::ip::tcp::resolver::endpoint_type recv_entry = *recv_itr;
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 89,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Check recv_endpoint END. "
				"%s:%d", 
				recv_entry.address().to_string().c_str(),
				recv_entry.port());
		}
		/*------ DEBUG LOG END ------*/

		// Create sslproxy_server and create SSLcontext.
		log_msg_id = 44;
		function_str = "Create sslproxy_server";
		sslproxy_server server(ioservice, target_itr, recv_entry);

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 90,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Create sslproxy_server END.");
		}
		/*------ DEBUG LOG END ------*/

		// Daemon start.
		if (daemon(1, 1) < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 37, 
				"Start daemon() error. [%s]", 
				target_id.c_str());
			std::cerr << "Start daemon() error. [" << target_id << "]" << std::endl;
			throw EXIT_FAILURE;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 91,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Start daemon END.");
		}
		/*------ DEBUG LOG END ------*/

		// Create threads for IO event dispatcher.
		// Threadgroup class. (thread pool)
		boost::thread_group threadgroup;

		// Create num_thread threads.
		log_msg_id = 45;
		function_str = "create_thread()";
		for (int i = 0; i < num_thread; ++i) {
			threadgroup.create_thread(boost::bind(&boost::asio::io_service::run, &ioservice));
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 92,
					"function : void sslproxy_main(int argc, char* argv[]) : "
					"Thread %d create.", 
					i);
			}
			/*------ DEBUG LOG END ------*/
		}

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 93,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Create threads END.");
		}
		/*------ DEBUG LOG END ------*/

		LOGGER_PUT_LOG_INFO(LOG_CAT_SSLPROXY_COMMON, 2,
				"SSLproxy process start success. %s", 
				target_id.c_str());

		// Wait for threads finish.
		log_msg_id = 46;
		function_str = "join_all()";
		threadgroup.join_all();

		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 94,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"join_all END.");
		}
		/*------ DEBUG LOG END ------*/

	} catch (std::exception& e) {
		std::cerr << function_str << " error : " << e.what() << "." << std::endl;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, log_msg_id, 
			"%s error : %s.", function_str.c_str(), e.what());
		result = EXIT_FAILURE;
	} catch (int e) {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 95,
				"function : void sslproxy_main(int argc, char* argv[]) : "
				"Catch int exception. %d", 
				e);
		}
		/*------ DEBUG LOG END ------*/
		result = e;
	} catch (...) {
		std::cout << "Unknown exception." << std::endl;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_COMMON, 38, "Unknown exception.");
		result = EXIT_FAILURE;
	}

	fflush(stdout);
	fsync(1);

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_COMMON)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_COMMON, 96,
			"out_function : void sslproxy_main(int argc, char* argv[]) : "
			"return value = %d.", 
			result);
	}
	/*------ DEBUG LOG END ------*/
	_exit(result);
}
