/*
 * @file  parameter_impl.cpp
 * @brief parameter module implementation class.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <vector>
#include <fstream>
#include <stdexcept>
#include "parameter_impl.h"
#include "logger_wrapper.h"

#include <boost/lexical_cast.hpp>

#ifndef INIT_PARAMETER_FILE
#define INIT_PARAMETER_FILE "/etc/l7vs/sslproxy/sslproxy.logger_init.cf"
#endif //INIT_PARAMETER_FILE

#define	LINE_LENGTH	4096

LOG_CATEGORY_TAG parameter_cat = LOG_CAT_SSLPROXY_PARAMETER;

void	loadLoggerConf(){
	l7vs::Logger::getInstance().loadConf();
}

/*!
 * ParameterImpl class Constructor
 */
l7vs::ParameterImpl::ParameterImpl()
{
}

/*!
 * Initialize ParameterImpl class
 */
bool	l7vs::ParameterImpl::init()
{
	bool	retbool = true;
	//clrear map objects
	compTable.clear();
	stringMap.clear();
	intMap.clear();

	//add member to component table
	compTable[PARAM_COMP_ALL]	= component("all", NULL);
	compTable[PARAM_COMP_NOCAT]	= component("nocategory", NULL);
	compTable[PARAM_COMP_LOGGER]	= component("logger", loadLoggerConf);
	compTable[PARAM_COMP_SSLPROXY]	= component("sslproxy", NULL);

	//read all parameters
	retbool	= rereadFile( PARAM_COMP_ALL, INIT_PARAMETER_FILE);
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 4,
			"function : bool l7vs::ParameterImpl::init() : "
			"rereadFile() END. "
			"retbool = %d",
			retbool);
	}
	/*------ DEBUG LOG END for sslproxy ------*/

	return	retbool;
}

/*!
 * checks that an argument is a numeric.
 * @param[in]	const std::string
 * @return		is numeric = true / not numeric = false
 */
bool	l7vs::ParameterImpl::isNumeric(const std::string& val)
{
	try {
		boost::lexical_cast<int>(val);
	}
	catch (boost::bad_lexical_cast& ex) {
		return false;
	}
	return true;
}

/*!
 * delete blank before and behind a sentence
 * @param[in]	const std::string
 * @return		std::string
 */
std::string	l7vs::ParameterImpl::removebrank( const std::string& val )
{
	std::string	str = val;
	//remove brank(head of the sentence and end of the sentence)
	std::string::iterator its;
	its = str.begin();
	while( its != str.end() ){
		if( ( ' ' == *its ) || ( '\t' == *its ) ){
			str.erase(its);
		}else break;
	}
	if( "" == str )return str;
	//remove brank(head of the sentence and end of the sentence)
	its = str.end() - 1;
	while( its != str.begin() ){
		if( ( ' ' == *its ) || ( '\t' == *its ) ){
			str.erase(its);
			its--;
		}else break;
	}
	return str;
}

/*!
 * Read Parameter from file
 * @param[in]	const std::string
 * @return		success = true / fail = false
 */
bool	l7vs::ParameterImpl::readParameterFile( const std::string& val )
{
	bool	retval = true;
	char	readbuf[LINE_LENGTH];

	LOGGER_PUT_LOG_INFO( parameter_cat,1, "read parameter file : %s", val.c_str() );

	std::vector<std::string>	paramlines;
	try{
		paramlines.clear();
		preparse.clear();

		std::ifstream	ifs( val.c_str() );
		if( ifs.fail() ){
			return false;
		}
		while( !ifs.eof() ){
			memset( readbuf, 0, LINE_LENGTH );
			ifs.getline( readbuf, ( LINE_LENGTH - 1 ) );
			std::string	str = readbuf;
			str = removebrank( str );
			//remove coment line
			if( '#' == str[0] )continue;
			//remove brank
			if( !str.empty() )paramlines.push_back( str );
		}
		std::string	sectionname = compTable[PARAM_COMP_NOCAT].section;
		for( std::vector<std::string>::iterator it = paramlines.begin();
			it != paramlines.end(); it++ ){
			//pre-parse
			std::string str = *it;
			if( '[' == str[0] ){
				std::string	tmpstr = str.substr( str.rfind("]")+1 );
				tmpstr = removebrank( tmpstr.substr( 0, tmpstr.find( "#" ) ) );
				if( !tmpstr.empty() )continue;
				//section string validity check
				//invalid "#","[","]","\" character
				tmpstr = removebrank( str.substr( str.find( "[" )+1, str.rfind( "]", str.length() )-1 ) );
				if( std::string::npos != tmpstr.find( "[" ) )tmpstr = compTable[PARAM_COMP_NOCAT].section;
				if( std::string::npos != tmpstr.find( "]" ) )tmpstr = compTable[PARAM_COMP_NOCAT].section;
				if( std::string::npos != tmpstr.find( "\\" ) )tmpstr = compTable[PARAM_COMP_NOCAT].section;
				if( std::string::npos != tmpstr.find( "#" ) )tmpstr = compTable[PARAM_COMP_NOCAT].section;
				if( std::string::npos != tmpstr.find( "=" ) )tmpstr = compTable[PARAM_COMP_NOCAT].section;
				sectionname = tmpstr;
			}else{
				preparse.insert( std::pair<std::string,std::string>( sectionname,str ) );
			}
		}
	}
	catch( ... ){
		preparse.clear();
		retval	= false;
	}
	if( preparse.empty() )retval = false;
	return	retval;
}

/*!
 * read parameter and parse
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	filename	config file name
 * @return		success = true / fail = false
 */
bool	l7vs::ParameterImpl::rereadFile(PARAMETER_COMPONENT_TAG comp, const std::string& filename)
{
	bool	retbool = true;

	LOGGER_PUT_LOG_INFO( parameter_cat,2, "read parameter : COMPONENT = %s", compTable[comp].section.c_str() );

	if( !readParameterFile( filename ) )return false;

	stringMap.clear();
	intMap.clear();

	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 5,
			"function : bool l7vs::ParameterImpl::rereadFile("
			"PARAMETER_COMPONENT_TAG comp, "
			"const std::string& filename) : "
			"comp = %s, "
			"filename = %s",
			compTable[comp].section.c_str(),
			filename.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/

	std::multimap<std::string,std::string>::iterator	it;
	std::multimap<std::string,std::string>::iterator	ite;
	if( PARAM_COMP_ALL == comp ){
		it = preparse.begin();
	}else{
		it = preparse.find( compTable[comp].section );
	}
	ite = preparse.end();
		
	for( std::multimap<std::string,std::string>::iterator its = it; its !=ite; its++ ){
		//be made an invalid line if there is no = in a setting line.
		if( std::string::npos == its->second.find( "=" ) )continue;
		std::string	key = removebrank( its->second.substr( 0, its->second.find( "=" ) ) );
		//Key string validity check
		if( key.empty() )continue;
		//invalid "#","[","]","\" character
		if( std::string::npos != key.find( "[" ))continue;
		if( std::string::npos != key.find( "]" ))continue;
		if( std::string::npos != key.find( "\\" ))continue;
		if( std::string::npos != key.find( "#" ) )continue;

		key	= its->first + "." + key;

		std::string	value = removebrank( its->second.substr( its->second.find( "=" )+1, std::string::npos ) );
		//Value string validity check
		if( '"' == value[0] ) {
			if( std::string::npos == value.find( "\"", 1 ) )continue;
			std::string tmpstr = removebrank( value.substr( value.find( "\"", 1 )+1, value.find( "#", value.find( "\"", 1 )+1  ) ) );
			if( !tmpstr.empty() )continue;
			tmpstr = removebrank( value.substr( 1, value.find( "\"", 1 )-1 ) );
			stringMap.insert( std::pair<std::string, std::string>( key, tmpstr ) );
			/*-------- DEBUG LOG for sslproxy --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
				LOGGER_PUT_LOG_DEBUG(parameter_cat, 6,
					"function : bool l7vs::ParameterImpl::rereadFile() : "
					"stringMap.insert("
					"key = %s, "
					"tmpstr = %s)",
					key.c_str(),
					tmpstr.c_str());
			}
			/*------ DEBUG LOG END for sslproxy ------*/
		}else{
			value = value.substr( 0, value.find_first_of( "#", 0 ) );
			if( std::string::npos != value.find( "\\" ))continue;
			if( std::string::npos != value.find( "=" ))continue;
			if( value.empty() )continue;
			if( isNumeric( value ) ){
				try{
					intMap.insert( std::pair<std::string, int>( key, boost::lexical_cast<int>( value ) ) );
					/*-------- DEBUG LOG for sslproxy --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
						LOGGER_PUT_LOG_DEBUG(parameter_cat, 7,
							"function : bool l7vs::ParameterImpl::rereadFile() : "
							"intMap.insert("
							"key = %s, "
							"value = %s)",
							key.c_str(),
							value.c_str());
					}
					/*------ DEBUG LOG END for sslproxy ------*/
				}
				catch (boost::bad_lexical_cast& ex) {
					stringMap.insert( std::pair<std::string, std::string>( key, value ) );
					/*-------- DEBUG LOG for sslproxy --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
						LOGGER_PUT_LOG_DEBUG(parameter_cat, 8,
							"function : bool l7vs::ParameterImpl::rereadFile() : "
							"stringMap.insert("
							"key = %s, "
							"value = %s)",
							key.c_str(),
							value.c_str());
					}
					/*------ DEBUG LOG END for sslproxy ------*/
				}
			}else{
				stringMap.insert( std::pair<std::string, std::string>( key, value ) );
				/*-------- DEBUG LOG for sslproxy --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat)) {
					LOGGER_PUT_LOG_DEBUG(parameter_cat, 9,
						"function : bool l7vs::ParameterImpl::rereadFile() : "
						"stringMap.insert("
						"key = %s, "
						"value = %s)",
						key.c_str(),
						value.c_str());
				}
				/*------ DEBUG LOG END for sslproxy ------*/
			}
		}
	}
	//set parameter
	std::map<PARAMETER_COMPONENT_TAG, component>::iterator itr = compTable.find( comp );
	if( itr != compTable.end() ){
		try{
			if( NULL != itr->second.function )itr->second.function();
		}
		catch( const std::logic_error& err ){
			LOGGER_PUT_LOG_ERROR( parameter_cat,1, "set parameter function failure at category: %s", itr->second.section.c_str() );
			retbool	= false;
		}
	}
	if( PARAM_COMP_ALL == comp ){
		for( itr = compTable.begin(); itr != compTable.end(); ++itr ){
			try{
				if( NULL != itr->second.function )itr->second.function();
			}
			catch( const std::logic_error& err ){
				LOGGER_PUT_LOG_ERROR( parameter_cat,2, "set parameter function failure at category: %s", itr->second.section.c_str() );
				retbool	= false;
			}
		}
	}

	return	retbool;
}

/*!
 * Existence of a integer parameter is checked
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	key	key string
 * @return		exist = true / not exist = false
 */
bool	l7vs::ParameterImpl::isIntExist(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	bool	retval = false;
	std::string comp_key = compTable[comp].section + "." + key;
	if( intMap.end() != intMap.find( comp_key ) ){
		retval = true;
	}
	return	retval;
}

/*!
 * Existence of a string parameter is checked
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	key	key string
 * @return		exist = true / not exist = false
 */
bool	l7vs::ParameterImpl::isStringExist(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	bool	retval = false;
	std::string comp_key = compTable[comp].section + "." + key;
	if( stringMap.end() != stringMap.find( comp_key ) ){
		retval = true;
	}
	return	retval;
}

/*!
 * get a integer parameter
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	key	key string
 * @return		integer data
 */
int	l7vs::ParameterImpl::getIntValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 10,
			"in_function : int l7vs::ParameterImpl::getIntValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key) : "
			"comp = %s, "
			"key = %s",
			compTable[comp].section.c_str(),
			key.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/

	int retint = 0;
	std::string comp_key = compTable[comp].section + "." + key;
	std::multimap<std::string, int>::iterator it = intMap.begin();
	std::multimap<std::string, int>::iterator ite = intMap.end();

	for( std::multimap<std::string, int>::iterator its = it; its !=ite; its++ ) {
		/*-------- DEBUG LOG for sslproxy --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
		    comp != PARAM_COMP_LOGGER) {
			LOGGER_PUT_LOG_DEBUG(parameter_cat, 11,
				"function : int l7vs::ParameterImpl::getIntValue() : "
				"Search intMap [%s][%d]",
				its->first.c_str(),
				its->second);
		}
		/*------ DEBUG LOG END for sslproxy ------*/
		if (its->first == comp_key) {
			/*-------- DEBUG LOG for sslproxy --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
			    comp != PARAM_COMP_LOGGER) {
				LOGGER_PUT_LOG_DEBUG(parameter_cat, 12,
					"function : int l7vs::ParameterImpl::getIntValue() : "
					"Key matched.");
			}
			/*------ DEBUG LOG END for sslproxy ------*/
			retint = its->second;
		}
	}

	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 13,
			"out_function : int l7vs::ParameterImpl::getIntValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key) : "
			"retint = %d",
			retint);
	}
	/*------ DEBUG LOG END for sslproxy ------*/
	return retint;
}

/*!
 * get a string parameter
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	key	key string
 * @return		string data
 */
std::string	l7vs::ParameterImpl::getStringValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 14,
			"in_function : std::string l7vs::ParameterImpl::getStringValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key) : "
			"comp = %s, "
			"key = %s",
			compTable[comp].section.c_str(),
			key.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/

	std::string	retstr = "";
	std::string	comp_key = compTable[comp].section + "." + key;
	std::multimap<std::string, std::string>::iterator it = stringMap.begin();
	std::multimap<std::string, std::string>::iterator ite = stringMap.end();

	for( std::multimap<std::string, std::string>::iterator its = it; its !=ite; its++ ) {
		/*-------- DEBUG LOG for sslproxy --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
		    comp != PARAM_COMP_LOGGER) {
			LOGGER_PUT_LOG_DEBUG(parameter_cat, 15,
				"function : std::string l7vs::ParameterImpl::getStringValue() : "
				"Search stringMap [%s][%s]",
				its->first.c_str(),
				its->second.c_str());
		}
		/*------ DEBUG LOG END for sslproxy ------*/
		if (its->first == comp_key) {
			/*-------- DEBUG LOG for sslproxy --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
			    comp != PARAM_COMP_LOGGER) {
				LOGGER_PUT_LOG_DEBUG(parameter_cat, 16,
					"function : std::string l7vs::ParameterImpl::getStringValue() : "
					"Key matched.");
			}
			/*------ DEBUG LOG END for sslproxy ------*/
			retstr = its->second;
		}
	}

	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 17,
			"out_function : std::string l7vs::ParameterImpl::getStringValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key) : "
			"retstr = %s",
			retstr.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/
	return retstr;
}

/*!
 * get map parameter
 * @param[in]	comp	PARAMETER_COMPONENT_TAG
 * @param[in]	key	key string
 * @param[out]	retMap	map data
 */
void	l7vs::ParameterImpl::getStringMapValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key, std::multimap<std::string, std::string>& retMap)
{
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 18,
			"in_function : void l7vs::ParameterImpl::getStringMapValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key, "
			"std::multimap<std::string, std::string>& retMap) : "
			"comp = %s, "
			"key = %s",
			compTable[comp].section.c_str(),
			key.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/

	std::string	comp_key = compTable[comp].section + "." + key;
	std::multimap<std::string, std::string>::iterator it = stringMap.begin();
	std::multimap<std::string, std::string>::iterator ite = stringMap.end();

	for( std::multimap<std::string, std::string>::iterator its = it; its !=ite; its++ ) {
		/*-------- DEBUG LOG for sslproxy --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
		    comp != PARAM_COMP_LOGGER) {
			LOGGER_PUT_LOG_DEBUG(parameter_cat, 19,
				"function : void l7vs::ParameterImpl::getStringMapValue() : "
				"Search stringMap [%s][%s]",
				its->first.c_str(),
				its->second.c_str());
		}
		/*------ DEBUG LOG END for sslproxy ------*/
		if (its->first == comp_key) {
			/*-------- DEBUG LOG for sslproxy --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
			    comp != PARAM_COMP_LOGGER) {
				LOGGER_PUT_LOG_DEBUG(parameter_cat, 20,
					"function : void l7vs::ParameterImpl::getStringMapValue() : "
					"Key matched. Insert map.");
			}
			/*------ DEBUG LOG END for sslproxy ------*/
			retMap.insert( std::pair<std::string, std::string>( key, its->second ) );
		}
	}

	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(parameter_cat) &&
	    comp != PARAM_COMP_LOGGER) {
		LOGGER_PUT_LOG_DEBUG(parameter_cat, 21,
			"out_function : void l7vs::ParameterImpl::getStringMapValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key, "
			"std::multimap<std::string, std::string>& retMap)");
	}
	/*------ DEBUG LOG END for sslproxy ------*/
}

/*!
 * set-parameter function pointer relates component-tag
 * @param[in]	comp	section TAG
 * @param[in]	p_func	function pointer
 */
void	l7vs::ParameterImpl::registerFunctionPointer( const PARAMETER_COMPONENT_TAG comp, void (*p_func)() )
{
	compTable[comp].function = p_func;
}

