/*
 * @file  parameter_wrapper.h
 * @brief parameter module c wrapper.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *      
 **********************************************************************/

#ifndef PARAMETER_WRAPPER
#define PARAMETER_WRAPPER

#include "parameter.h"

/*!
 * check whether integer data exists.
 * @param[in]	comp	section TAG
 * @param[in]	*key	key string
 * @return	1 = exist setting value / 0 = non exist setting value
 */
inline int parameter_is_int_exist(const PARAMETER_COMPONENT_TAG comp, const char* key)
{
	if (l7vs::Parameter::getInstance().isIntExist(comp, key)) {
		return 1;
	}
	return 0;
}

/*!
 * check whether character data exists.
 * @param[in]	comp	section TAG
 * @param[in]	*key	key string
 * @return	1 = exist setting value / 0 = non exist setting value
 */
inline int parameter_is_char_exist(const PARAMETER_COMPONENT_TAG comp, const char* key)
{
	if (l7vs::Parameter::getInstance().isStringExist(comp, key)) {
		return 1;
	}
	return 0;
}

/*!
 * get integer data.
 * @param[in]	comp	section TAG
 * @param[in]	*key	key string
 * @return	value
 */
inline int parameter_get_int_value(const PARAMETER_COMPONENT_TAG comp, const char* key)
{
	return l7vs::Parameter::getInstance().getIntValue(comp, key);
}

/*!
 * get character data.
 * @param[in]	comp	section TAG
 * @param[in]	*key	key string
 * @return	value
 */
inline const char* parameter_get_char_value(const PARAMETER_COMPONENT_TAG comp, const char* key)
{
	return (l7vs::Parameter::getInstance().getStringValue(comp, key)).c_str();
}

/*!
 * reload config file
 * @param[in]	comp	section TAG
 * @param[in]	filename	config file name
 * @return	0 = success read file / -1 = failure read file
 */
inline int parameter_reread_file(const PARAMETER_COMPONENT_TAG comp, const std::string& filename)
{
	if (l7vs::Parameter::getInstance().rereadFile(comp, filename)) {
		/*-------- DEBUG LOG for sslproxy --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_PARAMETER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_PARAMETER, 22,
				"function : inline int parameter_reread_file("
				"const PARAMETER_COMPONENT_TAG comp, "
				"const std::string& filename) : "
				"Parameter.rereadFile("
				"comp = %d, "
				"filename = %s) END.",
				comp,
				filename.c_str());
		}
		/*------ DEBUG LOG END for sslproxy ------*/
		return 0;
	}
	return -1;
}

/*!
 * set-parameter function pointer relates component-tag
 * @param[in]	comp	section TAG
 * @param[in]	p_func	function pointer
 */
inline void parameter_register_function_pointer(const PARAMETER_COMPONENT_TAG comp, void(*p_func)())
{
	l7vs::Parameter::getInstance().registerFunctionPointer(comp, p_func);
}

#endif	//PARAMETER_WRAPPER

