/*
 * @file  sslproxyserver.cpp
 * @brief module of SSLproxy_server
 * @brief SSL setting and Accept connection.
 *
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************
 *
 * Distributed under the Boost Software Licence, Version 1.0
 * http://www.boost.org/LICENSE_1_0.txt
 *
 **********************************************************************/

#include <sys/syscall.h>	// for SYS_gettid
#include "sslproxyserver.h"

//! SSLproxy session management table.
std::map<long, int> sessionTable;
pthread_mutex_t sessionTable_mutex = PTHREAD_MUTEX_INITIALIZER;

/*!
 * SSLproxy_server constructor.
 * Set SSL context and Create session and Accept start.
 *
 * @param[in]	io_service	IO event dispatcher
 * @param[in]	target_itr	target endpoint iterator
 * @param[in]	recv_endpoint	recv endpoint
 */
sslproxy_server::sslproxy_server(boost::asio::io_service& io_service, 
				 boost::asio::ip::tcp::resolver::iterator target_itr, 
				 boost::asio::ip::tcp::endpoint recv_endpoint)
				 :
				 sslservice(io_service),
				 sslacceptor(io_service, recv_endpoint),
				 sslcontext(io_service, DEFAULT_SSL_METHOD),
				 target_endpoint_iterator(target_itr),
				 private_key_passwd("")
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 1,
			"in_function : Constructor sslproxy_server::sslproxy_server("
			"boost::asio::io_service& io_service, "
			"boost::asio::ip::tcp::resolver::iterator target_itr, "
			"boost::asio::ip::tcp::endpoint recv_endpoint)");
	}
	/*------ DEBUG LOG END ------*/

	std::string function_str = "";
	try {
		// SSL context setting.
		// Set root CA file or path.
		// If ca_file is valid then specified file. (load_verify_file())
		// If ca_file is invalid then specified path. (add_verify_path())
		if (ca_file.size() == 0) {
			try {
				sslcontext.add_verify_path(ca_dir);
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 2,
						"function : Constructor sslproxy_server::sslproxy_server() : "
						"add_verify_path() END. "
						"path = %s",
						ca_dir.c_str());
				}
				/*------ DEBUG LOG END ------*/
			} catch (std::exception& e) {
				std::cerr << "Set root CA path error : " << e.what() << "." << std::endl;
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 1, 
					"Set root CA path error : %s.", e.what());
				throw -1;
			}
		} else {
			try {
				sslcontext.load_verify_file(ca_dir + ca_file);
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 3,
						"function : Constructor sslproxy_server::sslproxy_server() : "
						"load_verify_file() END. "
						"file = %s",
						(ca_dir + ca_file).c_str());
				}
				/*------ DEBUG LOG END ------*/
			} catch (std::exception& e) {
				std::cerr << "Set root CA file error : " << e.what() << "." << std::endl;
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 2, 
					"Set root CA file error : %s.", e.what());
				throw -1;
			}
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 4,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set root CA END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set certificate chain file.
		try {
			sslcontext.use_certificate_chain_file(cert_chain_dir + cert_chain_file);
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 5,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"use_certificate_chain_file() END. "
					"file = %s",
					(cert_chain_dir + cert_chain_file).c_str());
			}
			/*------ DEBUG LOG END ------*/
		} catch (std::exception& e) {
			std::cerr << "Set certificate chain file error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 3, 
				"Set certificate chain file error : %s.", e.what());
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 6,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set certificate chain END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set password callback function.
		try {
			sslcontext.set_password_callback(boost::bind(&sslproxy_server::get_password, this));
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 7,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"set_password_callback() END. "
					"callback = get_password()");
			}
			/*------ DEBUG LOG END ------*/
		} catch (std::exception& e) {
			std::cerr << "Set password callback error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 4, 
				"Set password callback error : %s.", e.what());
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 8,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set password callback END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set private key file and filetype.
		try {
			sslcontext.use_private_key_file(private_key_dir + private_key_file, private_key_filetype);
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 9,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"use_private_key_file() END. "
					"file = %s, "
					"filetype = %d",
					(private_key_dir + private_key_file).c_str(), 
					private_key_filetype);
			}
			/*------ DEBUG LOG END ------*/
		} catch (std::exception& e) {
			std::cerr << "Set private key file and filetype error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 5, 
				"Set private key file and filetype error : %s.", e.what());
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 10,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set private key END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set verify options on the context.
		try {
			sslcontext.set_verify_mode(verify_options);
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 11,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"set_verify_mode() END. "
					"verify_options = %d",
					verify_options);
			}
			/*------ DEBUG LOG END ------*/
		} catch (std::exception& e) {
			std::cerr << "Set verify option error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 6, 
				"Set verify option error : %s.", e.what());
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 12,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set verify option END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set verify depth on the context.
		SSL_CTX_set_verify_depth(sslcontext.impl(), verify_cert_depth);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 13,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set verify depth END. (SSL_CTX_set_verify_depth()) "
				"depth = %d",
				verify_cert_depth);
		}
		/*------ DEBUG LOG END ------*/

		// Set SSL options on the context.
		try {
			sslcontext.set_options(ssl_options);
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 14,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"set_options() END. "
					"ssl_options = %ld",
					ssl_options);
			}
			/*------ DEBUG LOG END ------*/
		} catch (std::exception& e) {
			std::cerr << "Set SSL option error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 7, 
				"Set SSL option error : %s.", e.what());
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 15,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set SSL option END.");
		}
		/*------ DEBUG LOG END ------*/

		// Set temporary Diffie-Hellman parameters file.
		if (tmp_dh_use == true) {
			try {
				sslcontext.use_tmp_dh_file(tmp_dh_dir + tmp_dh_file);
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 16,
						"function : Constructor sslproxy_server::sslproxy_server() : "
						"use_tmp_dh_file() END. "
						"file = %s",
						(tmp_dh_dir + tmp_dh_file).c_str());
				}
				/*------ DEBUG LOG END ------*/
			} catch (std::exception& e) {
				std::cerr << "Set tmp DH file error : " << e.what() << "." << std::endl;
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 8, 
					"Set tmp DH file error : %s.", e.what());
				throw -1;
			}

			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 17,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"Set tmp DH file END.");
			}
			/*------ DEBUG LOG END ------*/
		}

		// Set cipher list on the context.
		if (SSL_CTX_set_cipher_list(sslcontext.impl(), cipher_list.c_str()) != 1) {
			std::cerr << "Set cipher list error : no valid ciphers." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 9, 
				"Set cipher list error : no valid ciphers.");
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 18,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set cipher list END. "
				"cipher_list = %s",
				cipher_list.c_str());
		}
		/*------ DEBUG LOG END ------*/

		// Set session cache mode on the context.
		SSL_CTX_set_session_cache_mode(sslcontext.impl(), session_cache_mode);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 42,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set session cache mode END. "
				"session_cache_mode = %ld",
				SSL_CTX_get_session_cache_mode(sslcontext.impl()));
		}
		/*------ DEBUG LOG END ------*/

		// Set session cache size on the context.
		SSL_CTX_sess_set_cache_size(sslcontext.impl(), session_cache_size);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 43,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set session cache size END. "
				"session_cache_size = %ld",
				SSL_CTX_sess_get_cache_size(sslcontext.impl()));
		}
		/*------ DEBUG LOG END ------*/

		// Set session cache timeout on the context.
		SSL_CTX_set_timeout(sslcontext.impl(), session_cache_timeout);
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 44,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Set session cache timeout END. "
				"session_cache_timeout = %ld",
				SSL_CTX_get_timeout(sslcontext.impl()));
		}
		/*------ DEBUG LOG END ------*/

		// Create sslproxy_session for first acception.
		function_str = "Create sslproxy_session for first";
		sslproxy_session* session = new sslproxy_session(sslservice,
								 sslcontext,
								 target_endpoint_iterator);
		sessionTable.clear();
		sessionTable[(long)session] = 1;
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 19,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Create sslproxy_session END.");
		}
		/*------ DEBUG LOG END ------*/

		// Start an asynchronous accept for first.
		try {
			ssl_socket::lowest_layer_type& sock = session->low_socket();
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 20,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"low_socket() END.");
			}
			/*------ DEBUG LOG END ------*/

			sslacceptor.async_accept(sock,
						 boost::bind(&sslproxy_server::handle_accept,
							     this,
							     session,
							     boost::asio::placeholders::error));
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 21,
					"function : Constructor sslproxy_server::sslproxy_server() : "
					"async_accept() END.");
			}
			/*------ DEBUG LOG END ------*/
		} catch (int e) {
			std::cerr << "Get session socket for first error." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 10, 
				"Get session socket for first error.");
			delete session;
			throw -1;
		} catch (std::exception& e) {
			std::cerr << "Asynchronous accept for first error : " << e.what() << "." << std::endl;
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 11, 
				"Asynchronous accept for first error : %s.", e.what());
			delete session;
			throw -1;
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 22,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Asynchronous accept for first END.");
		}
		/*------ DEBUG LOG END ------*/

	} catch (std::exception& e) {
		std::cerr << function_str << " error : " << e.what() << "." << std::endl;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 12, 
			"%s error : %s.", function_str.c_str(), e.what());
		throw -1;
	} catch (int e) {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 23,
				"function : Constructor sslproxy_server::sslproxy_server() : "
				"Catch int exception. %d", 
				e);
		}
		/*------ DEBUG LOG END ------*/
		throw -1;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 24,
			"out_function : Constructor sslproxy_server::sslproxy_server("
			"boost::asio::io_service& io_service, "
			"boost::asio::ip::tcp::resolver::iterator target_itr, "
			"boost::asio::ip::tcp::endpoint recv_endpoint)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * SSLproxy_server destructor.
 */
sslproxy_server::~sslproxy_server()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 25,
			"in/out_function : Destructor sslproxy_server::~sslproxy_server()");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Get password callback function.
 *
 * @return	password string
 * @retval	string	success
 * @retval	""	failed
 */
std::string sslproxy_server::get_password()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 26,
			"in_function : std::string sslproxy_server::get_password(void)");
	}
	/*------ DEBUG LOG END ------*/

	private_key_passwd = "";
	if (private_key_passwd_from == "console") {
		// Get password from console.
		std::cout << "Input private key passwd : ";
		std::cin >> private_key_passwd;
		if (private_key_passwd.size() > MAX_PASSWD_SIZE) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 13, "Password is too long.");
			private_key_passwd = "";
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 27,
				"function : std::string sslproxy_server::get_password(void) : "
				"Get password from console END. "
				"pass = %s",
				private_key_passwd.c_str());
		}
		/*------ DEBUG LOG END ------*/
	} else if (private_key_passwd_from == "file") {
		// Get password from file.
		FILE  *fp;
		char buf[MAX_PASSWD_SIZE + 3];
		if ((fp = fopen((private_key_passwd_dir + private_key_passwd_file).c_str(), "r")) == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 14, "Password file cannot open.");
		} else {
			if (fgets(buf, MAX_PASSWD_SIZE + 3, fp) == NULL) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 15, "Password not found in file.");
			} else {
				if (strlen(buf) > MAX_PASSWD_SIZE) {
					LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 16, "Password is too long.");
				} else {
					buf[strlen(buf) - 1] = '\0';
					private_key_passwd = buf;
					/*-------- DEBUG LOG --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 28,
							"function : std::string sslproxy_server::get_password(void) : "
							"Read password from file END.");
					}
					/*------ DEBUG LOG END ------*/
				}
			}
			fclose(fp);
		}
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 29,
				"function : std::string sslproxy_server::get_password(void) : "
				"Get password from file END. "
				"pass = %s",
				private_key_passwd.c_str());
		}
		/*------ DEBUG LOG END ------*/
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 30,
			"out_function : std::string sslproxy_server::get_password(void) : "
			"return value = %s",
			private_key_passwd.c_str());
	}
	/*------ DEBUG LOG END ------*/
	return private_key_passwd;
}

/*!
 * Acception handling function.
 *
 * @param[in]	session	session object
 * @param[in]	error	error code
 */
void sslproxy_server::handle_accept(sslproxy_session* session,
				    const boost::system::error_code& error)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 31,
			"in_function : void sslproxy_server::handle_accept("
			"sslproxy_session* session, "
			"const boost::system::error_code& error)");
	}
	/*------ DEBUG LOG END ------*/

	std::string function_str = "";
	try {
		// Check async_accept() result.
		if (!error) {
			// sslproxy_session start.
			session->start();
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 32,
					"function : void sslproxy_server::handle_accept() : "
					"Session start END.");
			}
			/*------ DEBUG LOG END ------*/

			// Create sslproxy_session for next acception.
			function_str = "Create sslproxy_session for next";
			session = new sslproxy_session(sslservice,
						       sslcontext,
						       target_endpoint_iterator);
			pthread_mutex_lock( &sessionTable_mutex );
			sessionTable[(long)session] = 1;
			pthread_mutex_unlock( &sessionTable_mutex );
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 33,
					"function : void sslproxy_server::handle_accept() : "
					"Create sslproxy_session for next END.");
			}
			/*------ DEBUG LOG END ------*/

			// Start an asynchronous accept for next.
			try {
				ssl_socket::lowest_layer_type& sock = session->low_socket();
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 34,
						"function : void sslproxy_server::handle_accept() : "
						"low_socket() END.");
				}
				/*------ DEBUG LOG END ------*/

				sslacceptor.async_accept(sock,
							 boost::bind(&sslproxy_server::handle_accept,
							 this,
							 session,
							 boost::asio::placeholders::error));
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 35,
						"function : void sslproxy_server::handle_accept() : "
						"async_accept() for next END.");
				}
				/*------ DEBUG LOG END ------*/
			} catch (int e) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 17, 
					"Get session socket for next error.");
				delete session;
				throw -1;
			} catch (std::exception& e) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 18, 
					"Asynchronous accept for next error : %s.", e.what());
				delete session;
				throw -1;
			}
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 36,
					"function : void sslproxy_server::handle_accept() : "
					"Asynchronous accept for next END.");
			}
			/*------ DEBUG LOG END ------*/

		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_SSLPROXY_SERVER, 1, 
				"Acception result NG. Retry acception : %s.",
				error.message().c_str());

			// Start an asynchronous accept for retry.
			try {
				ssl_socket::lowest_layer_type& sock = session->low_socket();
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 37,
						"function : void sslproxy_server::handle_accept() : "
						"low_socket() END.");
				}
				/*------ DEBUG LOG END ------*/

				sslacceptor.async_accept(sock,
							 boost::bind(&sslproxy_server::handle_accept,
							 this,
							 session,
							 boost::asio::placeholders::error));
				/*-------- DEBUG LOG --------*/
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 38,
						"function : void sslproxy_server::handle_accept() : "
						"async_accept() for retry END.");
				}
				/*------ DEBUG LOG END ------*/
			} catch (int e) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 19, 
					"Get session socket for retry error.");
				delete session;
				throw -1;
			} catch (std::exception& e) {
				LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 20, 
					"Asynchronous accept for retry error : %s.", e.what());
				delete session;
				throw -1;
			}
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 39,
					"function : void sslproxy_server::handle_accept() : "
					"Asynchronous accept for retry END.");
			}
			/*------ DEBUG LOG END ------*/
		}

	} catch (std::exception& e) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_SSLPROXY_SERVER, 21, 
			"%s error : %s.", function_str.c_str(), e.what());
	} catch (int e) {
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 40,
				"function : void sslproxy_server::handle_accept() : "
				"Catch int exception. %d", 
				e);
		}
		/*------ DEBUG LOG END ------*/
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_SSLPROXY_SERVER)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_SSLPROXY_SERVER, 41,
				"out_function : void sslproxy_server::handle_accept("
				"sslproxy_session* session, "
				"const boost::system::error_code& error)");
	}
	/*------ DEBUG LOG END ------*/
}
