/*
 * @file  protomod_cpassive.c
 * @brief protocol module of HTTP Cookie.
 * @brief this module provide session persistence by Original Cookie.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define __STDC_LIMIT_MACROS
#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include <arpa/inet.h>
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define SERVICE_ARG_MAXSIZE    (512)
#define CPASSIVE_SERVICE_NUMBER (128)
#define COOKIE_NAME_MAXSIZE    (128)
#define X_FORWARDED_FOR_LENGTH (48)
#define DEST_ENCODE_LENGTH     (16)

struct l7vs_cpassive_service {
	handle_t service_handle;
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int forwarded_for;
	int reschedule;
};

struct  l7vs_cpassive_service_arg {
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int forwarded_for;
	int reschedule;
};

static void  fini(void);
static int   create(void*, handle_t);
static void* create_sa(struct l7vs_service_arg*);
static int   compare(handle_t, handle_t);
static int   match_cldata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*, struct l7vs_dest**, int*);
static int   analyze_rsdata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*);
static int   destroy(handle_t);
static void  destroy_sa(void**);
static int   service_arg(struct l7vs_service_arg_multi*, handle_t);
static int   parse(void*, int, char**);

static struct l7vs_cpassive_service* l7vs_protomod_cpassive_search_service(handle_t);
static struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_service();
static struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_temp_service();

//static int    l7vs_protomod_insert_field(char*, int, char*, int);

static void l7vs_cpassive_service_c_str(char*, struct l7vs_cpassive_service*);
static void l7vs_cpassive_service_arg_c_str(char*, struct l7vs_cpassive_service_arg*);

struct l7vs_cpassive_service* cpassive_service_list[CPASSIVE_SERVICE_NUMBER];

static struct l7vs_protomod cpassive_protomod = {
	NULL,           /* handle */
	"cpassive",      /* modname */
	0,              /* refcnt */
	0,              /* fast schedule */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
	NULL,           /* initialize function */
	NULL,           /* finalize function */
	NULL,           /* get_log_level function */
	NULL,           /* put_log_debug function */
	NULL,           /* put_log_info function */
	NULL,           /* put_log_warn function */
	NULL,           /* put_log_error function */
	NULL            /* put_log_fatal function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param[in] handle dlopen's handle
 * @return l7vs_protomod struct
 */
extern "C" struct l7vs_protomod *
init(void *handle)
{
	struct l7vs_protomod* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,170,
	            "in_function: struct l7vs_protomod* init(void* handle): handle=%p", handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (handle == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,152, "Arg(handle) is NULL pointer.");
		goto init_out;
	}

	/* initialize cpassive service list */
	memset(cpassive_service_list, 0, sizeof(struct l7vs_cpassive_service *) * CPASSIVE_SERVICE_NUMBER);
	/* set dlopen's handle */
	cpassive_protomod.handle = handle;

	return_value = &cpassive_protomod;

init_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char protomod_str[DEBUG_STR_LEN];
		l7vs_protomod_c_str(protomod_str, &cpassive_protomod);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,171,
		    "out_function: struct l7vs_protomod* init(void* handle): return=&(%s)", protomod_str);
	}
	/*------ DEBUG LOG END ------*/
	return return_value;
}

/*!
 * Protocol module finalize function. free all cpassive service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* cpassive service list counter */
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,172, "in_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/

	/* check all cpassive service list */
	for (service_number = 0; service_number < CPASSIVE_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cpassive_service_list[service_number] != NULL) {

			/*-------- DEBUG LOG --------*/
			if (cpassive_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,46, "free: %p",
				    cpassive_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			/* free and points NULL */
			free(cpassive_service_list[service_number]);
			cpassive_service_list[service_number] = NULL;
		}
	}
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,173, "out_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create cpassive service struct.
 * @param[in] cpassive_arg    cpassive service argument struct
 * @param[in] service_handle a unique service ID
 * @retval 0  successfully create cpassive service.
 * @retval -1 some errors occur.
 */
static int
create(void *cpassive_arg, handle_t service_handle)
{
	struct l7vs_cpassive_service *cpassive_service;
	struct l7vs_cpassive_service_arg *cpassive_service_arg;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_arg_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_arg_c_str(cpassive_arg_str, (struct l7vs_cpassive_service_arg*) cpassive_arg);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,174,
		    "in_function: int create(void* cpassive_arg, handle_t service_handle):cpassive_arg=&(%s), "
		    "service_handle=%u", cpassive_arg_str, service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cpassive_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,153, "Arg(cpassive_arg) is NULL pointer.");
		return_value = -1;
		goto create_out;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty cpassive service list and create cpassive service */
		cpassive_service = l7vs_protomod_cpassive_create_service();
	} else {
		/* create temporary cpassive service */
		cpassive_service = l7vs_protomod_cpassive_create_temp_service();
	}

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(cpassive_str, cpassive_service);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,175, "pointer assign: cpassive_service=&(%s)",
		    cpassive_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_service == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,154, "Could not make cpassive service.");
		return_value = -1;
		goto create_out;
	}

	cpassive_service_arg = (struct l7vs_cpassive_service_arg *) cpassive_arg;

	/* set service handle, option value */
	cpassive_service->service_handle = service_handle;
	strncpy(cpassive_service->cookie_name, cpassive_service_arg->cookie_name, COOKIE_NAME_MAXSIZE);
	cpassive_service->forwarded_for = cpassive_service_arg->forwarded_for;
	cpassive_service->reschedule = cpassive_service_arg->reschedule;

create_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,176,
		    "out_function: int create(void* cpassive_arg, handle_t service_handle):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create cpassive service argument struct.
 * @param[out] srv_arg service argument struct
 * @return cpassive service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_cpassive_service_arg *cpassive_service_arg;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char service_arg_str[DEBUG_STR_LEN];
		l7vs_service_arg_c_str(service_arg_str, srv_arg);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,177,
		    "in_function: void* create_sa(struct l7vs_service_arg* srv_arg):srv_arg=&(%s)",
		    service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,155, "Arg(srv_arg) is NULL pointer.");
		cpassive_service_arg = NULL;
		goto create_sa_out;
	}

	/* create cpassive service argument struct */
	cpassive_service_arg = (struct l7vs_cpassive_service_arg *) calloc(1, sizeof(struct l7vs_cpassive_service_arg));

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,47, "calloc: addr=%p, size=%ld",
		    cpassive_service_arg, (unsigned long int) sizeof(struct l7vs_cpassive_service_arg));
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_service_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,47, "Could not allocate memory.");
		goto create_sa_out;
	}

	/* set cpassive service argument size and protomod name "cpassive" */
	srv_arg->len = sizeof(struct l7vs_cpassive_service_arg);
	strcpy(srv_arg->protomod, cpassive_protomod.modname);

create_sa_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_service_arg_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_arg_c_str(cpassive_service_arg_str, cpassive_service_arg);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,178,
		    "out_function: void* create_sa(struct l7vs_service_arg* srv_arg):return_value=&(%s)",
		    cpassive_service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	return (void*) cpassive_service_arg;
}

/*!
 * Compare two service.
 * @param[in] srv_handle1 one of a unique service ID
 * @param[in] srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(handle_t srv_handle1, handle_t srv_handle2)
{
	struct l7vs_cpassive_service *cpassive_srv1, *cpassive_srv2;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,179,
		    "in_function: int compare(handle_t srv_handle1, handle_t srv_handle2):"
		    "srv_handle1=%u, srv_handle2=%u", srv_handle1, srv_handle2);
	}
	/*------ DEBUG LOG END ------*/

	/* search service that has such a service ID(1) */
	cpassive_srv1 = l7vs_protomod_cpassive_search_service(srv_handle1);

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(cpassive_str, cpassive_srv1);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,180, "pointer assign: cpassive_srv1=&(%s)",
		    cpassive_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_srv1 == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,156,
		    "Could not find such service handle's cpassive service.");
		return_value = -1;
		goto compare_out;
	}

	/* search service that has such a service ID(2) */
	cpassive_srv2 = l7vs_protomod_cpassive_search_service(srv_handle2);

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(cpassive_str, cpassive_srv2);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,181, "pointer assign: cpassive_srv2=&(%s)",
		    cpassive_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_srv2 == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,157,
		    "Could not find such service handle's cpassive service.");
		return_value = -1;
		goto compare_out;
	}

	/* compare two cookie name */
	if (strcmp(cpassive_srv1->cookie_name, cpassive_srv2->cookie_name) != 0) {
		return_value = -1;
		goto compare_out;
	}

compare_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,182,
		    "out_function: int compare(handle_t srv_handle1, handle_t srv_handle2):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check the client packet and determine a real server.
 * @param[in]  srv service struct include service handle, protocol module and schedule module.
 * @param[in]  conn connection data.
 * @param[in]  request packet data from client
 * @param[in]  len length of packet data
 * @param[out] dest destination (real server) list
 * @param[out] tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	struct l7vs_cpassive_service *cpassive_service;
	struct l7vs_dest destination;
	int ret;
	int offset_length;
	char *cookie_value;
	char *x_forwarded_value;
	char *next_line = NULL;
	char encoded_address[DEST_ENCODE_LENGTH];
	char x_forwarded_for_header[X_FORWARDED_FOR_LENGTH];
	size_t uri_len = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN];
		char conn_str[DEBUG_STR_LEN];
		char dest_str[DEBUG_STR_LEN];
		char len_str[DEBUG_STR_LEN];
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (dest != NULL) {
			l7vs_dest_c_str(dest_str, *dest);
		}
		else {
			strncpy(dest_str, "NULL", DEBUG_STR_LEN);
		}
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,183,
		    "in_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):srv=&(%s), conn=&(%s), "
		    "request=\"%s\", len=&(%s), dest=&(&(%s)), tcps=&(%d)",
		    srv_str, conn_str, request, len_str, dest_str, *tcps);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,158, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (srv->pm == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,159, "Arg(srv->pm) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (request == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,160, "Arg(request) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,161, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (dest == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,162, "Arg(dest) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (tcps == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,163, "Arg(tcps) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* search service that has such a service ID */
	cpassive_service = l7vs_protomod_cpassive_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(cpassive_str, cpassive_service);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,184, "pointer assign: cpassive_service=&(%s)",
		    cpassive_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_service == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,164, "Could not find such service handle's cpassive service.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,165, "Could not initialize protomod.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* check the HTTP method in HTTP request header */
	uri_len = *len;
	if (http_check_request_method(request, &uri_len) == NULL) {
		PUT_LOG_INFO(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,13, "Client message is not HTTP request.");
		goto match_cldata_finalize;
	}

	/* add X-Forwarded-For field */
	if (cpassive_service->forwarded_for) {
		x_forwarded_value = http_search_header_field(request, "X-Forwarded-For");

		/* already exists X-Forwarded-For field */
		if (x_forwarded_value) {
			next_line = http_skip_header_line(x_forwarded_value);
			/* backtrack to look up insert point */
			if (next_line) {
				next_line--; // *next_line == '\n'
				if (*(next_line - 1) == '\r') {
					next_line--;
				}
				/* append client IP address */
				snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, ", %s", inet_ntoa(conn->caddr.sin_addr));
			}
		}

		/* not exists X-Forwarded-For field */
		if (!next_line) {
			/* construct new X-Forwarded-For header item */
			snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, "X-Forwarded-For: %s\r\n", inet_ntoa(conn->caddr.sin_addr));
	
			next_line = http_skip_header_line(request);
		}

		/* when insert point exist */
		if (next_line != NULL) {
			offset_length = (int) (next_line - request);
	
			/* insert X-Forwarded-For header field */
			http_insert_field(request, offset_length, x_forwarded_for_header, *len);
		
			/* add header length */
			*len += strlen(x_forwarded_for_header);
		}
	}

	/* search Cookie field in HTTP request header */
	cookie_value = http_search_header_cookie_value(request, cpassive_service->cookie_name);
	if (cookie_value == NULL) {
		/*-------- DEBUG LOG --------*/
		if (cpassive_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,185, "Cookie not found.");
		}
		/*------ DEBUG LOG END ------*/

		goto match_cldata_finalize;
	}

	/* memory copy encoded strings(16 byte) of IP address and port number */
	memcpy(encoded_address, cookie_value, DEST_ENCODE_LENGTH);
	encoded_address[DEST_ENCODE_LENGTH] = '\0';

	/* decode IP address and port number and set to destination struct */
	http_decode_address(encoded_address, (u_long *) &destination.addr.sin_addr.s_addr, &destination.addr.sin_port);

	/* set return value */
	*dest = &destination;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char dest_str[DEBUG_STR_LEN];
		l7vs_dest_c_str(dest_str, *dest);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,186, "pointer assign: dest=&(&(%s))",
		    dest_str);
	}
	/*------ DEBUG LOG END ------*/

match_cldata_finalize:
	*tcps = 0;

	/* finalize */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, cpassive_service->reschedule);

	if (ret != 0){
		PUT_LOG_INFO(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,14, "Could not finalize protomod. (Realserver decision failure)");
		return_value = -1;
		goto match_cldata_out;
	}

match_cldata_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,187,
		    "out_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check the real server packet and insert a Set-Cookie field.
 * @param[in] srv service struct include service handle, protocol module and schedule module.
 * @param[in] conn connection data.
 * @param[in] response packet data from real server
 * @param[in] len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn,
	char* response, size_t* len)
{
	struct l7vs_cpassive_service *cpassive_service;
	char *next_line;
	char encoded_address[DEST_ENCODE_LENGTH];
	char set_cookie_field[L7VS_PROTOMOD_MAX_ADD_BUFSIZE];
	int  offset_length;
	int  ret;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN];
		char conn_str[DEBUG_STR_LEN];
		char len_str[DEBUG_STR_LEN];
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,188,
		    "in_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):srv=&(%s), conn=&(%s), response=\"%s\", len=&(%s)",
		    srv_str, conn_str, response, len_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,166, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,167, "Arg(conn) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn->dest == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,168, "Arg(conn->dest) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (response == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,169, "Arg(response) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,170, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

analyze_rsdata_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,189,
		    "out_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy cpassive service
 * @param[in] srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(handle_t srv_handle)
{
	/* cpassive service list counter */
	int service_number = 0;
	int free_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,190,
		    "in_function: int destroy(handle_t srv_handle):srv_handle=%u",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all cpassive service list */
	for (service_number = 0; service_number < CPASSIVE_SERVICE_NUMBER; ++service_number) {
		/* found cpassive service that has srv_handle */
		if (cpassive_service_list[service_number] != NULL && 
		    cpassive_service_list[service_number]->service_handle == srv_handle) {

			/*-------- DEBUG LOG --------*/
			if (cpassive_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,48, "free: %p",
				    cpassive_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			/* free and NULL */
			free(cpassive_service_list[service_number]);
			cpassive_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* cpassive service was not found */
	if (free_flag == 0) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,171, "Could not find such service handle's cpassive service.");
		return_value = -1;
		goto destroy_out;
	}

destroy_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,191,
		    "out_function: int destroy(handle_t srv_handle):return_value=%d",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy cpassive service argument
 * @param[in] cpassive_arg cpassive service argument
 * @return void
 */
static void
destroy_sa(void **cpassive_arg)
{
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_arg_str[DEBUG_STR_LEN];
		if (cpassive_arg != NULL) {
			l7vs_cpassive_service_arg_c_str(cpassive_arg_str, (struct l7vs_cpassive_service_arg*) *cpassive_arg);
		}
		else {
			strncpy(cpassive_arg_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,192,
		    "in_function: void destroy_sa(void** cpassive_arg):cpassive_arg=&(&(%s))",
		    cpassive_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cpassive_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,172, "Arg(cpassive_arg) is NULL pointer.");
	}
	else if (*cpassive_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,173, "Arg(*cpassive_arg) is NULL pointer.");
	}
	else {
		/*-------- DEBUG LOG --------*/
		if (cpassive_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
			PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,49, "free: %p",
			    *cpassive_arg);
		}
		/*------ DEBUG LOG END ------*/

		/* free and NULL */
		free((struct l7vs_cpassive_service_arg*) *cpassive_arg);
		*cpassive_arg = NULL;
	}

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,193,
		    "out_function: void destroy_sa(void** cpassive_arg)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create strings for service list of l7vsadm
 * @param[out] srv_arg service argument struct
 * @param[in]  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, handle_t srv_handle)
{
	struct l7vs_cpassive_service *cpassive_service;
	struct l7vs_cpassive_service_arg c_sarg;
	char cpassive_argument[SERVICE_ARG_MAXSIZE];
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_arg_mt_str[DEBUG_STR_LEN];
		l7vs_service_arg_multi_c_str(srv_arg_mt_str, srv_arg_mt);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,194,
		    "in_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):srv_arg_mt=&(%s), srv_handle=%u",
		    srv_arg_mt_str, srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg_mt == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,174, "Arg(srv_arg_mt) is NULL pointer.");
		return_value = -1;
		goto service_arg_out;
	}

	/* search service that has such a service ID */
	cpassive_service = l7vs_protomod_cpassive_search_service(srv_handle);

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(cpassive_str, cpassive_service);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,195, "pointer assign: cpassive_service=&(%s)",
		    cpassive_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_service == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,175, "Could not find such service handle's cpassive service.");
		return_value = -1;
		goto service_arg_out;
	}

	/* initialize argument strings */
	memset(cpassive_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set cpassive args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = cpassive_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	snprintf(cpassive_argument, SERVICE_ARG_MAXSIZE, "--cookie-name %s", cpassive_service->cookie_name);
	strncpy(srv_arg_mt->srv_arg.protomod_key_string, cpassive_argument, 256);

	/* create verbose argument (l7vsadm option -V/-v) */
	if (cpassive_service->forwarded_for) {
		snprintf(cpassive_argument + strlen(cpassive_argument), SERVICE_ARG_MAXSIZE - strlen(cpassive_argument),
		    " --forwarded-for");
	}
	strncpy(srv_arg_mt->srv_arg.protomod_opt_string, cpassive_argument, 512);

	strncpy(c_sarg.cookie_name, cpassive_service->cookie_name, COOKIE_NAME_MAXSIZE);
	c_sarg.reschedule = cpassive_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_cpassive_service_arg));

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cpassive_arg_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_arg_c_str(cpassive_arg_str, &c_sarg);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,196,
		    "pointer assign: srv_arg_mt->protomod_arg=&(%s)", cpassive_arg_str);
	}
	/*------ DEBUG LOG END ------*/

service_arg_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,197,
		    "out_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Parse l7vsadm options to cpassive argument
 * @param[out] cpassive_arg cpassive service argument struct
 * @param[in]  argc number of l7vsadm argument
 * @param[in]  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *cpassive_arg, int argc, char *argv[])
{
	struct l7vs_cpassive_service_arg *cpassive_service_arg;
	static struct option opt[] = {
		{"cookie-name",   required_argument, NULL, 'C'},
		{"forwarded-for", no_argument,       NULL, 'F'},
		{"reschedule",    no_argument,       NULL, 'R'},
		{"no-reschedule", no_argument,       NULL, 'N'},
		{NULL,            0,                 NULL, 0  }
	};
	int c;
	unsigned long buffer;
	int cookie_name_flag = 0;
	int forwarded_for_flag = 0;
	int reschedule_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		int i;
		char argv_str[DEBUG_STR_LEN];
		char cpassive_arg_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_arg_c_str(cpassive_arg_str, (struct l7vs_cpassive_service_arg*) cpassive_arg);
		argv_str[0] = '\0';
		if (argv == NULL)
			snprintf(argv_str, DEBUG_STR_LEN, "NULL");
		else {
			for (i = 0; i < argc; i++) {
				snprintf(argv_str, DEBUG_STR_LEN, "%sargv[%d]=\"%s\", ", argv_str, i, argv[i]);
			}
			i = strnlen(argv_str, DEBUG_STR_LEN);
			if (i > 1)
				argv_str[i - 2] = '\0';
		}
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,198,
		    "in_function: int parse(void* cpassive_arg, int argc, char* argv[]):cpassive_arg=&(%s), "
		    "argc=%d, %s", cpassive_arg_str, argc, argv_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cpassive_arg == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,176, "Arg(cpassive_arg) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}
	if (argv == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,177, "Arg(argv) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}

	cpassive_service_arg = (struct l7vs_cpassive_service_arg *) cpassive_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "C:E:FRN", opt, NULL)) != -1) {
		switch (c) {
		/* --cookie-name / -C */
		case 'C':
			/* check maximum length */
			if (strlen(optarg) >= sizeof(cpassive_service_arg->cookie_name)) {
				PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,178,
				    "-C/--cookie-name option value '%s' path too long", optarg);
				return_value = -1;
				goto parse_out;
			}
			strncpy(cpassive_service_arg->cookie_name, optarg, COOKIE_NAME_MAXSIZE);
			cookie_name_flag++;
			break;

		/* --forwarded-for / -F */
		case 'F':
			/* x-forwarded-for on */
			cpassive_service_arg->forwarded_for = 1;
			forwarded_for_flag++;
			break;

		/* --reschedule / -R */
		case 'R':
			/* reschedule on */
			cpassive_service_arg->reschedule = 1;
			reschedule_flag++;
			break;

		/* --no-reschedule / -N */
		case 'N':
			/* reschedule off */
			cpassive_service_arg->reschedule = 0;
			reschedule_flag++;
			break;

		/* else error */
		default:
			PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,179, "Option error.");
			return_value = -1;
			goto parse_out;
		}
	}

	/* when set both -R and -N at the same time */
	if (reschedule_flag > 1) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,180,
		    "You have to choose either of reschdule or no-reschedule.");
		return_value = -1;
		goto parse_out;
	}
	/* same option */
	if (cookie_name_flag > 1) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,181,
		    "Cannot set multiple option '--cookie-name/-C'.");
		return_value = -1;
		goto parse_out;
	}

	/* set default no forwarded_for */
	if (forwarded_for_flag == 0) {
		cpassive_service_arg->forwarded_for = 0;
	}

	/* set default no reschedule */
	if (reschedule_flag == 0) {
		cpassive_service_arg->reschedule = 0;
	}

	/* set default cookie name */
	if (cookie_name_flag == 0) {
		strcpy(cpassive_service_arg->cookie_name, "CookieName");
	}

parse_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,199,
		    "out_function: int parse(void* cpassive_arg, int argc, char* argv[]):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Insert any header field to HTTP header
 * @param[in,out] header HTTP header strings
 * @param[in] offset_length offset length
 * @param[in] insert_field insert header field strings
 * @param[in,out] header_length all of HTTP header length
 * @retval 0  successfully insert header field
 * @retval -1 some errors occur.
 */
//static int
//l7vs_protomod_insert_field(char *header, int offset_length, char *insert_field, int header_length)
//{
//	char *copy_buffer;
//	int   return_value = 0;
//
//	/*-------- DEBUG LOG --------*/
//	if (cpassive_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
//		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0,
//		    "in_function: int l7vs_protomod_insert_field(char* header, int offset_length, "
//		    "char* insert_field, int header_length):header=\"%s\", "
//		    "offset_length=%d, insert_field=\"%s\", header_length=%d",
//		    header, offset_length, insert_field, header_length);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	/* check args */
//	if (header == NULL) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header) is NULL pointer.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (insert_field == NULL) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(insert_field) is NULL pointer.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (offset_length < 0) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(offset_length) must be positive number.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (header_length < 0) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header_length) must be positive number.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (header_length < offset_length) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header_length) must be "
//		    "longer than arg(offset_length)");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//
//	/* calloc buffer */
//	copy_buffer = (char *) calloc(1, header_length - offset_length);
//
//	/*-------- DEBUG LOG --------*/
//	if (cpassive_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
//		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "calloc: addr=%p, size=%d",
//		    copy_buffer, header_length - offset_length);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	if (copy_buffer == NULL) {
//		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "Could not allocate memory.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//
//	/* backup strings */
//	memcpy(copy_buffer, header + offset_length, header_length - offset_length);
//
//	/* insert Set-Cookie field */
//	memcpy(header + offset_length, insert_field, strlen(insert_field));
//
//	/* append backup strings and terminate null*/
//	memcpy(header + offset_length + strlen(insert_field), copy_buffer, header_length - offset_length);
//	header[offset_length + strlen(insert_field) + header_length - offset_length] = '\0';
//
//	/*-------- DEBUG LOG --------*/
//	if (cpassive_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
//		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "free: %p",
//		    copy_buffer);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	/* free */
//	free(copy_buffer);
//	copy_buffer = NULL;
//
//set_cookie_out:
//	/*-------- DEBUG LOG --------*/
//	if (cpassive_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
//		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL, 0,
//		    "out_function: int l7vs_protomod_insert_field(char* header, int offset_length, "
//		    "char* insert_field, int header_length):return_value=%d",
//		    return_value);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	return return_value;
//}

/*!
 * Search cpassive service from cpassive service list using service handle
 * @param[in] service_handle a unique service ID
 * @return cpassive service struct when service was found. NULL when service was not found.
 */
static struct l7vs_cpassive_service *
l7vs_protomod_cpassive_search_service(handle_t service_handle)
{
	/* cpassive service list counter */
	int service_number = 0;
	struct l7vs_cpassive_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,200,
		    "in_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_search_service(handle_t service_handle):"
		    "service_handle=%d", service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all cpassive service list */
	for (service_number = 0; service_number < CPASSIVE_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (cpassive_service_list[service_number] != NULL && 
		    cpassive_service_list[service_number]->service_handle == service_handle) {
			return_value = cpassive_service_list[service_number];
			break;
		}
	}
	
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,201,
		    "out_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_search_service(handle_t service_handle):"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create cpassive service.
 * @param  void
 * @return cpassive service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cpassive_service *
l7vs_protomod_cpassive_create_service()
{
	struct l7vs_cpassive_service* return_value = NULL;
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,202,
		    "in_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* check all cpassive service list */
	for (service_number = 0; service_number < CPASSIVE_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cpassive_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			cpassive_service_list[service_number] = (struct l7vs_cpassive_service *) calloc(1, sizeof(struct l7vs_cpassive_service));

			/*-------- DEBUG LOG --------*/
			if (cpassive_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,50, "calloc: addr=%p, size=%ld",
				    cpassive_service_list[service_number], (unsigned long int) sizeof(struct l7vs_cpassive_service));
			}
			/*------ DEBUG LOG END ------*/

			if (cpassive_service_list[service_number] == NULL) {
				PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,48, "Could not allocate memory.");
				goto create_service_out;
			}
			return_value = cpassive_service_list[service_number];
			goto create_service_out;
		}
	}
	
	/* all service list is full */
	PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,182, "cpassive service list is full.");

create_service_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,203,
		    "out_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create temporary cpassive service.
 * @param  void
 * @return cpassive service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cpassive_service *
l7vs_protomod_cpassive_create_temp_service()
{
	struct l7vs_cpassive_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,204,
		    "in_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_temp_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* if pointer that does not point NULL exists ... */
	if (cpassive_service_list[CPASSIVE_SERVICE_NUMBER - 1] != NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,183, "Temporary cpassive service is being used by other process.");
		goto create_temp_service_out;
	}

	/* create temp service */
	cpassive_service_list[CPASSIVE_SERVICE_NUMBER - 1] = (struct l7vs_cpassive_service *) calloc(1, sizeof(struct l7vs_cpassive_service));

	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,51, "calloc: addr=%p, size=%ld",
		    cpassive_service_list[CPASSIVE_SERVICE_NUMBER - 1], (unsigned long int) sizeof(struct l7vs_cpassive_service));
	}
	/*------ DEBUG LOG END ------*/

	if (cpassive_service_list[CPASSIVE_SERVICE_NUMBER - 1] == NULL) {
		PUT_LOG_ERROR(cpassive_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,49, "Could not allocate memory");
		goto create_temp_service_out;
	}

	return_value = cpassive_service_list[CPASSIVE_SERVICE_NUMBER - 1];

create_temp_service_out:
	/*-------- DEBUG LOG --------*/
	if (cpassive_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cpassive_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cpassive_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cpassive_protomod, LOG_CAT_L7VSD_PROTOCOL,205,
		    "out_function: struct l7vs_cpassive_service* l7vs_protomod_cpassive_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Serialize struct l7vs_cpassive_service for debug log.
 * @param[out] buf   serialized string
 * @param[in]  cpassive l7vs_cpassive_service struct
 */
static void l7vs_cpassive_service_c_str(char* buf, struct l7vs_cpassive_service* cpassive) {
	if (cpassive == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "service_handle=%d, cookie_name=\"%s\", "
		    "forwarded_for=%d, reschedule=%d", cpassive->service_handle, cpassive->cookie_name,
		    cpassive->forwarded_for, cpassive->reschedule);
	}
}

/*!
 * Serialize struct l7vs_cpassive_service_arg for debug log.
 * @param[out] buf       serialized string
 * @param[in]  cpassive_arg l7vs_cpassive_service_arg struct
 */
void l7vs_cpassive_service_arg_c_str(char* buf, struct l7vs_cpassive_service_arg* cpassive_arg) {
	if (cpassive_arg == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "cookie_name=\"%s\", forwarded_for=%d, "
		    "reschedule=%d", cpassive_arg->cookie_name,
		    cpassive_arg->forwarded_for, cpassive_arg->reschedule);
	}
}
