/*
 * @file  protomod_ip.c
 * @brief protocol module of any protocol.
 * @brief this module never keep session persistence.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define __STDC_LIMIT_MACROS
#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define SERVICE_ARG_MAXSIZE    (512)
#define IP_SERVICE_NUMBER (128)
#define X_FORWARDED_FOR_LENGTH (48)

#define HASH_TABLE_BITS 8
#define HASH_TABLE_SIZE (1 << HASH_TABLE_BITS)
/* 2^31 + 2^29 - 2^25 + 2^22 - 2^19 - 2^16 + 1 */
#define GOLDEN_RATIO_PRIME 0x9e370001UL

struct l7vs_ip_service {
    handle_t service_handle;
    struct sockaddr_in dest[HASH_TABLE_SIZE];
    int forwarded_for;
    int reschedule;
};

struct  l7vs_ip_service_arg {
    int forwarded_for;
    int reschedule;
};

static void  fini(void);
static int   create(void*, handle_t);
static void* create_sa(struct l7vs_service_arg*);
static int   compare(handle_t, handle_t);
static int   match_cldata(struct l7vs_service*, struct l7vs_conn*,
        char*, size_t*, struct l7vs_dest**, int*);
static int   analyze_rsdata(struct l7vs_service*, struct l7vs_conn*,
        char*, size_t*);
static int   destroy(handle_t);
static void  destroy_sa(void**);
static int   service_arg(struct l7vs_service_arg_multi*, handle_t);
static int   parse(void*, int, char**);

static struct l7vs_ip_service *l7vs_protomod_ip_search_service(handle_t);
static struct l7vs_ip_service *l7vs_protomod_ip_create_service();
static struct l7vs_ip_service *l7vs_protomod_ip_create_temp_service();

static void  l7vs_ip_service_c_str(char*, struct l7vs_ip_service*);
static void  l7vs_ip_service_arg_c_str(char*, struct l7vs_ip_service_arg*);
unsigned int l7vs_ip_service_calc_hash(struct l7vs_conn*);

struct l7vs_ip_service *ip_service_list[IP_SERVICE_NUMBER];

static struct l7vs_protomod ip_protomod = {
    NULL,           /* handle */
    "ip",      /* modname */
    0,              /* refcnt */
    1,              /* fast schedule */
    create,         /* create function */
    compare,        /* compare function */
    match_cldata,   /* match_cldata function */
    analyze_rsdata, /* analyze_rsdata function */
    destroy,        /* destroy function */
    fini,           /* fini function */
    create_sa,      /* create_sa function */
    service_arg,    /* service_arg function */
    parse,          /* parse function */
    destroy_sa,     /* destroy_sa function */
    NULL,           /* initialize function */
    NULL,           /* finalize function */
    NULL,           /* get_log_level function */
    NULL,           /* put_log_debug function */
    NULL,           /* put_log_info function */
    NULL,           /* put_log_warn function */
    NULL,           /* put_log_error function */
    NULL            /* put_log_fatal function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param[in] handle dlopen's handle
 * @return l7vs_protomod struct
 */
extern "C" struct l7vs_protomod *
init(void *handle)
{
    struct l7vs_protomod* return_value = NULL;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,249,
                "in_function: struct l7vs_protomod* init(void* handle): handle=%p", handle);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (handle == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,217, "Arg(handle) is NULL pointer.");
        goto init_out;
    }

    /* initialize ip service list */
    memset(ip_service_list, 0, sizeof(struct l7vs_ip_service *) * IP_SERVICE_NUMBER);
    /* set dlopen's handle */
    ip_protomod.handle = handle;

    return_value = &ip_protomod;

init_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char protomod_str[DEBUG_STR_LEN] = {0};
        l7vs_protomod_c_str(protomod_str, &ip_protomod);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,250,
            "out_function: struct l7vs_protomod* init(void* handle): return=&(%s)", protomod_str);
    }
    /*------ DEBUG LOG END ------*/
    return return_value;
}

/*!
 * Protocol module finalize function. free all ip service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
    /* ip service list counter */
    int service_number = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,251, "in_function: void fini(void)");
    }
    /*------ DEBUG LOG END ------*/

    /* check all ip service list */
    for (service_number = 0; service_number < IP_SERVICE_NUMBER; ++service_number) {
        /* if pointer that does not point NULL exists ... */
        if (ip_service_list[service_number] != NULL) {
            /*-------- DEBUG LOG --------*/
            if (ip_protomod.get_log_level != NULL &&
                LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
                PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,58, "free: %p",
                    ip_service_list[service_number]);
            }
            /*------ DEBUG LOG END ------*/

            /* free and points NULL */
            free(ip_service_list[service_number]);
            ip_service_list[service_number] = NULL;
        }
    }

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,252, "out_function: void fini(void)");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * Create ip service struct.
 * @param[in] ip_arg    ip service argument struct
 * @param[in] service_handle a unique service ID
 * @retval 0  successfully create ip service.
 * @retval -1 some errors occur.
 */
static int
create(void *ip_arg, handle_t service_handle)
{
    struct l7vs_ip_service *ip_service;
    struct l7vs_ip_service_arg *ip_service_arg;
    int return_value = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_arg_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_arg_c_str(ip_arg_str, (struct l7vs_ip_service_arg*) ip_arg);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,253,
            "in_function: int create(void* ip_arg, handle_t service_handle):ip_arg=&(%s), "
            "service_handle=%d", ip_arg_str, service_handle);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (ip_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,218, "Arg(ip_arg) is NULL pointer.");
        return_value = -1;
        goto create_out;
    }


    if (service_handle != TEMP_SERVICEHANDLE) {
        /* search empty ip service list and create ip service */
        ip_service = l7vs_protomod_ip_create_service();
    } else {
        /* create temporary ip service */
        ip_service = l7vs_protomod_ip_create_temp_service();
    }

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ip_str, ip_service);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,254, "pointer assign: ip_service=&(%s)",
            ip_str);
    }
    /*------ DEBUG LOG END ------*/

    if (ip_service == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,219, "Could not make ip service.");
        return_value = -1;
        goto create_out;
    }

    ip_service_arg = (struct l7vs_ip_service_arg *) ip_arg;

    /* set service handle */
    ip_service->service_handle = service_handle;
    /* set x-forwarded-for flag */
    ip_service->forwarded_for = ip_service_arg->forwarded_for;
    /* set reschedule  */
    ip_service->reschedule = ip_service_arg->reschedule;

create_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,255,
            "out_function: int create(void* ip_arg, handle_t service_handle):return_value=%d",
            return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Create ip service argument struct.
 * @param[out] srv_arg service argument struct
 * @return ip service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
    struct l7vs_ip_service_arg *ip_service_arg;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char service_arg_str[DEBUG_STR_LEN] = {0};
        l7vs_service_arg_c_str(service_arg_str, srv_arg);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,256,
            "in_function: void* create_sa(struct l7vs_service_arg* srv_arg):srv_arg=&(%s)",
            service_arg_str);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (srv_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,220, "Arg(srv_arg) is NULL pointer.");
        ip_service_arg = NULL;
        goto create_sa_out;
    }

    /* create ip service argument struct */
    ip_service_arg = (struct l7vs_ip_service_arg *) calloc(1, sizeof(struct l7vs_ip_service_arg));

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,59, "calloc: addr=%p, size=%ld",
            ip_service_arg, (unsigned long int) sizeof(struct l7vs_ip_service_arg));
    }
    /*------ DEBUG LOG END ------*/

    if (ip_service_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,53, "Could not allocate memory.");
        goto create_sa_out;
    }

    /* set ip service argument size and protomod name "ip" */
    srv_arg->len = sizeof(struct l7vs_ip_service_arg);
    strcpy(srv_arg->protomod, ip_protomod.modname);

create_sa_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_service_arg_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_arg_c_str(ip_service_arg_str, ip_service_arg);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,257,
            "out_function: void* create_sa(struct l7vs_service_arg* srv_arg):return_value=&(%s)",
            ip_service_arg_str);
    }
    /*------ DEBUG LOG END ------*/

    return (void *) ip_service_arg;
}

/*!
 * Compare two service.
 * @param[in] srv_handle1 one of a unique service ID
 * @param[in] srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(handle_t srv_handle1, handle_t srv_handle2)
{
    int return_value = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,258,
            "in_function: int compare(handle_t srv_handle1, handle_t srv_handle2):"
            "srv_handle1=%u, srv_handle2=%u", srv_handle1, srv_handle2);
    }
    /*------ DEBUG LOG END ------*/

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,259,
            "out_function: int compare(handle_t srv_handle1, handle_t srv_handle2):return_value=%d",
            return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Do not check the request packet.
 * @param[in]  srv service struct include service handle, protocol module and schedule module.
 * @param[in]  conn connection data.
 * @param[in]  request packet data from client
 * @param[in]  len length of packet data
 * @param[out] dest destination (real server) list
 * @param[out] tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
    struct l7vs_ip_service *ip_service;
    struct l7vs_dest destination;
    int ret;
    int offset_length;
    char *x_forwarded_value;
    char *next_line = NULL;
    char x_forwarded_for_header[X_FORWARDED_FOR_LENGTH];
    size_t uri_len = 0;
    int return_value = 0;
    unsigned int hash;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char srv_str[DEBUG_STR_LEN] = {0};
        char conn_str[DEBUG_STR_LEN] = {0};
        char dest_str[DEBUG_STR_LEN] = {0};
        char len_str[DEBUG_STR_LEN] = {0};
        l7vs_service_c_str(srv_str, srv);
        l7vs_conn_c_str(conn_str, conn);
        if (dest != NULL) {
            l7vs_dest_c_str(dest_str, *dest);
        }
        else {
            strncpy(dest_str, "NULL", DEBUG_STR_LEN);
        }
        if (len != NULL) {
            snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
        }
        else {
            strncpy(len_str, "NULL", DEBUG_STR_LEN);
        }
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,260,
            "in_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
            "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):srv=&(%s), conn=&(%s), "
            "request=\"%s\", len=&(%s), dest=&(&(%s)), tcps=&(%d)",
            srv_str, conn_str, request, len_str, dest_str, *tcps);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (srv == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,221, "Arg(srv) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }
    if (srv->pm == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,222, "Arg(srv->pm) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }
    if (request == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,223, "Arg(request) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }
    if (len == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,224, "Arg(len) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }
    if (dest == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,225, "Arg(dest) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }
    if (tcps == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,226, "Arg(tcps) is NULL pointer.");
        return_value = -1;
        goto match_cldata_out;
    }

    /* search service that has such a service ID */
    ip_service = l7vs_protomod_ip_search_service(srv->handle);

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ip_str, ip_service);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,261, "pointer assign: ip_service=&(%s)",
            ip_str);
    }
    /*------ DEBUG LOG END ------*/

    if (ip_service == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,227, "Could not find such service handle's ip service.");
        return_value = -1;
        goto match_cldata_out;
    }

    /* initialize protocol module ... clear destination list */
    ret = srv->pm->initialize(srv, conn, request, *len, dest);
    if (ret != 0){
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,228, "Could not initialize protomod.");
        return_value = -1;
        goto match_cldata_out;
    }

    hash = l7vs_ip_service_calc_hash(conn);
    if (ip_service->dest[hash].sin_addr.s_addr &&
        ip_service->dest[hash].sin_port) {
        destination.addr.sin_addr.s_addr = ip_service->dest[hash].sin_addr.s_addr;
        destination.addr.sin_port = ip_service->dest[hash].sin_port;
	    *dest = &destination;
    }

    /* set X-Forwarded-For field */
    if (ip_service->forwarded_for) {
        uri_len = *len;
        /* check request */
        if (http_check_request_method(request, &uri_len) == NULL)
            goto match_cldata_finalize;

        x_forwarded_value = http_search_header_field(request, "X-Forwarded-For");

        /* already exists X-Forwarded-For field */
        if (x_forwarded_value) {
            next_line = http_skip_header_line(x_forwarded_value);
            /* backtrack to look up insert point */
            if (next_line) {
                next_line--; // *next_line == '\n'
                if (*(next_line - 1) == '\r') {
                    next_line--;
                }
                /* append client IP address */
                snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, ", %s", inet_ntoa(conn->caddr.sin_addr));
            }
        }

        /* not exists X-Forwarded-For field */
        if (!next_line) {
            /* construct new X-Forwarded-For header item */
            snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, "X-Forwarded-For: %s\r\n", inet_ntoa(conn->caddr.sin_addr));
    
            next_line = http_skip_header_line(request);
        }

        /* when insert point exist */
        if (next_line != NULL) {
            offset_length = (int) (next_line - request);
    
            /* insert X-Forwarded-For header field */
            http_insert_field(request, offset_length, x_forwarded_for_header, *len);
        
            /* add header length */
            *len += strlen(x_forwarded_for_header);
        }
    }

match_cldata_finalize:
    *tcps = 0;

	/* finalize */
    ret = srv->pm->finalize(srv, conn, request, *len, dest, ip_service->reschedule);

    if (ret != 0){
        PUT_LOG_INFO(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,17, "Could not finalize protomod. (Realserver decision failure)");
        return_value = -1;
        goto match_cldata_out;
    }

match_cldata_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,262,
            "out_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
            "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):return_value=%d",
            return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Do nothing.
 * @param[in] srv service struct include service handle, protocol module and schedule module.
 * @param[in] conn connection data.
 * @param[in] response packet data from real server
 * @param[in] len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
    char *response, size_t *len)
{
	struct l7vs_ip_service *ip_service;
    unsigned int hash;
    int return_value = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char srv_str[DEBUG_STR_LEN] = {0};
        char conn_str[DEBUG_STR_LEN] = {0};
        char len_str[DEBUG_STR_LEN] = {0};
        l7vs_service_c_str(srv_str, srv);
        l7vs_conn_c_str(conn_str, conn);
        if (len != NULL) {
            snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
        }
        else {
            strncpy(len_str, "NULL", DEBUG_STR_LEN);
        }
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,263,
            "in_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
            "char* response, size_t* len):srv=&(%s), conn=&(%s), response=\"%s\", len=&(%s)",
            srv_str, conn_str, response, len_str);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (srv == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,229, "Arg(srv) is NULL pointer.");
        return_value = -1;
        goto analyze_rsdata_out;
    }
    if (conn == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,230, "Arg(conn) is NULL pointer.");
        return_value = -1;
        goto analyze_rsdata_out;
    }
    if (conn->dest == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,231, "Arg(conn->dest) is NULL pointer.");
        return_value = -1;
        goto analyze_rsdata_out;
    }
    if (response == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,232, "Arg(response) is NULL pointer.");
        return_value = -1;
        goto analyze_rsdata_out;
    }
    if (len == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,233, "Arg(len) is NULL pointer.");
        return_value = -1;
        goto analyze_rsdata_out;
    }

    /* sorry flag check */
    if (conn->sorry_conn_flag == 1) {
        /*-------- DEBUG LOG --------*/
        if (ip_protomod.get_log_level != NULL &&
            LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
            PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,264, "Response from sorry server.");
        }
        /*------ DEBUG LOG END ------*/
        goto analyze_rsdata_out;
    }

	/* search service that has such a service ID */
	ip_service = l7vs_protomod_ip_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (ip_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ip_str[DEBUG_STR_LEN];
		l7vs_ip_service_c_str(ip_str, ip_service);
		PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,265, "pointer assign: ip_service=&(%s)",
		    ip_str);
	}
	/*------ DEBUG LOG END ------*/

	if (ip_service == NULL) {
		PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,234,
		    "Could not find such service handle's ip service.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

    hash = l7vs_ip_service_calc_hash(conn);
    memcpy(&ip_service->dest[hash], &conn->dest->addr, sizeof(struct sockaddr_in));

analyze_rsdata_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,266,
            "out_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
            "char* response, size_t* len):return_value=%d", return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Destroy ip service
 * @param[in] srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(handle_t srv_handle)
{
    /* ip service list counter */
    int service_number = 0;
    int free_flag = 0;
    int return_value = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,267,
            "in_function: int destroy(handle_t srv_handle):srv_handle=%u",
            srv_handle);
    }
    /*------ DEBUG LOG END ------*/

    /* check all ip service list */
    for (service_number = 0; service_number < IP_SERVICE_NUMBER; ++service_number) {
        /* found ip service that has srv_handle */
        if (ip_service_list[service_number] != NULL && 
            ip_service_list[service_number]->service_handle == srv_handle) {

            /* free and NULL */
            free(ip_service_list[service_number]);
            ip_service_list[service_number] = NULL;

            free_flag = 1;
            break;
        }
    }
    
    /* ip service was not found */
    if (free_flag == 0) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,235, "Could not find such service handle's ip service.");
        return_value = -1;
        goto destroy_out;
    }

destroy_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,268,
            "out_function: int destroy(handle_t srv_handle):return_value=%d",
            srv_handle);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Destroy ip service argument
 * @param[in] ip_arg ip service argument
 * @return void
 */
static void
destroy_sa(void **ip_arg)
{
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_arg_str[DEBUG_STR_LEN] = {0};
        if (ip_arg != NULL) {
            l7vs_ip_service_arg_c_str(ip_arg_str, (struct l7vs_ip_service_arg*) *ip_arg);
        }
        else {
            strncpy(ip_arg_str, "NULL", DEBUG_STR_LEN);
        }
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,269,
            "in_function: void destroy_sa(void** ip_arg):ip_arg=&(&(%s))",
            ip_arg_str);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (ip_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,236, "Arg(ip_arg) is NULL pointer.");
    }
    else if (*ip_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,237, "Arg(*ip_arg) is NULL pointer.");
    }
    else {
        /*-------- DEBUG LOG --------*/
        if (ip_protomod.get_log_level != NULL &&
            LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
            PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,60, "free: %p",
                *ip_arg);
        }
        /*------ DEBUG LOG END ------*/

        /* free and NULL */
        free((struct l7vs_ip_service_arg*) *ip_arg);
        *ip_arg = NULL;
    }

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,270,
            "out_function: void destroy_sa(void** ip_arg)");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * Create strings for service list of l7vsadm
 * @param[out] srv_arg service argument struct
 * @param[in]  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, handle_t srv_handle)
{
    struct l7vs_ip_service *ip_service;
    struct l7vs_ip_service_arg c_sarg;
    char ip_argument[SERVICE_ARG_MAXSIZE];
    int return_value = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char srv_arg_mt_str[DEBUG_STR_LEN] = {0};
        l7vs_service_arg_multi_c_str(srv_arg_mt_str, srv_arg_mt);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,271,
            "in_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
            "handle_t srv_handle):srv_arg_mt=&(%s), srv_handle=%u",
            srv_arg_mt_str, srv_handle);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (srv_arg_mt == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,238, "Arg(srv_arg_mt) is NULL pointer.");
        return_value = -1;
        goto service_arg_out;
    }

    /* search service that has such a service ID */
    ip_service = l7vs_protomod_ip_search_service(srv_handle);

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ip_str, ip_service);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,272, "pointer assign: ip_service=&(%s)",
            ip_str);
    }
    /*------ DEBUG LOG END ------*/

    if (ip_service == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,239, "Could not find such service handle's ip service.");
        return_value = -1;
        goto service_arg_out;
    }

    /* initialize argument strings */
    memset(ip_argument, 0, SERVICE_ARG_MAXSIZE);

    /* set ip args to service argument struct */
    srv_arg_mt->srv_arg.reschedule = ip_service->reschedule;

    /* create long argument (l7vsadm option -L/-l) */
    strncpy(srv_arg_mt->srv_arg.protomod_key_string, "", 256);

    /* create verbose argument (l7vsadm option -V/-v) */
    strncpy(srv_arg_mt->srv_arg.protomod_opt_string, "", 512);
    if (ip_service->forwarded_for) {
        strncpy(srv_arg_mt->srv_arg.protomod_opt_string, "--forwarded-for", 512);
    }

    c_sarg.reschedule = ip_service->reschedule;

    memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_ip_service_arg));

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ip_arg_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_arg_c_str(ip_arg_str, &c_sarg);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,273,
            "pointer assign: srv_arg_mt->protomod_arg=&(%s)", ip_arg_str);
    }
    /*------ DEBUG LOG END ------*/

service_arg_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,274,
            "out_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
            "handle_t srv_handle):return_value=%d", return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Parse l7vsadm options to ip argument
 * @param[out] ip_arg ip service argument struct
 * @param[in]  argc number of l7vsadm argument
 * @param[in]  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *ip_arg, int argc, char *argv[])
{
    struct l7vs_ip_service_arg *ip_service_arg;
    static struct option opt[] = {
        {"forwarded-for", no_argument,       NULL, 'F'},
        {"reschedule",    no_argument,       NULL, 'R'},
        {"no-reschedule", no_argument,       NULL, 'N'},
        {NULL,            0,                 NULL, 0  }
    };
    int c;
    int return_value = 0;
    int forwarded_for_flag = 0;
    int reschedule_flag = 0;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        int i;
        char argv_str[DEBUG_STR_LEN] = {0};
        char ip_arg_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_arg_c_str(ip_arg_str, (struct l7vs_ip_service_arg*) ip_arg);
        argv_str[0] = '\0';
        if (argv == NULL)
            snprintf(argv_str, DEBUG_STR_LEN, "NULL");
        else {
            for (i = 0; i < argc; i++) {
                snprintf(argv_str, DEBUG_STR_LEN, "%sargv[%d]=\"%s\", ", argv_str, i, argv[i]);
            }
            i = strnlen(argv_str, DEBUG_STR_LEN);
            if (i > 1)
                argv_str[i - 2] = '\0';
        }
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,275,
            "in_function: int parse(void* ip_arg, int argc, char* argv[]):ip_arg=&(%s), "
            "argc=%d, %s", ip_arg_str, argc, argv_str);
    }
    /*------ DEBUG LOG END ------*/

    /* check null */
    if (ip_arg == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,240, "Arg(ip_arg) is NULL pointer.");
        return_value = -1;
        goto parse_out;
    }
    if (argv == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,241, "Arg(argv) is NULL pointer.");
        return_value = -1;
        goto parse_out;
    }

    ip_service_arg = (struct l7vs_ip_service_arg *) ip_arg;
    optind = 0;

    /* check all argument */
    while ((c = getopt_long(argc, argv, "FRN", opt, NULL)) != -1) {
        switch (c) {
        /* --forwarded-for / -F */
        case 'F':
            /* x-forwarded-for on */
            ip_service_arg->forwarded_for = 1;
            forwarded_for_flag++;
            break;
        /* --reschedule / -R */
        case 'R':
            /* reschedule on */
            ip_service_arg->reschedule = 1;
            reschedule_flag++;
            break;

        /* --no-reschedule / -N */
        case 'N':
            /* reschedule off */
            ip_service_arg->reschedule = 0;
            reschedule_flag++;
            break;

        /* else error */
        default:
            PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,242, "Option error.");
            return_value = -1;
            goto parse_out;
        }
    }

    /* when set both -R and -N at the same time */
    if (reschedule_flag > 1) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,243,
            "You have to choose either of reschdule or no-reschedule.");
        return_value = -1;
        goto parse_out;
    }

    /* set default no reschedule */
    if (reschedule_flag == 0) {
        ip_service_arg->reschedule = 0;
    }

    /* set default no forwarded_for */
    if (forwarded_for_flag == 0) {
        ip_service_arg->forwarded_for = 0;
    }

parse_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,276,
            "out_function: int parse(void* ip_arg, int argc, char* argv[]):return_value=%d",
            return_value);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Search ip service from ip service list using service handle
 * @param[in] service_handle a unique service ID
 * @return ip service struct when service was found. NULL when service was not found.
 */
static struct l7vs_ip_service *
l7vs_protomod_ip_search_service(handle_t service_handle)
{
    /* ip service list counter */
    int service_number = 0;
    struct l7vs_ip_service* return_value = NULL;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,277,
            "in_function: struct l7vs_ip_service* l7vs_protomod_ip_search_service(handle_t service_handle):"
            "service_handle=%d", service_handle);
    }
    /*------ DEBUG LOG END ------*/

    /* check all ip service list */
    for (service_number = 0; service_number < IP_SERVICE_NUMBER; ++service_number) {
        /* found the service has same service handle */
        if (ip_service_list[service_number] != NULL && 
            ip_service_list[service_number]->service_handle == service_handle) {
            return ip_service_list[service_number];
        }
    }
    
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ssl_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ssl_str, return_value);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,278,
            "out_function: struct l7vs_ip_service* l7vs_protomod_ip_search_service(handle_t service_handle):"
            "return_value=&(%s)", ssl_str);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Create ip service.
 * @param void
 * @return ip service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_ip_service *
l7vs_protomod_ip_create_service()
{
    /* ip service list counter */
    int service_number = 0;
    struct l7vs_ip_service* return_value = NULL;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,279,
            "in_function: struct l7vs_ip_service* l7vs_protomod_ip_create_service()");
    }
    /*------ DEBUG LOG END ------*/

    /* check all ip service list */
    for (service_number = 0; service_number < IP_SERVICE_NUMBER - 1; ++service_number) {
        /* if pointer that does not point NULL exists ... */
        if (ip_service_list[service_number] == NULL) {
            /* create a service at empty pointer */
            ip_service_list[service_number] = (struct l7vs_ip_service *) calloc(1, sizeof(struct l7vs_ip_service));

            /*-------- DEBUG LOG --------*/
            if (ip_protomod.get_log_level != NULL &&
                LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
                PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,61, "calloc: addr=%p, size=%ld",
                    ip_service_list[service_number], (unsigned long int) sizeof(struct l7vs_ip_service));
            }
            /*------ DEBUG LOG END ------*/

            if (ip_service_list[service_number] == NULL) {
                PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,54, "Could not allocate memory.");
                goto create_service_out;
            }
            return_value = ip_service_list[service_number];
            goto create_service_out;
        }
    }
    
    /* all service list is full */
    PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,244, "ip service list is full.");

create_service_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ssl_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ssl_str, return_value);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,280,
            "out_function: struct l7vs_ip_service* l7vs_protomod_ip_create_service():"
            "return_value=&(%s)", ssl_str);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Create temporary ip service.
 * @param  void
 * @return ip service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_ip_service *
l7vs_protomod_ip_create_temp_service()
{
    struct l7vs_ip_service* return_value = NULL;

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,281,
            "in_function: struct l7vs_ip_service* l7vs_protomod_ip_create_temp_service()");
    }
    /*------ DEBUG LOG END ------*/

    /* if pointer that does not point NULL exists ... */
    if (ip_service_list[IP_SERVICE_NUMBER - 1] != NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,245, "Temporary ip service is being used by other process.");
        goto create_temp_service_out;
    }

    /* create temp service */
    ip_service_list[IP_SERVICE_NUMBER - 1] = (struct l7vs_ip_service *) calloc(1, sizeof(struct l7vs_ip_service));

    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,62, "calloc: addr=%p, size=%ld",
            ip_service_list[IP_SERVICE_NUMBER - 1], (unsigned long int) sizeof(struct l7vs_ip_service));
    }
    /*------ DEBUG LOG END ------*/

    if (ip_service_list[IP_SERVICE_NUMBER - 1] == NULL) {
        PUT_LOG_ERROR(ip_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,55, "Could not allocate memory");
        goto create_temp_service_out;
    }

    return_value = ip_service_list[IP_SERVICE_NUMBER - 1];

create_temp_service_out:
    /*-------- DEBUG LOG --------*/
    if (ip_protomod.get_log_level != NULL &&
        LOG_LV_DEBUG == ip_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
        char ssl_str[DEBUG_STR_LEN] = {0};
        l7vs_ip_service_c_str(ssl_str, return_value);
        PUT_LOG_DEBUG(ip_protomod, LOG_CAT_L7VSD_PROTOCOL,282,
            "out_function: struct l7vs_ip_service* l7vs_protomod_ip_create_service():"
            "return_value=&(%s)", ssl_str);
    }
    /*------ DEBUG LOG END ------*/

    return return_value;
}

/*!
 * Serialize struct l7vs_ip_service for debug log.
 * @param[out] buf   serialized string
 * @param[in]  ip l7vs_ip_service struct
 */
static void l7vs_ip_service_c_str(char* buf, struct l7vs_ip_service* ip) {
    if (ip == NULL) {
        snprintf(buf, DEBUG_STR_LEN, "NULL");
    }
    else {
        snprintf(buf, DEBUG_STR_LEN, "service_handle=%d, forwarded_for=%d, reschedule=%d",
            ip->service_handle, ip->forwarded_for, ip->reschedule);
    }
}

/*!
 * Serialize struct l7vs_ip_service_arg for debug log.
 * @param[out] buf       serialized string
 * @param[in]  ip_arg l7vs_ip_service_arg struct
 */
void l7vs_ip_service_arg_c_str(char* buf, struct l7vs_ip_service_arg* ip_arg) {
    if (ip_arg == NULL) {
        snprintf(buf, DEBUG_STR_LEN, "NULL");
    }
    else {
        snprintf(buf, DEBUG_STR_LEN, "forwarded_for=%d, reschedule=%d",
            ip_arg->forwarded_for, ip_arg->reschedule);
    }
}

unsigned int l7vs_ip_service_calc_hash(struct l7vs_conn* conn) {
    unsigned int hash = ntohl(conn->caddr.sin_addr.s_addr) * GOLDEN_RATIO_PRIME;
    return hash >> 32 - HASH_TABLE_BITS;
}
