﻿using System;
using System.Globalization;
using System.Xml.Linq;
using Codeplex.Data;
using LinqToTwitter;

namespace Azyobuzi.UserStreamEx
{
    /// <summary>
    /// <see cref="UserStream.Stopped"/>イベントのデータを提供します。
    /// </summary>
    public class StoppedEventArgs : EventArgs
    {
        /// <summary>
        /// 新しい<see cref="StoppedEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="reason">切断された理由</param>
        /// <param name="ex">発生したエラー。null可。</param>
        public StoppedEventArgs(StopReason reason, Exception ex)
        {
            this.Reason = reason;
            this.Exception = ex;
        }

        /// <summary>
        /// 切断された理由
        /// </summary>
        public StopReason Reason { private set; get; }

        /// <summary>
        /// 発生したエラー。<see cref="Reason"/>が<see cref="StopReason.CloseResponse"/>の場合はnull参照です。
        /// </summary>
        public Exception Exception { private set; get; }
    }

    /// <summary>
    /// 切断された理由
    /// </summary>
    public enum StopReason
    {
        /// <summary>
        /// Twitterが切断した
        /// </summary>
        CloseResponse,
        /// <summary>
        /// Stopメソッドが呼ばれた
        /// </summary>
        UserStop,
        /// <summary>
        /// 例外が発生した
        /// </summary>
        Error
    }

    /// <summary>
    /// Twitterから受信したJSONデータを提供します。
    /// </summary>
    public class ReceiveJsonEventArgs : EventArgs
    {
        /// <summary>
        /// <see cref="ReceiveJsonEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">1行のJSON形式のデータ</param>
        public ReceiveJsonEventArgs(string line)
        {
            this.Line = line;
        }

        /// <summary>
        /// 1行のJSON形式のデータ
        /// </summary>
        public string Line { private set; get; }
    }

    /// <summary>
    /// <see cref="UserStream.ReceiveFriends"/>イベントのデータを提供します。
    /// </summary>
    public class ReceiveFriendsEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="ReceiveFriendsEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">friendsリストが含まれたJSON形式のデータ</param>
        public ReceiveFriendsEventArgs(string line)
            : base(line)
        {
            FriendIds = (string[])DynamicJson.Parse(line).friends;
        }

        /// <summary>
        /// friends(following)のIDリスト
        /// </summary>
        public string[] FriendIds { private set; get; }
    }

    /// <summary>
    /// <see cref="UserStream.NewTweet"/>イベントのデータを提供します。
    /// </summary>
    public class NewTweetEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="NewTweetEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">ステータスが含まれたJSON形式のデータ</param>
        public NewTweetEventArgs(string line)
            : base(line)
        {
            Status = Status.CreateStatus(line.JsonToXml());
        }

        /// <summary>
        /// 新しいツイート
        /// </summary>
        public Status Status { private set; get; }
    }

    /// <summary>
    /// <see cref="UserStream.NewDirectMessage"/>イベントのデータを提供します。
    /// </summary>
    public class NewDirectMessageEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="NewDirectMessageEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">ダイレクトメッセージが含まれたJSON形式のデータ</param>
        public NewDirectMessageEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line).direct_message;
            DirectMessage = new DirectMessage()
            {
                ID = (ulong)json.id,
                SenderID = (ulong)json.sender_id,
                Text = json.text,
                RecipientID = (ulong)json.recipient_id,
                CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture),
                SenderScreenName = json.sender_screen_name,
                RecipientScreenName = json.recipient_screen_name,
                Sender = User.CreateUser(((string)json.sender.ToString()).JsonToXml()),
                Recipient = User.CreateUser(((string)json.recipient.ToString()).JsonToXml())
            };
        }

        /// <summary>
        /// 新しいダイレクトメッセージ
        /// </summary>
        public DirectMessage DirectMessage { private set; get; }
    }

    /// <summary>
    /// <see cref="UserStream.DeleteStatus"/>イベントのデータを提供します。
    /// </summary>
    public class DeleteStatusEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="DeleteStatusEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">削除されたステータスのIDが含まれたJSON形式のデータ</param>
        public DeleteStatusEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line).delete;
            if (!json.direct_message())
            {
                //ツイート
                StatusId = json.status.id_str;
                UserId = json.status.user_id_str;
                IsDirectMessage = false;
            }
            else
            {
                //DM
                StatusId = json.direct_message.id.ToString();
                UserId = json.direct_message.user_id.ToString();
                IsDirectMessage = true;
            }
        }

        /// <summary>
        /// 削除されたツイート（ダイレクトメッセージ）のID
        /// </summary>
        public string StatusId { private set; get; }

        /// <summary>
        /// 削除したユーザーのID（番号）
        /// </summary>
        public string UserId { private set; get; }

        /// <summary>
        /// ダイレクトメッセージか
        /// </summary>
        public bool IsDirectMessage { private set; get; }
    }

    /// <summary>
    /// <see cref="UserStream.ReceiveEvent"/>イベントのデータを提供します。
    /// </summary>
    public class ReceiveEventEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="ReceiveEventEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">イベント情報が含まれたJSON形式のデータ</param>
        public ReceiveEventEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line);
            EventTypes eventType = EventTypes.Unknown;
            Enum.TryParse(((string)json.@event).Replace("_", ""), true, out eventType);
            EventType = eventType;
            CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture).ToLocalTime();
            Source = User.CreateUser(((string)json.source.ToString()).JsonToXml());
            Target = User.CreateUser(((string)json.target.ToString()).JsonToXml());
            if (json.target_object())
            {
                if (json.target_object.mode())
                {
                    //リスト
                    TargetList = List.CreateList(((string)json.target_object.ToString()).JsonToXml(), new XElement("lists_list"));
                }
                else
                {
                    //ツイート
                    TargetStatus = Status.CreateStatus(((string)json.target_object.ToString()).JsonToXml());
                }
            }
        }

        /// <summary>
        /// イベントの種類
        /// </summary>
        public EventTypes EventType { private set; get; }
        /// <summary>
        /// イベントが発生した日時
        /// </summary>
        public DateTime CreatedAt { private set; get; }
        /// <summary>
        /// イベントを発生させたユーザー
        /// </summary>
        public User Source { private set; get; }
        /// <summary>
        /// イベントの対象のユーザー
        /// </summary>
        public User Target { private set; get; }
        /// <summary>
        /// イベントの対象のツイート
        /// </summary>
        public Status TargetStatus { private set; get; }
        /// <summary>
        /// イベントの対象のリスト
        /// </summary>
        public List TargetList { private set; get; }
    }

    /// <summary>
    /// イベントの種類
    /// </summary>
    public enum EventTypes
    {
        /// <summary>
        /// 不明
        /// </summary>
        Unknown,
        /// <summary>
        /// お気に入りに追加した
        /// </summary>
        Favorite,
        /// <summary>
        /// お気に入りから削除した
        /// </summary>
        Unfavorite,
        /// <summary>
        /// フォローされた
        /// </summary>
        Follow,
        /// <summary>
        /// リストに入れられた
        /// </summary>
        ListMemberAdded,
        /// <summary>
        /// リストから外された
        /// </summary>
        ListMemberRemoved,
        /// <summary>
        /// ブロックに成功した
        /// </summary>
        Block,
        /// <summary>
        /// ブロックの解除に成功した
        /// </summary>
        Unblock,
        /// <summary>
        /// プロフィールを変更した
        /// </summary>
        UserUpdate,
        /// <summary>
        /// リストを作成した
        /// </summary>
        ListCreated,
        /// <summary>
        /// リストを削除した
        /// </summary>
        ListDestroyed
    }

    /// <summary>
    /// <see cref="UserStream.TrackLimit"/>イベントのデータを提供します。
    /// </summary>
    public class TrackLimitEventArgs : ReceiveJsonEventArgs
    {
        /// <summary>
        /// JSONを指定して<see cref="TrackLimitEventArgs"/>クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="line">JSON形式のデータ</param>
        public TrackLimitEventArgs(string line)
            : base(line)
        {
            Track = (ulong)DynamicJson.Parse(line).limit.track;
        }

        /// <summary>
        /// マッチしたツイートのインデックス
        /// </summary>
        public ulong Track { private set; get; }
    }
}
