/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <say@users.sourceforge.jp>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.GridView;
import android.widget.ImageView;
import android.widget.SimpleAdapter;
import android.widget.TextView;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.routine_work.simple_battery_logger.service.ServiceUtils;
import org.routine_work.simple_battery_logger.util.Log;

/**
 *
 * @author Masahiko, SAWAI <say@users.sourceforge.jp>
 */
public class DashboardActivity extends Activity implements OnItemClickListener
{
	private static final String LOG_TAG = "simple-battery-logger";
	// adater mapping
	private static final String MAP_KEY_ICON = "icon";
	private static final String MAP_KEY_LABEL = "label";
	private static final String MAP_KEY_CLASSNAME = "className";
	private static final String[] MAPPING_FROM =
	{
		MAP_KEY_ICON,
		MAP_KEY_LABEL,
		MAP_KEY_CLASSNAME,
	};
	private static final int[] MAPPING_TO =
	{
		R.id.icon_imageview,
		R.id.label_textview,
		R.id.classname_textview,
	};
	private List<Map<String, String>> listData;
	private GridView gridView;
	private BatteryChangeReceiver batteryChangeReceiver;

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "Hello");

		super.onCreate(savedInstanceState);
		setContentView(R.layout.dashboard_activity);

		batteryChangeReceiver = new BatteryChangeReceiver();

		// init views
		gridView = (GridView) findViewById(R.id.dashboard_gridview);
		gridView.setOnItemClickListener(this);

		// init grid or list
		initGridView();

		ServiceUtils.initializeLoggingService(this);
		ServiceUtils.initializeDeleteOldDataTask(this);
		ServiceUtils.initializeExportCsvFileTask(this);

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onPause()
	{
		Log.v(LOG_TAG, "Hello");

		unregisterReceiver(batteryChangeReceiver);
		Log.d(LOG_TAG, "unregisterReceiver BatteryChangeReceiver");
		super.onPause();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onResume()
	{
		Log.v(LOG_TAG, "Hello");
		super.onResume();

		IntentFilter filter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
		registerReceiver(batteryChangeReceiver, filter);
		Log.d(LOG_TAG, "registerReceiver BatteryChangeReceiver");

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		Log.v(LOG_TAG, "Hello");

		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		Log.v(LOG_TAG, "Bye");
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = false;
		Log.v(LOG_TAG, "Hello");

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.preference_menuitem:
				Intent intent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivityForResult(intent, AppConstants.REQUEST_CODE_CHILD_ACTIVITY);
				result = true;
				break;
			case R.id.quit_menuitem:
				finish();
				result = true;
				break;
			default:
				result = super.onOptionsItemSelected(item);
				break;
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data)
	{
		Log.v(LOG_TAG, "Hello");
		switch (requestCode)
		{
			case AppConstants.REQUEST_CODE_CHILD_ACTIVITY:
				if (resultCode == AppConstants.RESULT_QUIT)
				{
					setResult(AppConstants.RESULT_QUIT);
					finish();
				}
			default:
				break;
		}
		Log.v(LOG_TAG, "Hello");
	}

	public void onItemClick(AdapterView<?> parentView, View view, int position, long id)
	{
		Log.v(LOG_TAG, "Hello");

		TextView classnameTextView = (TextView) view.findViewById(R.id.classname_textview);
		String classname = classnameTextView.getText().toString();
		try
		{
			Class<?> clazz = Class.forName(classname);
			Intent intent = new Intent(this, clazz);
			startActivityForResult(intent, AppConstants.REQUEST_CODE_CHILD_ACTIVITY);
		}
		catch (ClassNotFoundException ex)
		{
			Log.e(LOG_TAG, "Activity class not found.", ex);
		}

		Log.v(LOG_TAG, "Bye");
	}

	protected List<Map<String, String>> getListData()
	{
		if (listData == null)
		{
			HashMap<String, String> data;
			listData = new ArrayList<Map<String, String>>();

			data = new HashMap<String, String>();
			data.put(MAP_KEY_ICON, String.valueOf(android.R.drawable.ic_menu_info_details));
			data.put(MAP_KEY_LABEL, getString(R.string.battery_status));
			data.put(MAP_KEY_CLASSNAME, BatteryStatusActivity.class.getName());
			listData.add(data);

			data = new HashMap<String, String>();
			data.put(MAP_KEY_ICON, String.valueOf(android.R.drawable.ic_menu_preferences));
			data.put(MAP_KEY_LABEL, getString(R.string.preferences));
			data.put(MAP_KEY_CLASSNAME, BatteryLoggerPreferenceActivity.class.getName());
			listData.add(data);

			data = new HashMap<String, String>();
			data.put(MAP_KEY_ICON, String.valueOf(android.R.drawable.ic_menu_agenda));
			data.put(MAP_KEY_LABEL, getString(R.string.history_view));
//			data.put(MAP_KEY_CLASSNAME, BatteryHistoryTabActivity.class.getName());
			data.put(MAP_KEY_CLASSNAME, DatabaseDateListActivity.class.getName());
			listData.add(data);

			data = new HashMap<String, String>();
			data.put(MAP_KEY_ICON, String.valueOf(android.R.drawable.ic_menu_agenda));
			data.put(MAP_KEY_LABEL, getString(R.string.csv_files));
			data.put(MAP_KEY_CLASSNAME, CsvFileListActivity.class.getName());
			listData.add(data);

		}
		return listData;
	}

	private void initGridView()
	{
		SimpleAdapter adapter = new SimpleAdapter(this,
			getListData(), R.layout.dashboard_grid_item, MAPPING_FROM, MAPPING_TO);

		gridView.setAdapter(adapter);
	}

	class BatteryChangeReceiver extends BroadcastReceiver
	{
		@Override
		public void onReceive(Context context, Intent intent)
		{
			// update icon
			int statusIconId = intent.getIntExtra(BatteryConstants.EXTRA_ICON_SMALL, 0);
			ImageView statusIconImageView = (ImageView) findViewById(R.id.status_icon_imageview);
			statusIconImageView.setImageResource(statusIconId);

			// update level value
			int level = intent.getIntExtra(BatteryConstants.EXTRA_LEVEL, 0);
			int scale = intent.getIntExtra(BatteryConstants.EXTRA_SCALE, 100);

			String levelPercentText = String.format("%d", level * 100 / scale);
			TextView levelPercentTextView = (TextView) findViewById(R.id.level_percent_textview);
			levelPercentTextView.setText(levelPercentText);

		}
	}
}
