/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <say@users.sourceforge.jp>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import org.routine_work.simple_battery_logger.util.Log;

/**
 *
 * @author sawai
 */
public class BatteryHistoryDBHelper extends SQLiteOpenHelper
	implements DBConstants
{

	public static final String LOG_TAG = "simple-battery-logger";
	private SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
	private SimpleDateFormat datetimeFormat = new SimpleDateFormat(DATETIME_FORMAT);

	public BatteryHistoryDBHelper(Context context)
	{
		super(context, DATABASE_NAME, null, DATABASE_VERSION);
	}

	@Override
	public void onCreate(SQLiteDatabase db)
	{
		Log.d(LOG_TAG, "CREATE_TABLE_SQL  : " + BatteryHistory.CREATE_TABLE_SQL);
		Log.d(LOG_TAG, "CREATE_DATETIME_INDEX_SQL  : " + BatteryHistory.CREATE_DATETIME_INDEX_SQL);
		Log.d(LOG_TAG, "CREATE_VIEW_SQL  : " + BatteryHistoryDate.CREATE_VIEW_SQL);
		db.execSQL(BatteryHistory.CREATE_TABLE_SQL);
		db.execSQL(BatteryHistory.CREATE_DATETIME_INDEX_SQL);
		db.execSQL(BatteryHistoryDate.CREATE_VIEW_SQL);
	}

	@Override
	public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion)
	{
		Log.d(LOG_TAG, "DROP_VIEW_SQL  : " + BatteryHistoryDate.DROP_VIEW_SQL);
		Log.d(LOG_TAG, "DROP_TABLE_SQL  : " + BatteryHistory.DROP_TABLE_SQL);
		db.execSQL(BatteryHistoryDate.DROP_VIEW_SQL);
		db.execSQL(BatteryHistory.DROP_TABLE_SQL);
		onCreate(db);
	}

	public long insertBatteryData(SQLiteDatabase db,
		int status, int health, int level, int scale,
		int plugged, int voltage, int temperature)
	{
		String nowDateTime = datetimeFormat.format(new Date());
		long rowID = insertBatteryData(db,
			nowDateTime,
			status, health, level, scale,
			plugged, voltage, temperature);
		return rowID;
	}

	public long insertBatteryData(SQLiteDatabase db,
		Date dateTime,
		int status, int health, int level, int scale,
		int plugged, int voltage, int temperature)
	{
		String nowDateTime = datetimeFormat.format(dateTime);
		long rowID = insertBatteryData(db,
			nowDateTime,
			status, health, level, scale,
			plugged, voltage, temperature);
		return rowID;
	}

	public long insertBatteryData(SQLiteDatabase db,
		String dateTime,
		int status, int health, int level, int scale,
		int plugged, int voltage, int temperature)
	{
		ContentValues values = new ContentValues();
		values.put(BatteryHistory.Columns.DATETIME, dateTime);
		values.put(BatteryHistory.Columns.STATUS, status);
		values.put(BatteryHistory.Columns.HEALTH, health);
		values.put(BatteryHistory.Columns.LEVEL, level);
		values.put(BatteryHistory.Columns.SCALE, scale);
		values.put(BatteryHistory.Columns.PLUGGED, plugged);
		values.put(BatteryHistory.Columns.VOLTAGE, voltage);
		values.put(BatteryHistory.Columns.TEMPERATURE, temperature);
		long rowID = db.insert(BatteryHistory.TABLE_NAME, null, values);
		return rowID;
	}

	public Cursor getAllBatteryHistoryCursor(SQLiteDatabase db)
	{
		Cursor cursor = db.query(BatteryHistory.TABLE_NAME,
			new String[]
			{
				BatteryHistory.Columns.ID,
				BatteryHistory.Columns.DATETIME,
				BatteryHistory.Columns.STATUS,
				BatteryHistory.Columns.HEALTH,
				BatteryHistory.Columns.LEVEL,
				BatteryHistory.Columns.SCALE,
				BatteryHistory.Columns.PLUGGED,
				BatteryHistory.Columns.VOLTAGE,
				BatteryHistory.Columns.TEMPERATURE,
			},
			null, null, null, null, BatteryHistory.Columns.DATETIME + " " + SORT_DESCENDING);

		return cursor;
	}

	public Cursor getOneDayBatteryHistoryCursor(SQLiteDatabase db, String dateString, String sortOrder) throws ParseException
	{
		Date date = dateFormat.parse(dateString);
		Cursor result = getOneDayBatteryHistoryCursor(db, date, sortOrder);
		return result;
	}

	public Cursor getOneDayBatteryHistoryCursor(SQLiteDatabase db, Date date, String sortOrder)
	{
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		long startTime = cal.getTimeInMillis();

		cal.add(Calendar.DAY_OF_MONTH, 1);
		cal.add(Calendar.MILLISECOND, -1);
		long endTime = cal.getTimeInMillis();

		return getBatteryHistoryCursorByDate(db, startTime, endTime, sortOrder);
	}

	/**
	 * Get battery history data between startTimeInMilis and endTimeInMilis.
	 * @param db
	 * @param startTimeInMilis 
	 * @param endTimeInMilis 
	 * @param sortOrder  - sort order by date, "ASC" or "DESC"
	 * @return db cursor
	 */
	public Cursor getBatteryHistoryCursorByDate(SQLiteDatabase db, long startTimeInMilis, long endTimeInMilis, String sortOrder)
	{
		String startDate = datetimeFormat.format(new Date(startTimeInMilis));
		String endDate = datetimeFormat.format(new Date(endTimeInMilis));

		Cursor cursor = db.query(BatteryHistory.TABLE_NAME,
			new String[]
			{
				BatteryHistory.Columns.ID,
				BatteryHistory.Columns.DATETIME,
				BatteryHistory.Columns.STATUS,
				BatteryHistory.Columns.HEALTH,
				BatteryHistory.Columns.LEVEL,
				BatteryHistory.Columns.SCALE,
				BatteryHistory.Columns.PLUGGED,
				BatteryHistory.Columns.VOLTAGE,
				BatteryHistory.Columns.TEMPERATURE,
			},
			BatteryHistory.Columns.DATETIME + " BETWEEN ? AND ? ", new String[]
			{
				startDate, endDate,
			}, null, null, BatteryHistory.Columns.DATETIME + " " + sortOrder);

		return cursor;
	}

	public Cursor getBatteryHistoryDateListCursor(SQLiteDatabase db)
	{
		Cursor cursor = db.query(BatteryHistoryDate.TABLE_NAME,
			new String[]
			{
				BatteryHistoryDate.Columns.ID,
				BatteryHistoryDate.Columns.DATE,
			},
			null, null, null, null, BatteryHistoryDate.Columns.DATE + " " + SORT_DESCENDING);

		return cursor;
	}

	public List<String> getBatteryHistoryDateStringList(SQLiteDatabase db)
	{
		Log.v(LOG_TAG, "Hello");
		List<String> dateList = new ArrayList<String>();

		Cursor cursor = getBatteryHistoryDateListCursor(db);
		if (cursor.moveToFirst())
		{
			int dateIndex = cursor.getColumnIndex(BatteryHistoryDate.Columns.DATE);
			do
			{
				String dateString = cursor.getString(dateIndex);
				dateList.add(dateString);
			}
			while (cursor.moveToNext());
		}

		Log.v(LOG_TAG, "Bye");
		return dateList;
	}

	public void deleteAll(SQLiteDatabase db)
	{
		db.delete(BatteryHistory.TABLE_NAME, null, null);
	}

	/**
	 * 
	 * @param db
	 * @param days 
	 */
	public void deleteOldData(SQLiteDatabase db, int days)
	{
		if (days > 0)
		{
			Calendar cal = Calendar.getInstance();
			Log.d(LOG_TAG, "nowTimeString  : " + datetimeFormat.format(cal.getTime()));
			cal.add(Calendar.DAY_OF_MONTH, -days);
			cal.set(Calendar.HOUR_OF_DAY, 0);
			cal.set(Calendar.MINUTE, 0);
			cal.set(Calendar.SECOND, 0);
			cal.set(Calendar.MILLISECOND, 0);
			String dateTimeString = datetimeFormat.format(cal.getTime());
			Log.d(LOG_TAG, "dateTimeString : " + dateTimeString);

			db.delete(BatteryHistory.TABLE_NAME,
				BatteryHistory.Columns.DATETIME + " < ?",
				new String[]
				{
					dateTimeString,
				});
		}
	}

	public void reindex(SQLiteDatabase db)
	{
		db.execSQL(BatteryHistory.REINDEX_SQL);
	}

	public void vaccum(SQLiteDatabase db)
	{
		db.execSQL(BatteryHistory.VACCUM_SQL);
	}
}
