/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.service;

import android.content.res.Resources;
import org.routine_work.simple_battery_logger.db.BatteryHistoryDBHelper;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.database.sqlite.SQLiteDatabase;
import android.os.BatteryManager;
import android.os.IBinder;
import android.os.SystemClock;
import org.routine_work.simple_battery_logger.R;
import org.routine_work.simple_battery_logger.util.Log;
import org.routine_work.simple_battery_logger.util.PreferenceUtils;

/**
 *
 * @author sawai
 */
public class BatteryLoggingService extends Service
{
	public static final String PREFERENCE_KEY_ROW_ID = "ROW_ID";
	public static final long INTERVAL_TIME_MSEC = 10 * 60 * 1000; //  10 min
	private static final String LOG_TAG = "simple-battery-logger";
	private boolean running;
	private BatteryHistoryDBHelper dbHelper;
	private SQLiteDatabase db;
	private BatteryChangeReceiver batteryChangeReceiver;
	private PendingIntent intervalPendingIntent;
	private boolean ignoreVoltageChangeOnly;

	@Override
	public void onCreate()
	{
		Log.v(LOG_TAG, "Hello");
		super.onCreate();

		running = false;
		dbHelper = new BatteryHistoryDBHelper(this);
		batteryChangeReceiver = new BatteryChangeReceiver();

		// init intervalPendingIntent
		Intent serviceIntent = new Intent(this, BatteryLoggingService.class);
		intervalPendingIntent = PendingIntent.getService(this, 0, serviceIntent, 0);


		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onStart(Intent intent, int startId)
	{
		Log.v(LOG_TAG, "Hello");
		super.onStart(intent, startId);

		boolean loggingServiceEnabled = PreferenceUtils.isLoggingServiceEnabled(this);
		Log.i(LOG_TAG, "loggingServiceEnabled : " + loggingServiceEnabled);

		ignoreVoltageChangeOnly = PreferenceUtils.isIgnoreVoltageChangeOnly(this);
		Log.i(LOG_TAG, "ignoreVoltageChangeOnly : " + ignoreVoltageChangeOnly);

		if (loggingServiceEnabled)
		{
			initializeService();
		}
		else
		{
			Log.d(LOG_TAG, "BatteryLoggingService#onStart() stopSelf");
			stopSelf();
		}

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onDestroy()
	{
		Log.v(LOG_TAG, "Hello");

		finalizeService();

		super.onDestroy();
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public IBinder onBind(Intent arg0)
	{
		return null;
	}

	private synchronized void initializeService()
	{
		Log.v(LOG_TAG, "Hello");

		if (!running)
		{
			Log.i(LOG_TAG, "initialize battery logging service");

			running = true;

			// initialize database
			db = dbHelper.getWritableDatabase();

			// enable receiver
			IntentFilter filter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
			registerReceiver(batteryChangeReceiver, filter);

			// set alarm
			AlarmManager alarmManager = (AlarmManager) getSystemService(ALARM_SERVICE);
			alarmManager.setRepeating(AlarmManager.ELAPSED_REALTIME_WAKEUP,
				SystemClock.elapsedRealtime() + INTERVAL_TIME_MSEC, INTERVAL_TIME_MSEC,
				intervalPendingIntent);
		}

		Log.v(LOG_TAG, "BatteryLoggingService#initializeService() Bye");
	}

	private synchronized void finalizeService()
	{
		Log.v(LOG_TAG, "Hello");

		if (running)
		{
			Log.i(LOG_TAG, "finalize battery logging service");
			// cancel alarm
			AlarmManager alarmManager = (AlarmManager) getSystemService(ALARM_SERVICE);
			alarmManager.cancel(intervalPendingIntent);

			// disable receiver
			unregisterReceiver(batteryChangeReceiver);

			// finalize database
			db.close();
			db = null;
			running = false;
		}

		Log.v(LOG_TAG, "Bye");
	}

	class BatteryChangeReceiver extends BroadcastReceiver
	{
		private int previousStatus = -1;
		private int previousHealth = -1;
		private int previousLevel = -1;
		private int previousScale = -1;
		private int previousPlugged = -1;
		private int previousVoltage = -1;
		private int previousTemperature = -1;

		@Override
		public void onReceive(Context context, Intent intent)
		{
			Log.v(LOG_TAG, "Hello");
			int status = intent.getIntExtra("status", BatteryManager.BATTERY_STATUS_UNKNOWN);
			int health = intent.getIntExtra("health", BatteryManager.BATTERY_HEALTH_UNKNOWN);
			int level = intent.getIntExtra("level", 0);
			int scale = intent.getIntExtra("scale", 100);
			int plugged = intent.getIntExtra("plugged", 0);
			int voltage = intent.getIntExtra("voltage", 0);
			int temperature = intent.getIntExtra("temperature", 0);

			Log.d(LOG_TAG,
				String.format(" battery status changed : (%d, %d, %d, %d, %d, %d, %d)",
				status, health, level, scale, plugged, voltage, temperature));
			Log.v(LOG_TAG,
				String.format(" battery status previous : (%d, %d, %d, %d, %d, %d, %d)",
				previousStatus, previousHealth, previousLevel, previousScale,
				previousPlugged, previousVoltage, previousTemperature));

			//
			if (isIgnoreData(status, health, level, scale, plugged, voltage, temperature) == false)
			{
				// insert to DB
				long rowID = dbHelper.insertBatteryData(db, status, health, level, scale, plugged, voltage, temperature);

				Log.d(LOG_TAG, "Inserted rowID => " + rowID);
				// save data to prefs
				String packageName = getApplicationContext().getPackageName();
				SharedPreferences prefs = getApplicationContext().getSharedPreferences(packageName, MODE_PRIVATE);
				Editor editor = prefs.edit();
				editor.putLong(PREFERENCE_KEY_ROW_ID, rowID);
				editor.commit();
			}


			Log.v(LOG_TAG, "Bye");
		}

		private boolean isIgnoreData(int status, int health, int level,
			int scale, int plugged, int voltage, int temperature)
		{
			return ignoreVoltageChangeOnly && isVoltageChangeOnly(status, health, level, scale, plugged, voltage, temperature);
		}

		private boolean isVoltageChangeOnly(int status, int health, int level,
			int scale, int plugged, int voltage, int temperature)
		{
			boolean result = true;

			if ((previousStatus != status)
				|| (previousHealth != health)
				|| (previousLevel != level)
				|| (previousScale != scale)
				|| (previousPlugged != plugged)
				|| (previousTemperature != temperature))
			{
				result = false;
			}

			previousStatus = status;
			previousHealth = health;
			previousLevel = level;
			previousScale = scale;
			previousPlugged = plugged;
			previousVoltage = voltage;
			previousTemperature = temperature;

			return result;
		}
	}
}
