/*
 * IPRPC - Inter Process Remote Procedure Call
 *
 * Copyright (C) 2012-2013 by Hiroyuki KAJIURA. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 *     1:Redistributions of source code must retain the above copyright notice, 
 *       this list of conditions and the following disclaimer.
 *     2:Redistributions in binary form must reproduce the above copyright notice, 
 *       this list of conditions and the following disclaimer in the documentation 
 *       and/or other materials provided with the distribution.
 *     3:Neither the name of copyright owner nor the names of its contributors 
 *       may be used to endorse or promote products derived from this software 
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#define	IPC_PACK_MAIN

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include	"ipcInfo.h"
#include	"ipcLog.h"
#include	"ipcType.h"
#include	"ipcSyscall.h"
#include	"rpcMsg.h"
#include	"ipcShm.h"
#include	"ipcShmDB.h"
#include	"ipcQue.h"
#include	"ipcStub.h"
#include	"ipcMarshal.h"

static int32_t IpcMarshalDataSharedMemId = -1;

static RpcResult_t IPC_parseOneData(RpcParamData_t *inData,uint32_t *useSize);
static RpcResult_t IPC_putOneData(RpcParamData_t *inData,void *outData,uint32_t outSize,uint32_t *useSize);
static RpcResult_t IPC_parseMarshaledData(void *inData,uint32_t inSize,uint32_t *needSize,RpcParamData_t *pData);

void IPC_DumpParamData(RpcParamData_t *pData);
uint8_t *IPC_ConvertTypeToString(RpcParamType_t type);

/*--- MARSHAL LIBRARY ---*/

RpcResult_t IPC_InitMarshal(RpcBool_t masterProcess) {
	RpcResult_t ret = RPC_SUCCESS;

	if(IpcMarshalDataSharedMemId == -1) {
		if((ret = IPC_GetShmInfo(IPC_MARSHAL_DATA_SHARED_MEM_NAME, &IpcMarshalDataSharedMemId,(void**)NULL, (uint32_t*)NULL)) != RPC_SUCCESS) {
			return ret;
		}
		if(masterProcess) {
			if((ret = IPC_InitShmGapArea(IpcMarshalDataSharedMemId,IPC_SHM_MARSHAL_GAP_SIZE)) != RPC_SUCCESS) {
				(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN INIT SHM GAP AREA ret:%s;",RPC_LogConvertResultCode(ret));
				return ret;		
			}
			if((ret = IPC_InitShmAllocArea(IpcMarshalDataSharedMemId, IPC_MARSHAL_SHARED_MEM_REAL_SIZE, IPC_SHM_MARSHAL_NUM_OF_DEV, IPC_SHM_MARSHAL_MIN_SIZE, IPC_SHM_MARSHAL_RATE_OF_MAGNIFY)) != RPC_SUCCESS) {
				(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN INIT SHM ALLOC AREA ret:%s;",RPC_LogConvertResultCode(ret));
				return ret;		
			}
		}
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "SHM MEM ID:0x%x;",IpcMarshalDataSharedMemId);
	return ret;
}

RpcResult_t IPC_Marshal(RpcParamData_t *inData, void **outData, uint32_t *outSize) {
	RpcParamData_t *p;uint32_t useSize;RpcResult_t ret;
	uint32_t oneSize,tmpSize;void *outP;RpcParamHeader_t *header;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG1,"IN IPC_Marshal inData:0x%x; outData:0x%x; outSize:0x%x",inData,outData,outSize);
	if(IpcMarshalDataSharedMemId == -1) {
		*outSize = 0;*outData = NULL;
		return RPC_FATAL_ERROR;
	}
	if((inData == NULL)||(outData == NULL)||(outSize == NULL)) {
		return RPC_PARAM_ERROR;
	}
	*outSize = 0;
	for(p = inData;p->paramSpec.paramType != RPC_TYPE_UNKNOWN;p++) {
		if((ret = IPC_parseOneData(p,&oneSize)) != RPC_SUCCESS) {
			*outSize = 0;*outData = NULL;
			return ret;
		}
		*outSize += oneSize;
	}
	*outSize += (sizeof(RpcParamSpec_t)+sizeof(RpcParamHeader_t));
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG2," NOW 2nd Stage, outSize:%d;",*outSize);
	if((*outData = outP = IPC_AllocShmMem(IpcMarshalDataSharedMemId, *outSize)) == NULL) {
		*outSize = 0;*outData = NULL;
		return RPC_NO_MORE_RESOURCE;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG2," NOW 3rd Stage, outData:0x%x;",*outData);
	header = (RpcParamHeader_t*)outP;
	header->paramLength = *outSize;
	header->reserved = 0;
	tmpSize = *outSize - sizeof(RpcParamHeader_t);
	outP = (void*)&(((uint8_t*)outP)[sizeof(RpcParamHeader_t)]);
	for(p = inData;p->paramSpec.paramType != RPC_TYPE_UNKNOWN;p++) {
		if((ret = IPC_putOneData(p,outP,tmpSize,&useSize)) != RPC_SUCCESS) {
			IPC_FreeShmMem(IpcMarshalDataSharedMemId, *outData);
			*outSize = 0;*outData = NULL;
			return ret;
		}
		tmpSize -= useSize;
		outP = (void*)&(((uint8_t*)outP)[useSize]);
	}
	if((ret = IPC_putOneData(p,outP,tmpSize,&useSize)) != RPC_SUCCESS) {
		IPC_FreeShmMem(IpcMarshalDataSharedMemId, *outData);
		*outSize = 0;*outData = NULL;
		return ret;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_Marshal outData:0x%x;outSize:%d;",*outData,*outSize);
	return RPC_SUCCESS;
}

RpcResult_t IPC_Unmarshal(void *inData, uint32_t inSize, void *outData, uint32_t outSize) {
	RpcParamData_t *pData;void *inP,*outP;int32_t size,needSize,oSize;
	RpcResult_t ret;RpcParamHeader_t *header;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG1,"IN IPC_Unmarshal inData:0x%x; inSize:%d; outData:0x%x; outSize:%d;",
//		inData, inSize, outData, outSize);
	if(IpcMarshalDataSharedMemId == -1) {
		return RPC_FATAL_ERROR;
	}
	if((inData == NULL)||(inSize == 0)||(outData == NULL)||(outSize == 0)) {
		return RPC_PARAM_ERROR;
	}
//	inP = inData; size = inSize;
	header = (RpcParamHeader_t*)inData;
	if(header->paramLength != inSize) {
		return RPC_PARAM_ERROR;
	}
	size = inSize- sizeof(RpcParamHeader_t);
	inP = (void*)&(((uint8_t*)inData)[sizeof(RpcParamHeader_t)]);
	pData = (RpcParamData_t*)outData;
	oSize = outSize;
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2," STEP 1");
	for(;;) {
//		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2, "    inP:0x%x; size:%d; pData:0x%x;oSize:%d;",inP,size,pData,oSize);
		if(size < 0) {
			return RPC_PARAM_ERROR;		
		}
		if(oSize < sizeof(RpcParamData_t)) {
			return RPC_NO_MORE_RESOURCE;
		}
		if((ret = IPC_parseMarshaledData(inP,size,&needSize,pData)) != RPC_SUCCESS) {
			return ret;
		}
//		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2, "        PARSE TYPE:%s; LENG:%d;",IPC_ConvertTypeToString(pData->paramSpec.paramType),pData->paramSpec.paramLeng);
		oSize -= sizeof(RpcParamData_t);
		inP = (void*)&(((uint8_t*)inP)[needSize]);
		size -= needSize;
		if(pData->paramSpec.paramType == RPC_TYPE_UNKNOWN)
			break;
		pData++;
		outP = (void*)pData;
	}
	pData++;
	outP = (void*)pData;
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2," STEP 2");
	for(pData = (RpcParamData_t*)outData;pData->paramSpec.paramType != RPC_TYPE_UNKNOWN;pData++) {
//		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2, "    PUT:pData:0x%x TYPE:%s; LENG:%d; outP:0x%x; oSize:%d;",pData,IPC_ConvertTypeToString(pData->paramSpec.paramType),pData->paramSpec.paramLeng,outP,oSize);
		if(oSize < IPC_ALIGNMENT(pData->paramSpec.paramLeng)) {
			return RPC_NO_MORE_RESOURCE;
		}
		(void)memcpy(outP,pData->paramData,pData->paramSpec.paramLeng);
		outP = (void*)&(((uint8_t*)outP)[IPC_ALIGNMENT(pData->paramSpec.paramLeng)]);
		oSize -= IPC_ALIGNMENT(pData->paramSpec.paramLeng);
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG2, "  FREE MEM inData:0x%x;",inData);
	IPC_FreeShmMem(IpcMarshalDataSharedMemId, inData);
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_Unmarshal ");
	return RPC_SUCCESS;
}

void IPC_FreeMarshaledSharedData(void *data) {
	if(data == (void*)NULL)
		return;
	if(IpcMarshalDataSharedMemId != -1) {
		IPC_FreeShmMem(IpcMarshalDataSharedMemId, data);
	}
}

uint32_t IPC_GetMarshaledDataSize(void *data) {
	uint32_t size;RpcParamHeader_t *header;

	if(data == (void*)NULL)
		return (uint32_t)0;
	header = (RpcParamHeader_t*)data;
	size = header->paramLength;
	return size;
}

void *IPC_DuplicateMarshaledData(void *marshaledData,uint32_t marshaledDataSize) {
	void *ret;

	if((marshaledData == (void*)NULL)||(marshaledDataSize == (uint32_t)0))
		return (void*)NULL;
	if((ret = IPC_AllocShmMem(IpcMarshalDataSharedMemId, marshaledDataSize)) == NULL) {
		return ret;
	}
	(void)memcpy(ret,marshaledData,marshaledDataSize);
	return ret;
}

/* --- OTHER LIBRARY --- */

void IPC_SetMsgParam(RpcMsgParam_t *msgParam,uint32_t funcId,uint32_t pId,uint32_t queId,uint32_t seqNum,uint32_t extData,void* extParam) {
	msgParam->u.msgParam.funcId = funcId;
	msgParam->u.msgParam.srcPid = pId;
	msgParam->u.msgParam.srcQueId = queId;
	msgParam->u.msgParam.seqNum = seqNum;
	msgParam->u.msgParam.extData = extData;
	msgParam->u.msgParam.extParam = extParam;
}

/* ----- Internal Functions ----- */

static RpcResult_t IPC_parseOneData(RpcParamData_t *inData,uint32_t *useSize) {
	*useSize = sizeof(RpcParamSpec_t);
	switch(inData->paramSpec.paramType) {
	case RPC_INT_TYPE:
	case RPC_UINT_TYPE:
	case RPC_FLOAT_TYPE:
	case RPC_STRING_TYPE:
	case RPC_ENUM_TYPE:
	case RPC_OTHER_TYPE:
	case RPC_TYPE_UNKNOWN:
		break;
	default:
		*useSize = 0;
		return (RPC_PARAM_ERROR);
	}
	*useSize += IPC_ALIGNMENT(inData->paramSpec.paramLeng);
	return RPC_SUCCESS;
}

static RpcResult_t IPC_putOneData(RpcParamData_t *inData,void *outData,uint32_t outSize,uint32_t *useSize) {
	RpcResult_t ret = RPC_SUCCESS;
	RpcParamSpec_t *paramSpec;void *dataP;int rest;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG2,"IN IPC_putOneData inData:0x%x;(TYPE:0x%x; LENG:%d; DATA:0x%x;) outData:0x%x;, outSize:%d; useSize:0x%x;",
//		     inData,inData->paramSpec.paramType,inData->paramSpec.paramLeng,inData->paramData,outData,outSize,useSize);
	*useSize = 0;
	if(outSize < sizeof(RpcParamSpec_t)) {
		ret = RPC_NO_MORE_RESOURCE;
		goto errOut;
	}
	paramSpec = (RpcParamSpec_t*)outData;
	*useSize = sizeof(RpcParamSpec_t);
	dataP = (void*)&(((uint8_t*)outData)[*useSize]);
	switch(inData->paramSpec.paramType) {
	case RPC_INT_TYPE:
	case RPC_UINT_TYPE:
	case RPC_FLOAT_TYPE:
	case RPC_STRING_TYPE:
	case RPC_ENUM_TYPE:
	case RPC_OTHER_TYPE:
	case RPC_TYPE_UNKNOWN:
		paramSpec->paramType = inData->paramSpec.paramType;
		paramSpec->paramLeng = inData->paramSpec.paramLeng;
		if((paramSpec->paramLeng > 0)&&(outSize < (*useSize + IPC_ALIGNMENT(paramSpec->paramLeng)))) {
			ret = RPC_NO_MORE_RESOURCE;
			goto errOut;
		}
		break;
	default:
		ret = RPC_PARAM_ERROR;
		goto errOut;
	}
	if(paramSpec->paramLeng > 0) {
		*useSize += paramSpec->paramLeng;
		(void)memcpy(dataP,(void*)inData->paramData,paramSpec->paramLeng);
		rest = IPC_ALIGNMENT(paramSpec->paramLeng) - paramSpec->paramLeng;
		if(rest > 0) {
			dataP = (void*)&(((uint8_t*)outData)[*useSize]);
			(void)memset(dataP,0,rest);
			*useSize += rest;
		}
	}

errOut:
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_DEBUG2,"OUT IPC_putOneData useSize:%d; RET:0x%x;",*useSize,ret);
	return ret;
}

static RpcResult_t IPC_parseMarshaledData(void *inData,uint32_t inSize,uint32_t *needSize,RpcParamData_t *pData) {
	RpcResult_t ret = RPC_SUCCESS;RpcParamSpec_t *pSpec;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "STEP1");	
	if(inSize < sizeof(RpcParamSpec_t)) {
		*needSize = 0;
		return RPC_PARAM_ERROR;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "STEP2");	
	pSpec = (RpcParamSpec_t*)inData;
	switch(pSpec->paramType) {
	case RPC_INT_TYPE:
	case RPC_UINT_TYPE:
	case RPC_FLOAT_TYPE:
	case RPC_STRING_TYPE:
	case RPC_ENUM_TYPE:
	case RPC_OTHER_TYPE:
	case RPC_TYPE_UNKNOWN:
//		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "TYPE:%s",IPC_ConvertTypeToString(pSpec->paramType));	
		break;
	default:
		*needSize = 0;
		return RPC_PARAM_ERROR;
	}
	(void)memcpy((void*)pData,(void*)pSpec,sizeof(RpcParamSpec_t));
	*needSize = sizeof(RpcParamSpec_t) + IPC_ALIGNMENT(pSpec->paramLeng);
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "inSize:%d; NeedSize:%d TYPE:%s; LENG:%d;",inSize,*needSize,IPC_ConvertTypeToString(pData->paramSpec.paramType),pData->paramSpec.paramLeng);	
	if(inSize < *needSize) {
		*needSize = 0;
		return RPC_PARAM_ERROR;
	}
	pData->paramData = (void*)&(((uint8_t*)inData)[sizeof(RpcParamSpec_t)]);
//	(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL, RPC_LOG_LEVEL_DEBUG1, "paramData:0x%x",pData->paramData);	
	return ret;
}

/* ----- TEST Functions ----- */

uint8_t *IPC_ConvertTypeToString(RpcParamType_t type) {
	switch(type) {
	case RPC_INT_TYPE:
		return (uint8_t*)"INT   ";
	case RPC_UINT_TYPE:
		return (uint8_t*)"UINT  ";
	case RPC_FLOAT_TYPE:
		return (uint8_t*)"FLOAT ";
	case RPC_STRING_TYPE:
		return (uint8_t*)"STRING";
	case RPC_ENUM_TYPE:
		return (uint8_t*)"ENUM  ";
	case RPC_OTHER_TYPE:
		return (uint8_t*)"OTHER ";
	case RPC_TYPE_UNKNOWN:
		return (uint8_t*)"UNKNWN";
	default:
		return (uint8_t*)"ERROR ";
	}
}

void IPC_DumpParamData(RpcParamData_t *pData) {
	int i;char lData[128];

	fprintf(stdout,"--- PARAM DATA DATA:0x%x ---\n",(uint32_t)pData);
	i = 1;
	while(pData->paramSpec.paramType != RPC_TYPE_UNKNOWN) {
		fprintf(stdout," TYPE:%s LENG:%d DATA-P:0x%x\n",(char*)IPC_ConvertTypeToString(pData->paramSpec.paramType),pData->paramSpec.paramLeng,(uint32_t)pData->paramData);
		sprintf(lData,"NUM:%d; TYPE:%s",i,IPC_ConvertTypeToString(pData->paramSpec.paramType));
		RPC_MemDump(pData->paramData,pData->paramSpec.paramLeng,lData);
		pData++;i++;
	}
	fprintf(stdout," TYPE:%s LENG:%d DATA-P:0x%x\n",(char*)IPC_ConvertTypeToString(pData->paramSpec.paramType),pData->paramSpec.paramLeng,(uint32_t)pData->paramData);
	sprintf(lData,"NUM:%d; TYPE:%s",i,IPC_ConvertTypeToString(pData->paramSpec.paramType));
	RPC_MemDump(pData->paramData,pData->paramSpec.paramLeng,lData);
}

#ifdef	DEBUG

RpcResult_t marshalTest(void);

RpcResult_t marshalTest(void) {
	RpcResult_t ret;RpcParamData_t pData[11];
//	RpcParamData_t oData[20];
	int i;
	uint8_t *marshalData;void *shmData;uint32_t marshalSize;
	int8_t data0 = 1;uint16_t data1 = 2;int32_t data2 = -1;uint64_t data3 = 3;
	float data4 = 4.1;double data5 = 5.2;char *data6 = "abcdefghijklmn";
	RpcParamType_t data7 = RPC_ENUM_TYPE;
	uint8_t data8[128] = {0};uint8_t *data9;
	uint32_t unmarshalData[10*1024];

	fprintf(stdout,"@@@@@ NOW START MARSHAL TEST @@@@\n");
#if	0
	if((ret = IPC_InitSharedMem(RPC_TRUE)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_InitSharedMem ret:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
//	fprintf(stdout,"STEP2\n");
	if((ret = IPC_AttachSharedMem()) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_AttachSharedMem ret:%s;",RPC_LogConvertResultCode(ret));
		(void)IPC_DestroySharedMem();
		return ret;
	}
//	fprintf(stdout,"STEP3\n");
	if((ret = IPC_InitMarshal(RPC_TRUE)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_InitMarshal ret:%s;",RPC_LogConvertResultCode(ret));
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return ret;		
	}
#endif	/* 0 */
#if	0
//	fprintf(stdout,"STEP4\n");
	if((ret = IPC_InitShmGapArea(IpcMarshalDataSharedMemId, 1024)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN INIT SHM GAP AREA ret:%s;",RPC_LogConvertResultCode(ret));
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return ret;		
	}
//	fprintf(stdout,"STEP5\n");
	if((ret = IPC_InitShmAllocArea(IpcMarshalDataSharedMemId, IPC_MARSHAL_SHARED_MEM_REAL_SIZE, 6, 64, 2)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN INIT SHM ALLOC AREA ret:%s;",RPC_LogConvertResultCode(ret));
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return ret;		
	}
#endif	/* 0 */
//	fprintf(stdout,"STEP6\n");
	if((shmData = IPC_AllocShmMem(IpcMarshalDataSharedMemId, 2000)) == NULL) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN SHM ALLOCATION ");
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return RPC_FATAL_ERROR;			
	}
	fprintf(stdout,"@@@ SET DATA0 shmData:0x%x; @@@\n",(uint32_t)shmData);
	for(i = 0;i < 128;i++) {
		data8[i] = 128 -i;
	}
	data9 = (uint8_t*)shmData;
	for(i = 0;i < 2000;i++) {
		data9[i] = i;
	}
	fprintf(stdout,"@@@ SET DATA @@\n");
	pData[0].paramSpec.paramType = RPC_INT_TYPE;pData[0].paramSpec.paramLeng = 1;pData[0].paramData = (uint8_t*)&data0;
	pData[1].paramSpec.paramType = RPC_UINT_TYPE;pData[1].paramSpec.paramLeng = 2;pData[1].paramData = (uint8_t*)&data1;
	pData[2].paramSpec.paramType = RPC_INT_TYPE;pData[2].paramSpec.paramLeng = 4;pData[2].paramData = (uint8_t*)&data2;
	pData[3].paramSpec.paramType = RPC_UINT_TYPE;pData[3].paramSpec.paramLeng = 8;pData[3].paramData = (uint8_t*)&data3;
	pData[4].paramSpec.paramType = RPC_FLOAT_TYPE;pData[4].paramSpec.paramLeng = 4;pData[4].paramData = (uint8_t*)&data4;
	pData[5].paramSpec.paramType = RPC_FLOAT_TYPE;pData[5].paramSpec.paramLeng = 8;pData[5].paramData = (uint8_t*)&data5;
	pData[6].paramSpec.paramType = RPC_STRING_TYPE;pData[6].paramSpec.paramLeng = 15;pData[6].paramData = (uint8_t*)data6;
	pData[7].paramSpec.paramType = RPC_ENUM_TYPE;pData[7].paramSpec.paramLeng = 4;pData[7].paramData = (uint8_t*)&data7;
	pData[8].paramSpec.paramType = RPC_OTHER_TYPE;pData[8].paramSpec.paramLeng = 128;pData[8].paramData = (uint8_t*)data8;
	pData[9].paramSpec.paramType = RPC_OTHER_TYPE;pData[9].paramSpec.paramLeng = 4;pData[9].paramData = (uint8_t*)&data9;
	pData[10].paramSpec.paramType = RPC_TYPE_UNKNOWN;pData[10].paramSpec.paramLeng = 0;pData[10].paramData = (uint8_t*)NULL;
	fprintf(stdout,"\n\n@@@ BEFORE MARSHAL DATA @@@\n");
	IPC_DumpParamData(pData);
	RPC_MemDump((uint8_t*)(*((uint32_t*)(pData[9].paramData))),2000,"## SHM MEM DATA ##");


	if((ret = IPC_Marshal(pData, (void**)&marshalData, &marshalSize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN PACAKGE ret:%s;",RPC_LogConvertResultCode(ret));
		IPC_FreeShmMem(IpcMarshalDataSharedMemId, shmData);
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return ret;					
	}
	fprintf(stdout,"\n\n@@@ AFTER MARSHAL DATA @@@\n");
	RPC_MemDump(marshalData,marshalSize,"PACKED DATA");


	if((ret = IPC_Unmarshal(marshalData, marshalSize, unmarshalData, 10*1024)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_MARSHAL,RPC_LOG_LEVEL_FATAL,"ERROR IN UNMARSHAL ret:%s;",RPC_LogConvertResultCode(ret));
		IPC_FreeShmMem(IpcMarshalDataSharedMemId, marshalData);
		IPC_FreeShmMem(IpcMarshalDataSharedMemId, shmData);
		(void)IPC_DettachSharedMem();
		(void)IPC_DestroySharedMem();
		return ret;							
	}
	fprintf(stdout,"\n\n@@@ AFTER UNMARSHAL DATA @@@\n");
	IPC_DumpParamData((RpcParamData_t*)unmarshalData);
	RPC_MemDump((uint8_t*)(*((uint32_t*)((RpcParamData_t*)unmarshalData)[9].paramData)),2000,"## SHM MEM DATA ##");
	IPC_FreeShmMem(IpcMarshalDataSharedMemId, shmData);
	(void)IPC_DettachSharedMem();
	(void)IPC_DestroySharedMem();
	return RPC_SUCCESS;
}

#endif	/* DEBUG */

