/*
 * IPRPC - Inter Process Remote Procedure Call
 *
 * Copyright (C) 2012-2013 by Hiroyuki KAJIURA. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 *     1:Redistributions of source code must retain the above copyright notice, 
 *       this list of conditions and the following disclaimer.
 *     2:Redistributions in binary form must reproduce the above copyright notice, 
 *       this list of conditions and the following disclaimer in the documentation 
 *       and/or other materials provided with the distribution.
 *     3:Neither the name of copyright owner nor the names of its contributors 
 *       may be used to endorse or promote products derived from this software 
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef	USE_RENDEZVOUS

#define	IPC_RENDEZV_MAIN

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include	"ipcInfo.h"
#include	"ipcConf.h"
#include	"ipcType.h"
#include	"ipcLog.h"
#include	"ipcSyscall.h"
#include	"ipcLib.h"
#include	"ipcMarshal.h"
#include	"ipcQue.h"
#include	"ipcRendezv.h"

#define	IPC_RENDEZV_START_ID			(1)

typedef enum {
	IPC_RENDEZV_STATE_UNKNOWN = 0,
	IPC_RENDEZV_STATE_IDLE,
	IPC_RENDEZV_STATE_READY,
	IPC_RENDEZV_STATE_WAIT_RECEIVER,
	IPC_RENDEZV_STATE_WAIT_SENDER,
	IPC_RENDEZV_STATE_WAIT_GET_DATA,
	IPC_RENDEZV_STATE_WAIT_REPLY,
	IPC_RENDEZV_STATE_WAIT_END,
	IPC_RENDEZV_SATE_MAX
} IpcRendezvState_t;

typedef struct {
	uint32_t rendezvId;
	IpcRendezvState_t rendezvState;
	RpcTaskID_t senderTask;
	RpcTaskID_t receiverTask;
	void *sendData;
	uint32_t sendDataSize;
	void *replyData;
	uint32_t replyDataSize;
	RpcTaskControl_t tskCntl;
	RpcMutexID_t mutexId;
	RpcMutexAttribute_t mutexAttr;
} IpcRendezvEntry_t;

static int32_t IpcRendezvSharedMemId = -1;

static IpcRendezvControl_t *IpcLclRendezvControl = (IpcRendezvControl_t*)NULL;
static IpcRendezvControl_t *IpcShmRendezvControl = (IpcRendezvControl_t*)NULL;

static RpcResult_t IPC_rendezvSendState(IpcRendezvEntry_t *rep,RpcTaskID_t recvTask,void *sendData, uint32_t sendSize, RpcBool_t waitForever,struct timespec  *specTime);
static RpcResult_t IPC_rendezvPostSendState(IpcRendezvEntry_t *rep,void **replyData, uint32_t *replySize);
static RpcResult_t IPC_rendezvRecvState(IpcRendezvEntry_t *rep,RpcTaskID_t sendTask,RpcBool_t waitForever,struct timespec *specTime);
static RpcResult_t IPC_rendezvPostRecvState(IpcRendezvEntry_t *rep,void **recvData, uint32_t *recvSize);
static RpcResult_t IPC_rendezvReplyState(IpcRendezvEntry_t *rep,void *replyData, uint32_t replySize);
static RpcResult_t IPC_initRendezvResource(RpcBool_t IpcFlag,IpcRendezvControl_t *rcp);
static RpcResult_t IPC_allocRendezvResource(RpcBool_t IpcFlag,IpcRendezvControl_t *rcp,uint32_t *rendezvId);
static RpcResult_t IPC_freeRendezvResource(RpcBool_t IpcFlag, IpcRendezvControl_t *rcp, uint32_t rendezvId);
static IpcRendezvEntry_t *IPC_searchRendezvEntryByRendezvId(RpcBool_t IpcFlag,uint32_t rendezvId,IpcRendezvControl_t *rcp);
static IpcRendezvEntry_t *IPC_searchRendezvEntryByRecvTask(RpcBool_t IpcFlag,RpcTaskID_t recvTask,IpcRendezvControl_t *rcp);
static IpcRendezvEntry_t *IPC_searchRendezvEntryBySendTask(RpcBool_t IpcFlag,RpcTaskID_t sendTask,IpcRendezvControl_t *rcp);
static RpcResult_t RPC_createMutexForRendezv(RpcBool_t IpcFlag,IpcRendezvEntry_t *rep);
static void IPC_destroyMutexForRendezv(IpcRendezvEntry_t *rep);
static RpcResult_t IPC_checkTimeoutForRendezv(uint32_t timeOut,RpcBool_t *waitForever,struct timespec *specTime);
static uint32_t IPC_getNewRendezvId(void);
static RpcBool_t IPC_checkDuplicateRendezvId(IpcRendezvControl_t *srcp,IpcRendezvControl_t *lrcp,uint32_t rendezvId);
static void *IPC_rendezvAllocMem(RpcBool_t IpcFlag,uint32_t size);
static void IPC_rendezvFreeMem(RpcBool_t IpcFlag,void *addr);
static RpcBool_t IPC_compareRendezvByRendezvId(RpcList_t *list,uint32_t rendezvId);
static RpcBool_t IPC_compareRendezvByRecvTask(RpcList_t *list,uint32_t recvTask);
static RpcBool_t IPC_compareRendezvBySendTask(RpcList_t *list,uint32_t sendTask);

#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
static RpcHashEntry_t *IPC_initRendezvHash(RpcBool_t IpcFlag,uint32_t hashTableSize);
static RpcBool_t IPC_compareRendezvIdbyHash(void *entry, uint8_t *val, uint32_t valSize);
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */

/* ----- Global Functions ----- */

RpcResult_t IPC_InitRendezv(RpcBool_t masterProcess) {
	RpcResult_t ret;IpcRendezvControl_t *rcp;uint32_t fd;void *addr;uint32_t size;
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
	RpcHashEntry_t *hashEntry;
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_InitRendezv masterProcess:%d;",masterProcess);
	if(IpcRendezvSharedMemId != -1) {
		return RPC_SUCCESS;
	}
	if(IPC_GetShmInfo(IPC_RENDEZV_DATA_SHARED_MEM_NAME, &fd, &addr, &size) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
	IpcRendezvSharedMemId = fd;
	if(masterProcess) {
		if((ret = IPC_InitQue(masterProcess,RPC_TRUE)) != RPC_SUCCESS) {
			return ret;
		}
		if((rcp = IPC_GetRendezvControl()) == (IpcRendezvControl_t*)NULL) {
			return RPC_FATAL_ERROR;
		}
		if((ret = IPC_initRendezvResource(RPC_TRUE,rcp)) != RPC_SUCCESS) {
			return ret;
		}
		IpcShmRendezvControl = rcp;
		IpcShmRendezvControl->rendezvIdCounter = IPC_RENDEZV_START_ID;
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
		if((hashEntry = IPC_initRendezvHash(RPC_TRUE,IPC_GLOBAL_RENDEZV_HASH_TABLE)) == (RpcHashEntry_t*)NULL) {
			return RPC_NO_MORE_RESOURCE;
		}
		IpcShmRendezvControl->rendezvHashTable = hashEntry;
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	}
	else {
		if((rcp = IPC_GetRendezvControl()) == (IpcRendezvControl_t*)NULL) {
			return RPC_FATAL_ERROR;
		}
		IpcShmRendezvControl = (IpcRendezvControl_t*)rcp;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2," OK NOW INIT LOCAL RENDEZVOUS DATA");
	if((IpcLclRendezvControl = (IpcRendezvControl_t*)IPC_rendezvAllocMem(RPC_FALSE,sizeof(IpcRendezvControl_t))) == (IpcRendezvControl_t*)NULL) {
		return RPC_NO_MORE_RESOURCE;
	}
	rcp = (IpcRendezvControl_t*)IpcLclRendezvControl;
	if((ret = IPC_initRendezvResource(RPC_FALSE,rcp)) != RPC_SUCCESS) {
		(void)IPC_rendezvFreeMem(RPC_FALSE,rcp);
		IpcLclRendezvControl = (IpcRendezvControl_t*)NULL;
		return ret;
	}
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
	if((hashEntry = IPC_initRendezvHash(RPC_FALSE,IPC_LOCAL_RENDEZV_HASH_TABLE)) == (RpcHashEntry_t*)NULL) {
		(void)IPC_rendezvFreeMem(RPC_FALSE,rcp);
		IpcLclRendezvControl = (IpcRendezvControl_t*)NULL;
		return RPC_NO_MORE_RESOURCE;
	}
	rcp->rendezvHashTable = hashEntry;
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_InitRendezv masterProcess");
	return RPC_SUCCESS;
}

RpcResult_t IPC_CreateRendezv(RpcBool_t IpcFlag, uint32_t *rendezvId) {
	RpcResult_t ret;IpcRendezvControl_t *rcp;

	if(IpcFlag) {
		if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL)
			return RPC_NOT_READY;
	}
	else {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL)
			return RPC_NOT_READY;
	}
	if((ret = IPC_allocRendezvResource(IpcFlag,rcp,rendezvId)) != RPC_SUCCESS) {
		return ret;
	}
	return RPC_SUCCESS;
}

RpcResult_t IPC_DestoryRendezv(uint32_t rendezvId) {
	RpcResult_t ret;IpcRendezvControl_t *rcp;

	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((ret = IPC_freeRendezvResource(RPC_FALSE,rcp,rendezvId)) == RPC_SUCCESS) {
			return RPC_SUCCESS;
		}
		else {
			return ret;
		}
	}
	else if((ret = IPC_freeRendezvResource(RPC_TRUE,rcp,rendezvId)) == RPC_SUCCESS) {
		return RPC_SUCCESS;
	}
	else if(ret == RPC_PARAM_ERROR) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((ret = IPC_freeRendezvResource(RPC_FALSE,rcp,rendezvId)) == RPC_SUCCESS) {
			return RPC_SUCCESS;
		}
		else {
			return ret;
		}
	}
	else {
		return ret;
	}
}

RpcResult_t IPC_RendezvSend(uint32_t rendezvId, RpcTaskID_t recvTask,void *sendData, uint32_t sendSize,void **replyData, uint32_t *replySize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;

	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_TRUE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvSendState(rep,recvTask,sendData, sendSize,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostSendState(rep,replyData,replySize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	return ret;
}

RpcResult_t IPC_RendezvSendByTask(RpcBool_t IpcFlag, RpcTaskID_t recvTask,void *sendData, uint32_t sendSize,void **replyData, uint32_t *replySize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;uint32_t rendezvId;

	if(IpcFlag) {
		if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRecvTask(IpcFlag,recvTask,rcp)) == (IpcRendezvEntry_t*)NULL) {
			if((ret = IPC_allocRendezvResource(IpcFlag,rcp,&rendezvId)) != RPC_SUCCESS) {
				return ret;
			}
			else if((rep = IPC_searchRendezvEntryByRendezvId(IpcFlag,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
				return RPC_FATAL_ERROR;
			}
		}
	}
	else {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRecvTask(IpcFlag,recvTask,rcp)) == (IpcRendezvEntry_t*)NULL) {
			if((ret = IPC_allocRendezvResource(IpcFlag,rcp,&rendezvId)) != RPC_SUCCESS) {
				return ret;
			}
			else if((rep = IPC_searchRendezvEntryByRendezvId(IpcFlag,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
				return RPC_FATAL_ERROR;
			}
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvSendState(rep,recvTask,sendData, sendSize,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostSendState(rep,replyData,replySize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	return ret;
}

RpcResult_t IPC_RendezvSendByRendezvId(uint32_t rendezvId, void *sendData, uint32_t sendSize,void **replyData, uint32_t *replySize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_RendezvSendByRendezvId rendezvId:0x%x; sendData:0x%x; sendSize:%d; timeOut:%d;",
//		rendezvId,sendData,sendSize,timeOut);
	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_TRUE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvSendState(rep,(RpcTaskID_t)0,sendData, sendSize,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostSendState(rep,replyData,replySize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST SEND STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_RendezvSendByRendezvId replyData:0x%x; replySize:%d; RET:%s;",
//		*replyData,*replySize,RPC_LogConvertResultCode(ret));
	return ret;
}

RpcResult_t IPC_RendezvRecv(uint32_t rendezvId, RpcTaskID_t sendTask, void **recvData, uint32_t *recvSize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;

	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_TRUE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvRecvState(rep,sendTask,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostRecvState(rep,recvData,recvSize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	return ret;
}

RpcResult_t IPC_RendezvRecvByTask(RpcBool_t IpcFlag, RpcTaskID_t sendTask, uint32_t *rendezvId,void **recvData, uint32_t *recvSize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;

	if(IpcFlag) {
		if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryBySendTask(IpcFlag,sendTask,rcp)) == (IpcRendezvEntry_t*)NULL) {
			if((ret = IPC_allocRendezvResource(IpcFlag,rcp,rendezvId)) != RPC_SUCCESS) {
				return ret;
			}
			else if((rep = IPC_searchRendezvEntryByRendezvId(IpcFlag,*rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
				return RPC_FATAL_ERROR;
			}
		}
		else {
			*rendezvId = rep->rendezvId;
		}
	}
	else {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryBySendTask(IpcFlag,sendTask,rcp)) == (IpcRendezvEntry_t*)NULL) {
			if((ret = IPC_allocRendezvResource(IpcFlag,rcp,rendezvId)) != RPC_SUCCESS) {
				return ret;
			}
			else if((rep = IPC_searchRendezvEntryByRendezvId(IpcFlag,*rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
				return RPC_FATAL_ERROR;
			}
		}
		else {
			*rendezvId = rep->rendezvId;
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvRecvState(rep,sendTask,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostRecvState(rep,recvData,recvSize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	return ret;
}

RpcResult_t IPC_RendezvRecvByRendezvId(uint32_t rendezvId, void **recvData, uint32_t *recvSize, uint32_t timeOut) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;
	RpcBool_t waitForever;struct timespec specTime;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_RendezvRecvByRendezvId rendezvId:0x%x;timeOut:%d;",
//		rendezvId,timeOut);
	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_TRUE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	if((ret = IPC_checkTimeoutForRendezv(timeOut,&waitForever,&specTime)) != RPC_SUCCESS) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		return ret;
	}
	if((ret = IPC_rendezvRecvState(rep,(RpcTaskID_t)0,waitForever,&specTime)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_rendezvPostRecvState(rep,recvData,recvSize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS POST RECEIVE STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_RendezvRecvByRendezvId recvData:0x%x; recvSize:%d; RET:%s;",
//		*recvData,*recvSize,RPC_LogConvertResultCode(ret));
	return ret;
}

RpcResult_t IPC_RendezvReply(uint32_t rendezvId, void *replyData, uint32_t replySize) {
	IpcRendezvEntry_t *rep;IpcRendezvControl_t *rcp;RpcResult_t ret;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_RendezvReply rendezvId:0x%x; replyData:0x%x; replySize:%d;",
//		rendezvId,replyData,replySize);
	if((rcp = IpcShmRendezvControl) == (IpcRendezvControl_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_NOT_READY;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_TRUE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
		if((rcp = IpcLclRendezvControl) == (IpcRendezvControl_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
		else if((rep = IPC_searchRendezvEntryByRendezvId(RPC_FALSE,rendezvId,rcp)) == (IpcRendezvEntry_t*)NULL) {
			return RPC_PARAM_ERROR;
		}
	}
	if((ret = IPC_rendezvReplyState(rep,replyData, replySize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR," ERROR IN RENDEZVOUS REPLY STATE RET:%s;",RPC_LogConvertResultCode(ret));
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_RendezvReply RET:%s;",
//		RPC_LogConvertResultCode(ret));
	return ret;
}

/* ----- Internal Functions ----- */

static RpcResult_t IPC_rendezvSendState(IpcRendezvEntry_t *rep,RpcTaskID_t recvTask,void *sendData, uint32_t sendSize,RpcBool_t waitForever,struct timespec *specTime) {
	RpcResult_t ret;RpcTaskID_t myTask;struct timeval sleepTime;uint32_t errCode;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_rendezvSendState rep:0x%x; STATE:%d; recvTask:0x%x, sendData:0x%x; sendSize:%d; waitForever:%d;",
//		rep,rep->rendezvState,recvTask,sendData,sendSize,waitForever);
	if((ret = RPC_MutexLock(&(rep->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		return ret;
	}
	myTask = RPC_TaskSelf();
	switch(rep->rendezvState) {
	case IPC_RENDEZV_STATE_READY:
		rep->sendData = sendData;
		rep->sendDataSize = sendSize;
		rep->senderTask = myTask;
		rep->receiverTask = recvTask;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_RECEIVER;
		IPC_ResetTaskControl(&(rep->tskCntl));
		(void)RPC_MutexUnlock(&(rep->mutexId));
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"NOW WAIT SEND TASK:0x%x",myTask);
		if(waitForever) {
			ret = RPC_TaskControlWait(&(rep->tskCntl));
		}
		else {
			ret = RPC_TaskControlTimedWait(&(rep->tskCntl),specTime);
		}
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"OK WAKE UP SEND TASK:0x%x",myTask);
		break;
	case IPC_RENDEZV_STATE_WAIT_SENDER:
		if((recvTask != (RpcTaskID_t)0)&&(rep->receiverTask != recvTask)) {
			(void)RPC_MutexUnlock(&(rep->mutexId));
			ret = RPC_PARAM_ERROR;
			break;
		}
		else if((rep->senderTask != (RpcTaskID_t)0)&&(rep->senderTask != myTask)) {
			(void)RPC_MutexUnlock(&(rep->mutexId));
			ret = RPC_PARAM_ERROR;
			break;
		}
		rep->sendData = sendData;
		rep->sendDataSize = sendSize;
		rep->senderTask = myTask;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_GET_DATA;
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"NOW SIGNAL TO RECV TASK BY:0x%x",myTask);
		(void)RPC_TaskControlSignal(&(rep->tskCntl));
		(void)RPC_MutexUnlock(&(rep->mutexId));
		sleepTime.tv_sec = 0;
		sleepTime.tv_usec = IPC_RENDEZV_SLEEP_FOR_SWITCH_TASK*1000;
		(void)RPC_TaskSleep(&sleepTime,NULL,&errCode);
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"NOW WAIT SEND TASK:0x%x",myTask);
		if(waitForever) {
			ret = RPC_TaskControlWait(&(rep->tskCntl));
		}
		else {
			ret = RPC_TaskControlTimedWait(&(rep->tskCntl),specTime);
		}
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"OK WAKE UP SEND TASK:0x%x",myTask);
		break;
	default:
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_PARAM_ERROR;
		break;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_rendezvSendState State:%d; RET:%s;",
//		rep->rendezvState, RPC_LogConvertResultCode(ret));
	return ret;
}

static RpcResult_t IPC_rendezvPostSendState(IpcRendezvEntry_t *rep,void **replyData, uint32_t *replySize) {
	RpcResult_t ret;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_rendezvPostSendState rep:0x%x; STATE:%d;",
//		rep,rep->rendezvState);
	if((ret = RPC_MutexLock(&(rep->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		return ret;
	}
	switch(rep->rendezvState) {
	case IPC_RENDEZV_STATE_WAIT_END:
		*replyData = rep->replyData;
		*replySize = rep->replyDataSize;
		rep->replyData = (void*)NULL;
		rep->replyDataSize = 0;
		rep->senderTask = 0;
		rep->receiverTask = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_SUCCESS;
		break;
	case IPC_RENDEZV_STATE_WAIT_REPLY:
		rep->sendData = (void*)NULL;
		rep->sendDataSize = 0;
		rep->senderTask = 0;
		rep->receiverTask = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		*replyData = (void*)NULL;
		*replySize = 0;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_INTERUPTTED;
		break;
	case IPC_RENDEZV_STATE_WAIT_GET_DATA:
	case IPC_RENDEZV_STATE_WAIT_RECEIVER:
		*replyData = (void*)NULL;
		*replySize = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_TIMEOUT_ERROR;
		break;
	default:
		*replyData = (void*)NULL;
		*replySize = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_PARAM_ERROR;
		break;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_rendezvPostSendState State:%d; replyData:0x%x; replySize:%d; RET:%s; ",
//		rep->rendezvState, *replyData, *replySize,RPC_LogConvertResultCode(ret));
	return ret;

}

static RpcResult_t IPC_rendezvRecvState(IpcRendezvEntry_t *rep,RpcTaskID_t sendTask,RpcBool_t waitForever,struct timespec *specTime) {
	RpcResult_t ret;RpcTaskID_t myTask;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_rendezvRecvState rep:0x%x; STATE:%d; sendTask:0x%x,waitForever:%d;",
//		rep,rep->rendezvState,sendTask,waitForever);
	if((ret = RPC_MutexLock(&(rep->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		return ret;
	}
	myTask = RPC_TaskSelf();
	switch(rep->rendezvState) {
	case IPC_RENDEZV_STATE_READY:
		rep->receiverTask = myTask;
		rep->senderTask = sendTask;
		rep->replyData = (void*)NULL;
		rep->replyDataSize = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_SENDER;
		IPC_ResetTaskControl(&(rep->tskCntl));
		(void)RPC_MutexUnlock(&(rep->mutexId));
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"NOW WAIT RECV TASK:0x%x",myTask);
		if(waitForever) {
			ret = RPC_TaskControlWait(&(rep->tskCntl));
		}
		else {
			ret = RPC_TaskControlTimedWait(&(rep->tskCntl),specTime);
		}
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"OK WAKE UP RECV TASK:0x%x",myTask);
		break;
	case IPC_RENDEZV_STATE_WAIT_RECEIVER:
		if((sendTask != (RpcTaskID_t)0)&&(rep->senderTask != sendTask)) {
			(void)RPC_MutexUnlock(&(rep->mutexId));
			ret = RPC_PARAM_ERROR;
			break;
		}
		else if((rep->receiverTask != (RpcTaskID_t)0)&&(rep->receiverTask != myTask)) {
			(void)RPC_MutexUnlock(&(rep->mutexId));
			ret = RPC_PARAM_ERROR;
			break;
		}
		rep->receiverTask = myTask;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_GET_DATA;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_SUCCESS;
		break;
	default:
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_PARAM_ERROR;
		break;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_rendezvRecvState State:%d; RET:%s;",
//		rep->rendezvState, RPC_LogConvertResultCode(ret));
	return ret;
}

static RpcResult_t IPC_rendezvPostRecvState(IpcRendezvEntry_t *rep,void **recvData, uint32_t *recvSize) {
	RpcResult_t ret;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_rendezvPostRecvState rep:0x%x; STATE:%d;",
//		rep,rep->rendezvState);
	if((ret = RPC_MutexLock(&(rep->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		return ret;
	}
	switch(rep->rendezvState) {
	case IPC_RENDEZV_STATE_WAIT_GET_DATA:
		*recvData = rep->sendData;
		*recvSize = rep->sendDataSize;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_REPLY;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_SUCCESS;
		break;
	case IPC_RENDEZV_STATE_WAIT_SENDER:
		*recvData = (void*)NULL;
		*recvSize = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_TIMEOUT_ERROR;
		break;
	default:
		*recvData = (void*)NULL;
		*recvSize = 0;
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_PARAM_ERROR;
		break;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_rendezvPostRecvState State:%d; recvData:0x%x; recvSize:%d; RET:%s; ",
//		rep->rendezvState, *recvData, *recvSize,RPC_LogConvertResultCode(ret));
	return ret;
}

static RpcResult_t IPC_rendezvReplyState(IpcRendezvEntry_t *rep,void *replyData, uint32_t replySize) {
	RpcResult_t ret;RpcTaskID_t myTask;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_rendezvReplyState rep:0x%x; STATE:%d; replyData:0x%x; replySize:%d;",
//		rep,rep->rendezvState, replyData, replySize);
	if((ret = RPC_MutexLock(&(rep->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		if(rep->rendezvState == IPC_RENDEZV_STATE_READY) rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		return ret;
	}
	myTask = RPC_TaskSelf();
	switch(rep->rendezvState) {
	case IPC_RENDEZV_STATE_WAIT_REPLY:
		if(rep->receiverTask != myTask) {
			(void)RPC_MutexUnlock(&(rep->mutexId));
			ret = RPC_PARAM_ERROR;
			break;
		}
		rep->replyData = replyData;
		rep->replyDataSize = replySize;
		rep->rendezvState = IPC_RENDEZV_STATE_WAIT_END;
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"NOW SIGNAL TO SEND TASK BY:0x%x",myTask);
		(void)RPC_TaskControlSignal(&(rep->tskCntl));
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_SUCCESS;
		break;
	default:
		rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
		(void)RPC_MutexUnlock(&(rep->mutexId));
		ret = RPC_PARAM_ERROR;
		break;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_rendezvReplyState State:%d; RET:%s; ",
//		rep->rendezvState, RPC_LogConvertResultCode(ret));
	return ret;
}

static RpcResult_t IPC_initRendezvResource(RpcBool_t IpcFlag,IpcRendezvControl_t *rcp) {
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_initRendezvResource IpcFlag:%d; rcp:0x%x;",IpcFlag,rcp);
	if(RPC_MutexAttrInit(&(rcp->mutexAttr)) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
#ifdef	_LINUX_
	if(RPC_MutexAttrSetType(&(rcp->mutexAttr), PTHREAD_MUTEX_ADAPTIVE_NP) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
#else	/* _LINUX_ */
	if(RPC_MutexAttrSetType(&(rcp->mutexAttr), PTHREAD_MUTEX_RECURSIVE) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
#endif	/* _LINUX_ */
	if(IpcFlag) {
		if(RPC_MutexAttrSetPshared(&(rcp->mutexAttr), PTHREAD_PROCESS_SHARED) != RPC_SUCCESS) {
			return RPC_FATAL_ERROR;
		}
	}
	else {
		if(RPC_MutexAttrSetPshared(&(rcp->mutexAttr), PTHREAD_PROCESS_PRIVATE) != RPC_SUCCESS) {
			return RPC_FATAL_ERROR;
		}
	}
	if(RPC_InitMutex(&(rcp->mutexId),&(rcp->mutexAttr)) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
	rcp->rendezvList = (RpcList_t*)NULL;
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_initRendezvResource");
	return RPC_SUCCESS;
}

static RpcResult_t IPC_allocRendezvResource(RpcBool_t IpcFlag,IpcRendezvControl_t *rcp,uint32_t *rendezvId) {
	IpcRendezvEntry_t *rep;uint32_t tmp;RpcResult_t ret;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_allocRendezvResource IpcFlag:%d; rcp:0x%x; rendezvId:0x%x;",
//		IpcFlag, rcp, rendezvId);
	if(rendezvId == (uint32_t*)NULL)
		return RPC_PARAM_ERROR;
	*rendezvId = IPC_RENDEZV_ID_UNKNOWN;
	tmp = IPC_getNewRendezvId();
	if((ret = RPC_MutexLock(&(rcp->mutexId))) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock RET:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((rep = RPC_AllocListData(&(rcp->rendezvList),sizeof(IpcRendezvEntry_t),(uint32_t)IpcFlag,IPC_rendezvAllocMem,IPC_rendezvFreeMem)) == (IpcRendezvEntry_t*)NULL) {
		(void)RPC_MutexUnlock(&(rcp->mutexId));
		return RPC_NO_MORE_RESOURCE;		
	}
	(void)memset((void*)rep,0,sizeof(IpcRendezvEntry_t));
	if((ret = RPC_createMutexForRendezv(IpcFlag,rep)) != RPC_SUCCESS) {
		goto errOut;
	}
	if((ret = RPC_CreateTaskControl(IpcFlag,&(rep->tskCntl))) != RPC_SUCCESS) {
		IPC_destroyMutexForRendezv(rep);
		goto errOut;
	}
	rep->rendezvState = IPC_RENDEZV_STATE_IDLE;
	rep->rendezvId = *rendezvId = tmp;
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
	if(rcp->rendezvHashTable != (RpcHashEntry_t*)NULL) {
		if((ret = RPC_AddHashEntry((void*)rcp->rendezvHashTable,(void*)rep,(void*)rendezvId,sizeof(*rendezvId), IpcFlag?IPC_GLOBAL_RENDEZV_HASH_TABLE:IPC_LOCAL_RENDEZV_HASH_TABLE)) != RPC_SUCCESS) {
			RPC_DestroyTaskControl(&(rep->tskCntl));
			IPC_destroyMutexForRendezv(rep);
			goto errOut;
		}
	}
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	(void)RPC_MutexUnlock(&(rcp->mutexId));
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_allocRendezvResource rendezvId:0x%x; RENDEZV-ENTITY:0x%x; state:%d; hashTable:0x%x",
//		*rendezvId,rep,rep->rendezvState,rcp->rendezvHashTable);
	return RPC_SUCCESS;

errOut:
	RPC_FreeListData(&(rcp->rendezvList),(void*)rep,(uint32_t)IpcFlag,IPC_rendezvFreeMem);
	(void)RPC_MutexUnlock(&(rcp->mutexId));
	return ret;
}

static RpcResult_t IPC_freeRendezvResource(RpcBool_t IpcFlag, IpcRendezvControl_t *rcp, uint32_t rendezvId) {
	IpcRendezvEntry_t *rep;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN IPC_freeRendezvResource IpcFlag:0x%x; rcp:0x%x; rendezvId:0x%x",
//		IpcFlag, rcp, rendezvId);
	if(RPC_MutexLock(&(rcp->mutexId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
		return RPC_FATAL_ERROR;
	}
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
	if(rcp->rendezvHashTable != (RpcHashEntry_t*)NULL) {
		if((rep = (IpcRendezvEntry_t*)RPC_SearchEntryByHash((void*)rcp->rendezvHashTable,(void*)&rendezvId,sizeof(rendezvId), IpcFlag?IPC_GLOBAL_RENDEZV_HASH_TABLE:IPC_LOCAL_RENDEZV_HASH_TABLE,IPC_compareRendezvIdbyHash)) != (IpcRendezvEntry_t*)NULL) {
			if(rep->rendezvState != IPC_RENDEZV_STATE_IDLE) {
				(void)RPC_MutexUnlock(&(rcp->mutexId));
				return RPC_BUSY_ERROR;
			}
			(void)RPC_DeleteHashEntry((void*)rcp->rendezvHashTable,(void*)&rendezvId,sizeof(rendezvId),  IpcFlag?IPC_GLOBAL_RENDEZV_HASH_TABLE:IPC_LOCAL_RENDEZV_HASH_TABLE,IPC_compareRendezvIdbyHash);
			RPC_DestroyTaskControl(&(rep->tskCntl));
			IPC_destroyMutexForRendezv(rep);
			RPC_FreeListData(&(rcp->rendezvList),(void*)rep,(uint32_t)IpcFlag,IPC_rendezvFreeMem);
			(void)RPC_MutexUnlock(&(rcp->mutexId));
//			(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_freeRendezvResource SUCCESS");
			return RPC_SUCCESS;
		}
	}
	else {
		if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,rendezvId,IPC_compareRendezvByRendezvId)) != (IpcRendezvEntry_t*)NULL) {
			if(rep->rendezvState != IPC_RENDEZV_STATE_IDLE) {
				(void)RPC_MutexUnlock(&(rcp->mutexId));
				return RPC_BUSY_ERROR;
			}
			RPC_DestroyTaskControl(&(rep->tskCntl));
			IPC_destroyMutexForRendezv(rep);
			RPC_FreeListData(&(rcp->rendezvList),(void*)rep,(uint32_t)IpcFlag,IPC_rendezvFreeMem);
			(void)RPC_MutexUnlock(&(rcp->mutexId));
//			(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_freeRendezvResource SUCCESS");
			return RPC_SUCCESS;
		}
	}
#else	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,rendezvId,IPC_compareRendezvByRendezvId)) != (IpcRendezvEntry_t*)NULL) {
		if(rep->rendezvState != IPC_RENDEZV_STATE_IDLE) {
			(void)RPC_MutexUnlock(&(rcp->mutexId));
			return RPC_BUSY_ERROR;
		}
		RPC_DestroyTaskControl(&(rep->tskCntl));
		IPC_destroyMutexForRendezv(rep);
		RPC_FreeListData(&(rcp->rendezvList),(void*)rep,(uint32_t)IpcFlag,IPC_rendezvFreeMem);
		(void)RPC_MutexUnlock(&(rcp->mutexId));
//		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_freeRendezvResource SUCCESS");
		return RPC_SUCCESS;
	}
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	(void)RPC_MutexUnlock(&(rcp->mutexId));
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"OUT IPC_freeRendezvResource ERROR");
	return RPC_PARAM_ERROR;
}

static IpcRendezvEntry_t *IPC_searchRendezvEntryByRendezvId(RpcBool_t IpcFlag,uint32_t rendezvId,IpcRendezvControl_t *rcp) {
	IpcRendezvEntry_t *rep;

	if(RPC_MutexLock(&(rcp->mutexId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
		return (IpcRendezvEntry_t*)NULL;
	}
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
	if(rcp->rendezvHashTable != (RpcHashEntry_t*)NULL) {
		if((rep = (IpcRendezvEntry_t*)RPC_SearchEntryByHash((void*)rcp->rendezvHashTable,(void*)&rendezvId,sizeof(rendezvId), IpcFlag?IPC_GLOBAL_RENDEZV_HASH_TABLE:IPC_LOCAL_RENDEZV_HASH_TABLE,IPC_compareRendezvIdbyHash)) != (IpcRendezvEntry_t*)NULL) {
			if(rep->rendezvState == IPC_RENDEZV_STATE_IDLE)
				rep->rendezvState = IPC_RENDEZV_STATE_READY;
			(void)RPC_MutexUnlock(&(rcp->mutexId));
			return rep;
		}
	}
	else {
		if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,rendezvId,IPC_compareRendezvByRendezvId)) != (IpcRendezvEntry_t*)NULL) {
			if(rep->rendezvState == IPC_RENDEZV_STATE_IDLE)
				rep->rendezvState = IPC_RENDEZV_STATE_READY;
			(void)RPC_MutexUnlock(&(rcp->mutexId));
			return rep;
		}
	}
#else	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,rendezvId,IPC_compareRendezvByRendezvId)) != (IpcRendezvEntry_t*)NULL) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_IDLE)
			rep->rendezvState = IPC_RENDEZV_STATE_READY;
		(void)RPC_MutexUnlock(&(rcp->mutexId));
		return rep;
	}
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	(void)RPC_MutexUnlock(&(rcp->mutexId));
	return (IpcRendezvEntry_t*)NULL;
}

static IpcRendezvEntry_t *IPC_searchRendezvEntryByRecvTask(RpcBool_t IpcFlag,RpcTaskID_t recvTask,IpcRendezvControl_t *rcp) {
	IpcRendezvEntry_t *rep;

	if(RPC_MutexLock(&(rcp->mutexId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
		return (IpcRendezvEntry_t*)NULL;
	}
	if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,(uint32_t)recvTask,IPC_compareRendezvByRecvTask)) != (IpcRendezvEntry_t*)NULL) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_IDLE)
			rep->rendezvState = IPC_RENDEZV_STATE_READY;
		(void)RPC_MutexUnlock(&(rcp->mutexId));
		return rep;
	}
	(void)RPC_MutexUnlock(&(rcp->mutexId));
	return (IpcRendezvEntry_t*)NULL;
}

static IpcRendezvEntry_t *IPC_searchRendezvEntryBySendTask(RpcBool_t IpcFlag,RpcTaskID_t sendTask,IpcRendezvControl_t *rcp) {
	IpcRendezvEntry_t *rep;

	if(RPC_MutexLock(&(rcp->mutexId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
		return (IpcRendezvEntry_t*)NULL;
	}
	if((rep = (IpcRendezvEntry_t*)RPC_SearchListEntryById((RpcList_t*)rcp->rendezvList,(uint32_t)sendTask,IPC_compareRendezvBySendTask)) != (IpcRendezvEntry_t*)NULL) {
		if(rep->rendezvState == IPC_RENDEZV_STATE_IDLE)
			rep->rendezvState = IPC_RENDEZV_STATE_READY;
		(void)RPC_MutexUnlock(&(rcp->mutexId));
		return rep;
	}
	(void)RPC_MutexUnlock(&(rcp->mutexId));
	return (IpcRendezvEntry_t*)NULL;
}

static RpcResult_t RPC_createMutexForRendezv(RpcBool_t IpcFlag,IpcRendezvEntry_t *rep) {
	RpcResult_t ret;

	if((ret = RPC_MutexAttrInit(&(rep->mutexAttr))) != RPC_SUCCESS) {
		return ret;
	}
#ifdef	_LINUX_
	if(RPC_MutexAttrSetType(&(rep->mutexAttr), PTHREAD_MUTEX_ADAPTIVE_NP) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
#else	/* _LINUX_ */
	if(RPC_MutexAttrSetType(&(rep->mutexAttr), PTHREAD_MUTEX_RECURSIVE) != RPC_SUCCESS) {
		return RPC_FATAL_ERROR;
	}
#endif	/* _LINUX_ */
	if((ret = RPC_MutexAttrSetPshared(&(rep->mutexAttr), IpcFlag?PTHREAD_PROCESS_SHARED:PTHREAD_PROCESS_PRIVATE)) != RPC_SUCCESS) {
		return ret;
	}
	if((ret = RPC_InitMutex(&(rep->mutexId),&(rep->mutexAttr))) != RPC_SUCCESS) {
		return ret;
	}
	return RPC_SUCCESS;
}

static void IPC_destroyMutexForRendezv(IpcRendezvEntry_t *rep) {
	(void)RPC_MutexDestroy(&(rep->mutexId));
}

static RpcResult_t IPC_checkTimeoutForRendezv(uint32_t timeOut,RpcBool_t *waitForever,struct timespec *specTime) {
	struct timeval absTime;

	*waitForever = RPC_FALSE;
	if(timeOut == 0)
		timeOut = IPC_RENDEZV_DEFAULT_TIMEOUT;
	else if(timeOut == IPC_RENDEZV_WAIT_FOREVER) {
		timeOut = IPC_RENDEZV_DEFAULT_TIMEOUT;
		*waitForever = RPC_TRUE;
	}
	if(!(*waitForever)) {
		if(!RPC_SetTimeout(timeOut,&absTime)) {
			return RPC_SYSCALL_ERROR;
		}
		if(!RPC_CheckTimeout(&absTime)) {
			return RPC_TIMEOUT_ERROR;
		}
		RPC_ConvertTime(&absTime,specTime);
	}
	return RPC_SUCCESS;
}

static uint32_t IPC_getNewRendezvId(void) {
	uint32_t ret;IpcRendezvControl_t *lrcp,*srcp;

	srcp = IpcShmRendezvControl;
	lrcp = IpcLclRendezvControl;
	if(srcp == (IpcRendezvControl_t*)NULL) {
		return IPC_RENDEZV_ID_UNKNOWN;
	}
	if(RPC_MutexLock(&(srcp->mutexId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
		return IPC_RENDEZV_ID_UNKNOWN;
	}
	if(lrcp != (IpcRendezvControl_t*)NULL) {
		if(RPC_MutexLock(&(lrcp->mutexId)) != RPC_SUCCESS) {
			(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL," ERROR in RPC_MutexLock");
			(void)RPC_MutexUnlock(&(srcp->mutexId));
			return IPC_RENDEZV_ID_UNKNOWN;
		}
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"  OK NOW ASSIGN NEW ID COUNTER:0x%x; VAL:%d;",
//		&(srcp->rendezvIdCounter),srcp->rendezvIdCounter);
	for(;;) {
		ret = (srcp->rendezvIdCounter)++;
		if((ret == IPC_RENDEZV_ID_UNKNOWN)||(ret == IPC_RENDEZV_ID_MAX)) {
//			(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"  RESET THE RENDEZV ID COUNTER NUM IS 1 COUNTER:0x%x; VAL:%d;",
//				&(srcp->rendezvIdCounter),srcp->rendezvIdCounter);
			ret = srcp->rendezvIdCounter = IPC_RENDEZV_START_ID;
		}
		if(IPC_checkDuplicateRendezvId(srcp, lrcp,ret))
			break;
	}
	if(lrcp != (IpcRendezvControl_t*)NULL) (void)RPC_MutexUnlock(&(lrcp->mutexId));
	(void)RPC_MutexUnlock(&(srcp->mutexId));
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG2,"   ASSIGN NEW NUM:0x%x;",ret);
	return ret;
}

static RpcBool_t IPC_checkDuplicateRendezvId(IpcRendezvControl_t *srcp,IpcRendezvControl_t *lrcp,uint32_t rendezvId) {
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"IN IPC_checkDuplicateRendezvId PID:0x%x; rendezvId:0x%x;",RPC_GetPid(), rendezvId);
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"   NOW CHECK GLOBAL RENDEZV");
	if(srcp != NULL) {
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
		if(srcp->rendezvHashTable != (RpcHashEntry_t*)NULL) {
			if(RPC_SearchEntryByHash((void*)srcp->rendezvHashTable,(void*)&rendezvId,sizeof(rendezvId), IPC_GLOBAL_RENDEZV_HASH_TABLE,IPC_compareRendezvIdbyHash) != (void*)NULL) {
				return RPC_FALSE;
			}
		}
		else {
			if(RPC_SearchListEntryById((void*)(srcp->rendezvList),(uint32_t)rendezvId,IPC_compareRendezvByRendezvId) != (void*)NULL) {
				return RPC_FALSE;
			}
		}
#else	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
		if(RPC_SearchListEntryById((void*)(srcp->rendezvList),(uint32_t)rendezvId,IPC_compareRendezvByRendezvId) != (void*)NULL) {
			return RPC_FALSE;
		}
#endif	/*IPC_USE_HASH_FUNC_FOR_RENDEZV */
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"   NOW CHECK LOCAL RENDEZV");
	if(lrcp != NULL) {
#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
		if(lrcp->rendezvHashTable != (RpcHashEntry_t*)NULL) {
			if(RPC_SearchEntryByHash((void*)lrcp->rendezvHashTable,(void*)&rendezvId,sizeof(rendezvId), IPC_LOCAL_RENDEZV_HASH_TABLE,IPC_compareRendezvIdbyHash) != (void*)NULL) {
				return RPC_FALSE;
			}
		}
		else {
			if(RPC_SearchListEntryById((void*)(lrcp->rendezvList),(uint32_t)rendezvId,IPC_compareRendezvByRendezvId) != (void*)NULL) {
				return RPC_FALSE;
			}
		}
#else	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
		if(RPC_SearchListEntryById((void*)(lrcp->rendezvList),(uint32_t)rendezvId,IPC_compareRendezvByRendezvId) != (void*)NULL) {
			return RPC_FALSE;
		}
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"OUT IPC_checkDuplicateRendezvId PID:0x%x;",RPC_GetPid());
	return RPC_TRUE;
}

static void *IPC_rendezvAllocMem(RpcBool_t IpcFlag,uint32_t size) {
	if(IpcFlag) {
		if(IpcRendezvSharedMemId == -1) {
			return (void*)NULL;
		}
		else {
			return IPC_AllocShmMem(IpcRendezvSharedMemId, size);
		}
	}
	else {
		return RPC_AllocMem(size);
	}
}

static void IPC_rendezvFreeMem(RpcBool_t IpcFlag,void *addr) {
	if(IpcFlag) {
		if(IpcRendezvSharedMemId == -1) {
			return;
		}
		else {
			IPC_FreeShmMem(IpcRendezvSharedMemId, addr);
		}
	}
	else {
		RPC_FreeMem(addr);
	}
	return;
}

static RpcBool_t IPC_compareRendezvByRendezvId(RpcList_t *list,uint32_t rendezvId) {
	if(((IpcRendezvEntry_t*)(list->entry))->rendezvId == rendezvId) {
		return RPC_TRUE;
	}
	return RPC_FALSE;
}

static RpcBool_t IPC_compareRendezvByRecvTask(RpcList_t *list,uint32_t recvTask) {
	if(((IpcRendezvEntry_t*)(list->entry))->receiverTask == recvTask) {
		return RPC_TRUE;
	}
	return RPC_FALSE;
}

static RpcBool_t IPC_compareRendezvBySendTask(RpcList_t *list,uint32_t sendTask) {
	if(((IpcRendezvEntry_t*)(list->entry))->senderTask == sendTask) {
		return RPC_TRUE;
	}
	return RPC_FALSE;
}

#ifdef	IPC_USE_HASH_FUNC_FOR_RENDEZV
static RpcHashEntry_t *IPC_initRendezvHash(RpcBool_t IpcFlag,uint32_t hashTableSize) {
	RpcHashEntry_t *hashEntry;int i;

//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"IN IPC_initRendezvHash IpcFlag:%d; hashTableSize:%d;",IpcFlag,hashTableSize);
	if((hashEntry = (RpcHashEntry_t*)IPC_rendezvAllocMem(IpcFlag,hashTableSize*sizeof(RpcHashEntry_t))) == (RpcHashEntry_t*)NULL) {
		return (RpcHashEntry_t*)NULL;
	}
	for(i = 0;i < hashTableSize;i++) {
		hashEntry[i].entry = (void*)NULL;
		hashEntry[i].usedFlag = RPC_FALSE;
	}
//	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG3,"OUT IPC_initRendezvHash hashEntry:0x%x;",hashEntry);
	return hashEntry;
}

static RpcBool_t IPC_compareRendezvIdbyHash(void *entry, uint8_t *val, uint32_t valSize) {
	return (memcmp((void*)&(((IpcRendezvEntry_t*)entry)->rendezvId),val,valSize)==0)?RPC_TRUE:RPC_FALSE;
}
#endif	/* IPC_USE_HASH_FUNC_FOR_RENDEZV */

/* ----- TEST ----- */

#ifdef	DEBUG

RpcResult_t RendezvTest(RpcBool_t IpcFlag);

static void *rendezvSender(void* rendP);
static void *rendezvReceiver(void* rendP);

static RpcTaskID_t senderTask,receiverTask;

RpcResult_t RendezvTest(RpcBool_t IpcFlag) {
//	RpcTaskID_t senderTask,receiverTask;
	RpcTaskAttribute_t senderAttr,receiverAttr;
	uint32_t rendezvId;void *retVal;RpcResult_t ret;
#if	0
	struct timeval curTime;
	uint32_t rend2, rend3,rend4,rend5,rend6,rend7;
#endif	/* 0 */

	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW TEST RENDEZV SEQUENCE IpcFlag:%d;",IpcFlag);
	if((ret = IPC_CreateRendezv(IpcFlag, &rendezvId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = RPC_TaskAttrInit(&senderAttr)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN RPC_TaskAttrInit:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = RPC_TaskAttrInit(&receiverAttr)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN RPC_TaskAttrInit:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = RPC_CreateTask(&senderTask,&senderAttr,rendezvSender,(void*)&rendezvId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN RPC_CreateTask:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = RPC_CreateTask(&receiverTask,&receiverAttr,rendezvReceiver,(void*)&rendezvId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN RPC_CreateTask:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
#if	0
	if((ret == RPC_GetTimeOfDay(&curTime,NULL,&errCode)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN GetTimeOfDay:%s; errCode:%d;",RPC_LogConvertResultCode(ret),errCode);
		return ret;
	}
	RPC_Srand((uint32_t)(curTime.tv_sec+curTime.tv_usec));
#endif	/* 0 */
	if((ret = RPC_JoinTask(senderTask, &retVal)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN JoinTask:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
#if	0
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW TEST RENDEZV CREATE AND DESTROY");
	if((ret = IPC_CreateRendezv(IpcFlag, &rend2)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_CreateRendezv(IpcFlag, &rend3)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_CreateRendezv(IpcFlag, &rend4)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_CreateRendezv(IpcFlag, &rend5)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_CreateRendezv(IpcFlag, &rend6)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_CreateRendezv(IpcFlag, &rend7)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_FATAL,"ERROR IN IPC_CreateRendezv:%s;",RPC_LogConvertResultCode(ret));
		return ret;
	}
	if((ret = IPC_DestoryRendezv(rend2)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_DestoryRendezv RET:%s",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_DestoryRendezv(rend7)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_DestoryRendezv RET:%s",RPC_LogConvertResultCode(ret));
	}
	if((ret = IPC_DestoryRendezv(rend4)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_DestoryRendezv RET:%s",RPC_LogConvertResultCode(ret));
	}
#endif	/* 0 */
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"END OF RENDEZVOUS TEST");
	return ret;
}

static void *rendezvSender(void* rendP) {
	uint32_t rendezvId;static char *sendData="abcdefghijklmn";uint32_t sendSize;
	char *replyData;uint32_t replySize;uint32_t timeOut;RpcResult_t ret;uint32_t errCode;
	struct timeval sleepTime;

	rendezvId = (uint32_t)(*((uint32_t*)rendP));
	sendSize = strlen(sendData)+1;
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN rendezvSender Task:0x%x; rendezvId:0x%x;",RPC_TaskSelf(),rendezvId);
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"SEND DATA:%s; SIZE:%d;",sendData,sendSize);
	sleepTime.tv_sec = 1;
	sleepTime.tv_usec = 0;
	(void)RPC_TaskSleep(&sleepTime,NULL,&errCode);
	timeOut = 10*1000;

#if	1
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvSendByRendezvId rendezvId:0x%x;",rendezvId);
	if((ret = IPC_RendezvSendByRendezvId(rendezvId,(void*)sendData, sendSize,(void**)&replyData,&replySize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvSendByRendezvId RET:%s",RPC_LogConvertResultCode(ret));
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvSendByRendezvId replyData:%s; replySize:%d;",replyData,replySize);
#else	/* 0 */
#if	0
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvSend rendezvId:0x%x; receiverTask:0x%x;",rendezvId,receiverTask);
	if((ret = IPC_RendezvSend(rendezvId,receiverTask,(void*)sendData, sendSize,(void**)&replyData,&replySize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvSend RET:%s",RPC_LogConvertResultCode(ret));
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvSend replyData:%s; replySize:%d;",replyData,replySize);
#else	/* 0 */
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvSendByTask IpcFlag:%d; receiverTask:0x%x;",RPC_TRUE,receiverTask);
	if((ret = IPC_RendezvSendByTask(RPC_TRUE,receiverTask,(void*)sendData, sendSize,(void**)&replyData,&replySize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvSendByTask RET:%s",RPC_LogConvertResultCode(ret));
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvSendByTask replyData:%s; replySize:%d;",replyData,replySize);
#endif	/* 0 */
#endif	/* 0 */
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_DestoryRendezv");
	if((ret = IPC_DestoryRendezv(rendezvId)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_DestoryRendezv RET:%s",RPC_LogConvertResultCode(ret));
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"SENDER END");
	return (void*)NULL;
}

static void *rendezvReceiver(void* rendP) {
	uint32_t rendezvId;static char *replyData = "opqrstuvwxyz";uint32_t replySize;
	char *recvData;uint32_t recvSize;uint32_t timeOut;RpcResult_t ret;uint32_t errCode;
	struct timeval sleepTime;

	rendezvId = (uint32_t)(*((uint32_t*)rendP));
	replySize = strlen(replyData)+1;
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"IN rendezvReceiver TASK:0x%x; rendezvId:0x%x;",RPC_TaskSelf(),rendezvId);
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"REPLY DATA:%s; SIZE:%d;",replyData,replySize);
	sleepTime.tv_sec = 2;
	sleepTime.tv_usec = 0;
	(void)RPC_TaskSleep(&sleepTime,NULL,&errCode);
	timeOut = 10*1000;

#if	0
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvRecvByRendezvId rendezvId:0x%x;",rendezvId);
	if((ret = IPC_RendezvRecvByRendezvId(rendezvId,(void**)&recvData, &recvSize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvRecvByRendezvId RET:%s",RPC_LogConvertResultCode(ret));
		return;
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvRecvByRendezvId recvData:%s; recvSize:%d;",
		recvData,recvSize);
#else	/* 0 */
#if	0
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvRecv rendezvId:0x%x; senderTask:0x%x;",rendezvId,senderTask);
	if((ret = IPC_RendezvRecv(rendezvId,senderTask,(void**)&recvData, &recvSize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvRecv RET:%s",RPC_LogConvertResultCode(ret));
		return;
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvRecv recvData:%s; recvSize:%d;",
		recvData,recvSize);
#else	/* 0 */
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvRecvByTask IpcFlag:%d; senderTask:0x%x;",RPC_TRUE,senderTask);
	if((ret = IPC_RendezvRecvByTask(RPC_TRUE,senderTask,&rendezvId,(void**)&recvData, &recvSize, timeOut)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvRecvByTask RET:%s",RPC_LogConvertResultCode(ret));
		return (void*)NULL;
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvRecvByTask rendezvId:0x%x; recvData:%s; recvSize:%d;",
		rendezvId,recvData,recvSize);
#endif	/* 0 */
#endif	/* 0 */
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"NOW CALL IPC_RendezvReply rendezvId:0x%x;",rendezvId);
	if((ret = IPC_RendezvReply(rendezvId, (void*)replyData, replySize)) != RPC_SUCCESS) {
		(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_ERROR,"ERROR IN IPC_RendezvReply RET:%s",RPC_LogConvertResultCode(ret));
		return (void*)NULL;
	}
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RETURN IPC_RendezvReply SUCCESS");
	(void)RPC_LogPrint(RPC_LOG_MODULE_RENDEZV,RPC_LOG_LEVEL_DEBUG1,"RECEIVER END");
	return (void*)NULL;
}

#endif	/* DEBUG */

#endif	/* USE_RENDEZVOUS */
