/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.event;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.ecs.xhtml.comment;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTEvent;
import com.aimluck.eip.cayenne.om.portlet.OriTEventMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.event.util.EventUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * イベントのフォームデータを管理するクラスです。 <BR>
 *
 */
public class EventFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(EventFormData.class.getName());

  /** イベントID */
  private ALNumberField event_id;

  /** イベント名 */
  private ALStringField event_name;

  /** コメント */
  private ALStringField event_comment;

  /** 締切日  */
  private ALDateTimeField event_closing;

  /** 締切時 */
  private ALStringField closing_date_hour;

  /** 締切分 */
  private ALStringField closing_date_minute;

  /** メンバーリスト */
  private List<ALEipUser> memberList;




  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

	// イベントID
	event_id = new ALNumberField();

	// イベント名
    event_name = new ALStringField();
    event_name.setFieldName("タイトル");
    event_name.setTrim(true);

    // イベントコメント
    event_comment = new ALStringField();
    event_comment.setFieldName("内容");
    event_comment.setTrim(false);

    // 締切日
    event_closing = new ALDateTimeField(ALDateTimeField.DEFAULT_DATE_TIME_FORMAT);
    event_closing.setFieldName("締切日時");

    // 締切時
    closing_date_hour = new ALStringField();

    // 締切分
    closing_date_minute = new ALStringField();

    // メンバーリスト
    memberList = new ArrayList();
  }


  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      try {

    	 // 日時
        event_closing.setValue(Calendar.getInstance().getTime());

        // メンバーリスト
        String[] str = rundata.getParameters().getStrings("member_to");
        if (str != null && str.length > 0) {
          SelectQuery query = new SelectQuery(TurbineUser.class);
          Expression exp =
            ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, str);
          query.setQualifier(exp);
          memberList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
        }

      } catch (Exception ex) {
        logger.error("Exception", ex);
      }
    }
    return res;
  }


  /**
   * イベントの各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {

	// イベント名必須項目
	event_name.setNotNull(true);
	// イベント名文字数制限
	event_name.limitMaxLength(100);
	// イベントcomment文字数制限
	event_comment.setNotNull(true);
	event_comment.limitMaxLength(10000);

  }

  /**
   * イベントのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {

	 // イベント名
	 event_name.validate(msgList);
	 // イベントコメント
	 event_comment.validate(msgList);
	 // 締切日
	 event_closing.validate(msgList);

     // 回答メンバーが選択されていない場合エラーメッセージを返す
     if (memberList.size() == 0) {
       msgList.add("『 <span class='em'>回答メンバー</span> 』は必ず指定してください。");
     }


    return (msgList.size() == 0);
  }


  /**
   * イベントをデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // オブジェクトモデルを取得
      OriTEvent event = EventUtils.getOriTEvent(rundata, context);
      if (event == null) {
    	  return false;
      }

      // イベントIDを取得
      List<OriTEventMap> eventMap = EventUtils.getOriTEventMap(rundata, context);
      if (eventMap == null) {
    	  return false;
      }


      // イベントマップを削除
      Database.deleteAll(eventMap);
      // イベントを削除
      Database.delete(event);

      // コミット
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        event.getOriTEventId(),
        ALEventlogConstants.PORTLET_TYPE_MEMO,
        event.getEventName());

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * Memoをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */

  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTEvent event = Database.create(OriTEvent.class);

      // 現在の時刻を設定
      Date now = Calendar.getInstance().getTime();

      // イベント作成者
      event.setCreateId(Integer.valueOf(ALEipUtils.getUserId(rundata)));
      // イベント名
      event.setEventName(event_name.getValue());
      // イベントコメント
      event.setComment(event_comment.getValue());
      // 締切日時
      SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm");
      Date date = format.parse(rundata.getParameters().getString("event_closing").replaceAll("/", "-") + " " + closing_date_hour.getValue() + ":" + closing_date_minute.getValue());

      event.setClosingDate(date);

      // 作成日
      event.setCreateDate(now);
      // 更新日
      event.setUpdateDate(now);


      // 回答者を追加する
      for (int i = 0 ; i < memberList.size(); i++) {

    	  // 親オブジェクトを生成
    	  OriTEventMap eventMap = Database.create(OriTEventMap.class);

    	  eventMap.setOriTEvent(event);

    	  int memberId = (int) (memberList.get(i)).getUserId().getValue();
    	  eventMap.setUserId(memberId);
    	  // ステータスを未選択「0」
    	  eventMap.setStatus(EventUtils.STATUS_UNSELECTED);
    	  // 作成日
    	  eventMap.setCreateDate(now);
    	  // 更新日
    	  eventMap.setUpdateDate(now);

      }

      // イベントを登録
      Database.commit();

      // 新規フラグを立てる
      ALEipUtils.setTemp(rundata, context, EventUtils.NEW_FLAG, event.getOriTEventId().toString());

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
    	event.getOriTEventId(),
        ALEventlogConstants.PORTLET_TYPE_EVENT,
        event.getEventName());

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }


  /**
   * データベースに格納されているイベントを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */

  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      OriTEvent event = EventUtils.getOriTEvent(rundata, context);
      if (event == null) {
    	  return false;
      }

      // 現在の時刻
      Date now = Calendar.getInstance().getTime();

      // イベント名
      event.setEventName(event_name.getValue());
      // イベント内容
      event.setComment(event_comment.getValue());
      // イベント締切日時
      SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm");
      Date date = format.parse(rundata.getParameters().getString("event_closing").replaceAll("/", "-") + " " + closing_date_hour.getValue() + ":" + closing_date_minute.getValue());
      event.setClosingDate(date);

      // 更新時間
      event.setUpdateDate(now);

      List<ALEipUser> selectMember = new ArrayList<ALEipUser>();
      selectMember.addAll(memberList);

      List<Integer> new_member = new ArrayList<Integer>();
      for (int z = 0; z < selectMember.size(); z++) {
          int memberId = (int) (selectMember.get(z)).getUserId().getValue();

          new_member.add(memberId);
        }

      List<OriTEventMap> citem2 = event.getOriTEventMap();
      for (int n = 0; n < citem2.size(); n++) {
        OriTEventMap choiceitem = citem2.get(n);
        int answer = choiceitem.getUserId();

        if (new_member.indexOf(answer) == -1) {
          Database.delete(choiceitem);
        }
      }

      List count = new ArrayList();
      for (int a = 0; a < selectMember.size(); a++) {

        int memberId = (int) (selectMember.get(a)).getUserId().getValue();

        OriTEventMap itemlist = EventUtils.getOriTEventMap(rundata, context, memberId);

        if (itemlist != null) {
          count.add(selectMember.get(a));
        }
      }

      memberList.removeAll(count);

      if (memberList.size() != 0){
    	 for(int i = 0; i < memberList.size(); i++) {
    		 OriTEventMap eventMap = Database.create(OriTEventMap.class);
    		 eventMap.setOriTEvent(event);
    		 eventMap.setUserId((int) (memberList.get(i)).getUserId().getValue());
    		 eventMap.setStatus(EventUtils.STATUS_UNSELECTED);
    		 eventMap.setCreateDate(now);
    		 eventMap.setUpdateDate(now);
    	 }
      }

      // データベースコミット
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        event.getOriTEventId(),
        ALEventlogConstants.PORTLET_TYPE_MEMO,
        event.getEventName());
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }




/**
 * イベント名を取得する
 * @return event_name
 */
public ALStringField getEventName() {
	return event_name;
}

/**
 * イベントコメントを取得する。
 * @return event_comment
 */
public ALStringField getEventComment() {
	return event_comment;
}

/**
 * @return event_closing
 */
public ALDateTimeField getEventClosing() {
	return event_closing;
}


/**
 * @return closing_date_hour
 */
public ALStringField getClosingDateHour() {
	return closing_date_hour;
}

/**
 * @return closing_date_minute
 */
public ALStringField getClosingDateMinute() {
	return closing_date_minute;
}



@Override
protected boolean loadFormData(RunData rundata, Context context, List<String> msgList) {
    try {
    	// オブジェクトの取得
    	OriTEvent event = EventUtils.getOriTEvent(rundata, context);
    	if (event == null) {
    		return false;
    	}

    	// イベントID
    	event_id.setValue(event.getOriTEventId());
    	// イベント名
    	event_name.setValue(event.getEventName());
    	// イベント内容
    	event_comment.setValue(event.getComment());

    	// オブジェクト取得
    	List<OriTEventMap> eventMap = EventUtils.getOriTEventMap(rundata, context);
    	if(eventMap == null) {
    		return false;
    	}

        // メンバーのリスト
        List<Integer> users = new ArrayList<Integer>();

    	for(int i = 0; i < eventMap.size(); i++) {
    		OriTEventMap map = eventMap.get(i);
    		users.add(map.getUserId().intValue());
    	}

    	if (users.size() > 0) {
            SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
            Expression exp =
              ExpressionFactory.inDbExp(TurbineUser.USER_ID_PK_COLUMN, users);
            query.setQualifier(exp);
            memberList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
    	}

    	// 締切日
    	event_closing.setValue(event.getClosingDate());
    	// 締切時
    	closing_date_hour.setValue(String.valueOf(event.getClosingDate().getHours()));
    	// 締切分
    	closing_date_minute.setValue(String.valueOf(event.getClosingDate().getMinutes()));


    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
}


/**
 * グループメンバーを取得します。
 *
 * @return
 */
public List getMemberList() {
  return memberList;
}

/**
 * @return event_id
 */
public ALNumberField getEventId() {
	return event_id;
}

/**
 * @param event_id セットする event_id
 */
public void setEventId(ALNumberField event_id) {
	this.event_id = event_id;
}



}
