/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.opinionbox;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBox;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBoxMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.modules.actions.opinionbox.OpinionBoxAction;
import com.aimluck.eip.modules.screens.OpinionBoxDetailScreen;
import com.aimluck.eip.modules.screens.OpinionBoxFormJSONScreen;
import com.aimluck.eip.opinionbox.util.OpinionBoxUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.social.ALActivityService;
import com.aimluck.eip.services.social.model.ALActivityPutRequest;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 掲示板返信のフォームデータを管理するクラスです。 <BR>
 *
 */
public class OpinionBoxReplyFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(OpinionBoxReplyFormData.class.getName());

  /** トピック名 */
  private ALStringField opinionbox_name;

  /** メモ */
  private ALStringField note;

  private int uid;

  private String orgId;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /** 他ユーザーの作成したトピックの編集権限 */
  private boolean hasAclUpdateOpinionBoxOthers;

  /** 記事への返信権限 */
  private boolean hasAclInsertOpinionBoxReply;

  /** 記事への返信権限 */
  private boolean hasAclDeleteOpinionBoxReply;

  /** 他ユーザーの作成したトピックの削除権限 */
  private boolean hasAclDeleteOpinionBoxOthers;

  /** <code>login_user</code> ログインユーザー */
  private ALEipUser login_user;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    login_user = ALEipUtils.getALEipUser(rundata);
    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclDeleteOpinionBoxOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_OTHER,
        ALAccessControlConstants.VALUE_ACL_DELETE);

    hasAclUpdateOpinionBoxOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_OTHER,
        ALAccessControlConstants.VALUE_ACL_UPDATE);

    hasAclInsertOpinionBoxReply =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_REPLY,
        ALAccessControlConstants.VALUE_ACL_INSERT);
    hasAclDeleteOpinionBoxReply =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_REPLY,
        ALAccessControlConstants.VALUE_ACL_DELETE);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {
    // トピック名
    opinionbox_name = new ALStringField();
    opinionbox_name.setFieldName("返信タイトル");
    opinionbox_name.setTrim(true);
    // メモ
    note = new ALStringField();
    note.setFieldName("内容");
    note.setTrim(false);

  }

  /**
   * 掲示板の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // メモ必須項目
    note.setNotNull(true);
    // メモの文字数制限
    note.limitMaxLength(10000);
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param opinionboxList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> opinionboxList) {
    // トピック名
    opinionbox_name.validate(opinionboxList);
    // メモ
    note.validate(opinionboxList);
    return (opinionboxList.size() == 0);
  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param opinionboxList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> opinionboxList) {
    return false;
  }

  /**
   * 返信記事をデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param opinionboxList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> opinionboxList) throws ALPageNotFoundException,
      ALDBErrorException {
    try {
      String opinionid = rundata.getParameters().getString("opinionbox_reply_id");

      // オブジェクトモデルを取得
      OriTOpinionBox opinionbox;

      if (this.hasAclDeleteOpinionBoxReply) {
        opinionbox =
          OpinionBoxUtils.getOriTOpinionBoxReply(rundata, context, opinionid, true);
      } else {
        opinionbox =
          OpinionBoxUtils.getOriTOpinionBoxReply(rundata, context, opinionid, false);
      }
      if (opinionbox == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[OpinionBoxopinionboxReplyFormData] Not found ID...");
        throw new ALPageNotFoundException();
      }

      // 返信記事を削除
      // 添付ファイルはカスケードで自動的に削除される．
      Database.delete(opinionbox);
      Database.commit();

//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        opinionbox.getOpinionId(),
//        ALEventlogConstants.PORTLET_TYPE_OPINIONBOX,
//        opinionbox.getOpinionBoxName());

    } catch (Exception e) {
      Database.rollback();
      logger.error("[OpinionBoxopinionboxReplyFormData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * トピックをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param opinionboxList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> opinionboxList) throws ALPageNotFoundException,
      ALDBErrorException {
    try {
      // オブジェクトモデルを取得
      OriTOpinionBox parentopinionbox =
        OpinionBoxUtils.getOriTOpinionBoxParentReply(rundata, context, false);
      if (parentopinionbox == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[OpinionBoxopinionboxReplyFormData] Not found ID...");
        throw new ALPageNotFoundException();
      }
      Date updateDate = Calendar.getInstance().getTime();

      // 新規オブジェクトモデル
      OriTOpinionBox opinionbox = Database.create(OriTOpinionBox.class);

      // 報告書名
      opinionbox.setOpinionBoxName(opinionbox_name.getValue());
      // ユーザーID
      opinionbox
        .setUserId(Integer.valueOf((int) login_user.getUserId().getValue()));
      // メモ
      opinionbox.setNote(note.getValue());
      // 作成日
      opinionbox.setCreateDate(updateDate);
      // 更新日
      opinionbox.setUpdateDate(Calendar.getInstance().getTime());
      // ユーザーID
      opinionbox.setTurbineUser(ALEipUtils.getTurbineUser(ALEipUtils
        .getUserId(rundata)));
      // 匿名フラグ
      opinionbox.setTokumeiFlg(0);
      // 親トピック ID
      opinionbox.setParentId(parentopinionbox.getOpinionId());

      // ファイルをデータベースに登録する．

      Database.commit();

      // 更新情報を送るユーザーを取得する
      List<ALEipUser> recipientList = new ArrayList<ALEipUser>();
      recipientList = getRecipientList(rundata, context);

      List<String> recipients = new ArrayList<String>();
      for (ALEipUser user : recipientList) {
    	  recipients.add(ALEipUtils.getTurbineUser((int)user.getUserId().getValue()).getLoginName().toString());
      }


      // 通知先に通知
      String userName = "匿名希望";
      String loginName = "";
      if (ALEipUtils.getUserId(rundata) == parentopinionbox.getUserId().intValue()) {
	      if ((int)parentopinionbox.getTokumeiFlg().intValue() == 0) {
	    	  userName = ALEipUtils.getUserFullName(parentopinionbox.getUserId().intValue());
	    	  loginName = login_user.getName().getValue();
	      }
      } else {
    	  userName = ALEipUtils.getUserFullName(Integer.valueOf(login_user.getUserId().toString()));
    	  loginName = login_user.getName().getValue();
      }

      ALActivityService.create(new ALActivityPutRequest().withAppId(
        "OpinionBox").withUserId(uid).withLoginName(loginName)
        .withPortletParams("?template=OpinionBoxDetailScreen&entityid="+parentopinionbox.getOpinionId())
        .withRecipients(recipients)
        .withTitle(userName + "さん より 目安箱 「"+ parentopinionbox.getOpinionBoxName() + "」 に返信がありました。")
        .withPriority(1f)
        .withExternalId(String.valueOf(parentopinionbox.getOpinionId())));


//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        opinionbox.getOpinionId(),
//        ALEventlogConstants.PORTLET_TYPE_OPINIONBOX,
//        opinionbox.getOpinionBoxName());

//      try {
//        // メール送信
//        List<ALEipUserAddr> destMemberList =
//          ALMailUtils.getALEipUserAddrs(recipientList, ALEipUtils
//            .getUserId(rundata), false);
//
//        String subject = "[" + ALOrgUtilsService.getAlias() + "]目安箱への返信";
//        String orgId = Database.getDomainName();
//
//        // パソコン、携帯電話へメールを送信
//        List<ALAdminMailMessage> messageList =
//          new ArrayList<ALAdminMailMessage>();
//        for (ALEipUserAddr destMember : destMemberList) {
//          ALAdminMailMessage message = new ALAdminMailMessage(destMember);
//          message.setPcSubject(subject);
//          message.setPcBody(OpinionBoxUtils.createReplyMsgForPc(
//            rundata,
//            opinionbox,
//            parentopinionbox));
//          message.setCellularSubject(subject);
//          message.setCellularBody(OpinionBoxUtils.createReplyMsgForCellPhone(
//            rundata,
//            opinionbox,
//            parentopinionbox,
//            destMember.getUserId()));
//          messageList.add(message);
//        }
//
//        ALMailService.sendAdminMailAsync(new ALAdminMailContext(
//          orgId,
//          ALEipUtils.getUserId(rundata),
//          messageList,
//          ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_OPINIONBOX)));
//
//      } catch (Exception ex) {
//        logger.error("Exception", ex);
//        return false;
//      }

    } catch (Exception e) {
      logger.error("[OpinionBoxopinionboxReplyFormData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * 当該投稿の更新通知ユーザーを習得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   * @throws ALDBErrorException
   * @throws ALPageNotFoundException
   */
  private List<ALEipUser> getRecipientList(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    List<Integer> userIdList = new ArrayList<Integer>();
    Integer loginUserId = ALEipUtils.getUserId(rundata);
    OriTOpinionBox parenttopic = OpinionBoxUtils.getOriTOpinionBox(rundata, context);

    // 通知先をすべて取得する
    SelectQuery<OriTOpinionBoxMap> mapQuery = Database.query(OriTOpinionBoxMap.class);
    Expression mapExp =
      ExpressionFactory.matchExp(OriTOpinionBoxMap.OPINION_ID_PROPERTY, parenttopic
        .getOpinionId());
    mapQuery.setQualifier(mapExp);
    List<OriTOpinionBoxMap> mapList = mapQuery.fetchList();
    for (OriTOpinionBoxMap map : mapList) {
      Integer userId = map.getUserId();
      if (!userId.equals(loginUserId) && !userIdList.contains(userId)) {
        userIdList.add(map.getUserId());
      }
    }

    // 関連トピックをすべて取得する
    SelectQuery<OriTOpinionBox> topicQuery = Database.query(OriTOpinionBox.class);
    Expression topicExp =
      ExpressionFactory.matchExp(OriTOpinionBox.PARENT_ID_PROPERTY, parenttopic
        .getOpinionId());
    topicQuery.setQualifier(topicExp);

    // 全ての返信者のIDを取得する
    List<OriTOpinionBox> topicList = topicQuery.fetchList();
    topicList.add(parenttopic);
    for (OriTOpinionBox topic : topicList) {
      Integer userId = topic.getUserId();
      if (!userId.equals(loginUserId) && !userIdList.contains(userId)) {
        userIdList.add(userId);
      }
    }

    // ユーザーIDからユーザー情報を取得する。
    SelectQuery<TurbineUser> userQuery = Database.query(TurbineUser.class);
    Expression userExp =
      ExpressionFactory.inDbExp(TurbineUser.USER_ID_PK_COLUMN, userIdList);
    userQuery.setQualifier(userExp);
    return ALEipUtils.getUsersFromSelectQuery(userQuery);

  }

  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param opinionboxList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> opinionboxList) {
    return false;
  }

  /**
   * トピック詳細表示ページからデータを新規登録します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doInsert(ALAction action, RunData rundata, Context context) {
    try {
      if (!doCheckSecurity(rundata, context)) {
        return false;
      }
      init(action, rundata, context);

      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_INSERT);

      action.setMode(ALEipConstants.MODE_INSERT);
      List<String> opinionboxList = new ArrayList<String>();
      setValidator();
      boolean res =
        (setFormData(rundata, context, opinionboxList) && validate(opinionboxList) && insertFormData(
          rundata,
          context,
          opinionboxList));
      if (!res) {
        action.setMode(ALEipConstants.MODE_NEW_FORM);
        setMode(action.getMode());
      }
      if (action instanceof OpinionBoxFormJSONScreen) {
        action.setResultData(this);
        action.addErrorMessages(opinionboxList);
        action.putData(rundata, context);
      } else {
        OpinionBoxAction opinionboxAction = (OpinionBoxAction) action;
        opinionboxAction.setResultDataOnOpinionBoxDetail(this);
        opinionboxAction.addErrorMessagesOnOpinionBoxDetail(opinionboxList);
        opinionboxAction.putDataOnOpinionBoxDetail(rundata, context);
      }
      return res;
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   * トピック詳細表示ページにフォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);

      // doCheckAclPermission(rundata, context,
      // ALAccessControlConstants.VALUE_ACL_INSERT);

      action.setMode("reply");
      // mode = action.getMode();
      List<String> opinionboxList = new ArrayList<String>();
      boolean res = setFormData(rundata, context, opinionboxList);
      if (action instanceof OpinionBoxDetailScreen) {
        OpinionBoxDetailScreen opinionboxAction = (OpinionBoxDetailScreen) action;
        opinionboxAction.setResultDataOnOpinionBoxDetail(this);
        opinionboxAction.addErrorMessagesOnOpinionBoxDetail(opinionboxList);
        opinionboxAction.putDataOnOpinionBoxDetail(rundata, context);
      } else {
        OpinionBoxAction opinionboxAction = (OpinionBoxAction) action;
        opinionboxAction.setResultDataOnOpinionBoxDetail(this);
        opinionboxAction.addErrorMessagesOnOpinionBoxDetail(opinionboxList);
        opinionboxAction.putDataOnOpinionBoxDetail(rundata, context);
      }
      return res;
      // } catch (ALPermissionException e) {
      // ALEipUtils.redirectPermissionError(rundata);
      // return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @param opinionboxList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> opinionboxList) throws ALPageNotFoundException,
      ALDBErrorException {

    boolean res = super.setFormData(rundata, context, opinionboxList);

    try {
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  public void setAclPortletFeature(String featureName) {
    aclPortletFeature = featureName;
  }

  /**
   * トピック名を取得します。 <BR>
   *
   * @return
   */
  public ALStringField getOpinionBoxName() {
    return opinionbox_name;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    if (aclPortletFeature == null || "".equals(aclPortletFeature)) {
      return ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_REPLY;
    } else {
      return aclPortletFeature;
    }
  }



  public boolean hasAclUpdateopinionboxOthers() {
    return hasAclUpdateOpinionBoxOthers;
  }

  /**
   * 他ユーザのトピックを削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteOpinionBoxOthers() {
    return hasAclDeleteOpinionBoxOthers;
  }

  /**
   * 報告書の返信を削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteOpinionBoxReply() {
    return hasAclDeleteOpinionBoxReply;
  }

  /**
   * 他ユーザのトピックを追加する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclInsertOpinionBoxReply() {
    return hasAclInsertOpinionBoxReply;
  }
}
