﻿using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;

namespace nack
{
    /// <summary>
    /// ファイルを検索するクラス。
    /// </summary>
    public class Find
    {
        /// <summary>
        /// ファイルを検索する。
        /// </summary>
        /// <param name="dir">検索するディレクトリ。</param>
        /// <param name="fileMatchPattern">ファイル名にマッチするパターン。</param>
        /// <param name="dirNotMatchPattern">除外するディレクトリ名のパターン。</param>
        /// <returns>見つかったファイルのIEnumerable。</returns>
        public IEnumerable<string> FindFiles(string dir, string fileMatchPattern, string dirNotMatchPattern)
        {
            var fileMatchRegex = new Regex(fileMatchPattern, RegexOptions.IgnoreCase);
            var dirNotMatchRegex = new Regex(dirNotMatchPattern, RegexOptions.IgnoreCase);

            foreach (var file in FindFiles(dir, fileMatchRegex, dirNotMatchRegex))
            {
                yield return file;
            }
        }

        /// <summary>
        /// ファイルを検索する。
        /// </summary>
        /// <param name="dir">検索するディレクトリ。</param>
        /// <param name="fileMatchRegex">ファイル名にマッチする正規表現。</param>
        /// <param name="dirNotMatchRegex">除外するディレクトリ名の正規表現。</param>
        /// <returns>見つかったファイルのIEnumerable。</returns>
        private IEnumerable<string> FindFiles(string dir, Regex fileMatchRegex, Regex dirNotMatchRegex)
        {
            foreach (var path in Directory.GetFiles(dir))
            {
                if (fileMatchRegex.IsMatch(path))
                {
                    yield return path;
                }
            }

            foreach (var path in Directory.GetDirectories(dir))
            {
                if (dirNotMatchRegex.IsMatch(path) == false)
                {
                    FindFiles(path, fileMatchRegex, dirNotMatchRegex);
                }
            }
        }
    }
}
