﻿using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;

namespace nack
{
    /// <summary>
    /// 検索結果(行単位)。
    /// </summary>
    public class GrepResult
    {
        /// <summary>
        /// 行番号。
        /// </summary>
        public int LineNo { get; set; }

        /// <summary>
        /// 行本体。
        /// </summary>
        public IList<FoundString> FoudnStringList { get; set; }
    }

    /// <summary>
    /// 検索して見つかった文字列の種別。
    /// </summary>
    public enum FoundStringType
    {
        /// <summary>
        /// マッチしなかった文字列。
        /// </summary>
        Plain, 
        
        /// <summary>
        /// マッチした文字列。
        /// </summary>
        Matched
    }

    /// <summary>
    /// 検索して見つかった文字列を表すクラス。
    /// </summary>
    public class FoundString
    {
        /// <summary>
        /// 検索して見つかった文字列の種別。
        /// </summary>
        public FoundStringType Type { get; set; }

        /// <summary>
        /// 文字列の値。
        /// </summary>
        public string Value { get; set; }
    }

    /// <summary>
    /// ファイルを読み込んで検索するクラス。
    /// </summary>
    public class Grep
    {
        /// <summary>
        /// ファイルを読み込んで検索する
        /// </summary>
        /// <param name="file">読み込むファイル。</param>
        /// <param name="fileEncoding">読み込むファイルのエンコーディング。</param>
        /// <param name="pattern">検索するパターン。</param>
        /// <param name="regexOptions">検索オプション。</param>
        /// <returns>検索結果。</returns>
        public IEnumerable<GrepResult> GrepFile(string file, Encoding fileEncoding, string pattern, RegexOptions regexOptions)
        {
            Regex regex = new Regex(pattern, regexOptions);
            GrepResult gres = null;

            using (var reader = new StreamReader(file, fileEncoding))
            {
                for (int lineno = 1; reader.Peek() > -1; lineno++)
                {
                    var pos = 0;
                    var found = false;
                    var line = reader.ReadLine();
                    var mc = regex.Matches(line);

                    foreach (Match m in mc)
                    {
                        if (found == false)
                        {
                            gres = new GrepResult() { LineNo = lineno, FoudnStringList = new List<FoundString>() };
                            found = true;
                        }

                        var pre = line.Substring(pos, m.Index - pos);
                        var pat = m.Value;
                        pos = m.Index + m.Length;

                        gres.FoudnStringList.Add(new FoundString() { Type = FoundStringType.Plain, Value = pre });
                        gres.FoudnStringList.Add(new FoundString() { Type = FoundStringType.Matched, Value = pat });
                    }

                    if (found)
                    {
                        var rest = line.Substring(pos);
                        gres.FoudnStringList.Add(new FoundString() { Type = FoundStringType.Plain, Value = rest });
                        yield return gres;
                    }
                }
            }
        }
    }
}
