# encoding: utf-8
require 'win32ole'
require 'zipruby'
require 'uconv'
require 'fileutils'
require 'facets/pathname'

require 'rumix/registry'
require 'rumix/const'
require 'rumix/config'

module Rumix
	class OSInterface
		attr_accessor :installed_file_list_output

		def initialize(config, log_io = nil)
			@config = config
			@log_io = log_io

			@shell = WIN32OLE.new('Wscript.Shell') 
			@maked_dir_path_list = []
			@installed_file_list_output = nil

		end

		# 実際の変更を行う権限（オペレーションレベル）があればtrue
		def real_operating?
			@config.real_operating?
		end

		# システムにかかわる変更を行う権限（オペレーションレベル）があればtrue
		def system_changable?
			@config.system_changable?
		end

		# ログファイルが設定されていれば、そのログファイルへの書き込みを行う
		def log(str)
			@log_io.puts str if @log_io
		end

		# スタートメニューのフォルダパスを取得
		def start_menu_dir
			if system_changable? then
				File.join(@shell.SpecialFolders('AllUsersStartMenu'), Uconv.u8tosjis('プログラム/Rumix'))
			else
				File.join(@config.dest_dir, '.virtual-environment/StartMenu/Rumix')
			end
		end

		# FileUtils.cdのログ出力あり＆dryrun考慮版
		def cd(to)
			log "[cd] #{to}"
			if real_operating? then
				FileUtils.cd(to) do
					yield
				end
			else
				yield
			end
			log "[cd] -"
		end
		

		# NYAOS用のRumix向け設定ファイル（_nya_rumix）内の変数を書き換える
		def rewrite_nya_rumix(key, value)
				nya_config_path = File.join(@config.dest_dir, 'shell/nyaos/_nya_rumix')
				backup_file('shell/nyaos/_nya_rumix')
				nya_config_body = File.read(nya_config_path)

				nya_config_body.gsub!("@#{key}@", value.to_s.gsub('\\', '\\\\\\\\'))
				
				if real_operating? then
					open(nya_config_path, 'w'){|f|
						f.write(nya_config_body)
					}
				end
				log "rewrite #{nya_config_path} (@#{key}@ => #{value.to_s.gsub('\\', '\\\\')})"
		end

		# NYAOS用のRumix向け設定ファイル（_nya_rumix）内の不要なセクション除去
		def remove_nya_rumix_section(section_name)
			nya_config_path = File.join(@config.dest_dir, 'shell/nyaos/_nya_rumix')
			backup_file('shell/nyaos/_nya_rumix')
			nya_config_body = File.read(nya_config_path)

			nya_config_body.gsub!(/^=BEGIN\s*#{section_name}\s*$.+?^=END\s*#{section_name}\s*$/im, '')
			
			if real_operating? then
				open(nya_config_path, 'w'){|f|
					f.write(nya_config_body)
				}
			end
			log "rewrite #{nya_config_path} (@#{key}@ => #{value.to_s.gsub('\\', '\\\\')})"
		end

		# 指定した名前のgemを、packageディレクトリから探してインストールする
		def install_gems(gem_names, options = {}, &block)
			#no_document = options[:no_document] || false
			no_document = true

			gem_file_paths = gem_names.map{|x|  @config.get_gem_file_path(x)}
			doc_option = (no_document ? '--no-rdoc --no-ri' : '--rdoc --ri')

			@config.ruby_configs.each do |ruby_conf|
				cmd = "#{ruby_conf.ruby_gem_command} install --local #{doc_option} #{gem_file_paths.join(' ')}"
				sh(cmd) or raise "installing gems is failed - #{gem_names.join(', ')} to #{ruby_conf.ruby_short_description}"
			end
		end
		
		# zipファイルの内容物を、指定したディレクトリへインストールする
		def install_from_zip(src_zip, dest_dir, &block)
			list = []
			
			real_zip_path = @config.get_package_file_path(src_zip)
			Zip::Archive.open(real_zip_path){|zip|
				# contentディレクトリがあるかどうかをチェック
				zip.each do |file|
					list << file.name
				end
				
				from_content_dir = list.include?('content/')

				zip.each_with_index do |src_file, i|
					dest_name = nil
					if not from_content_dir or (dest_name = src_file.name[/^content\/(.+)/, 1]) then
						dest_name ||= src_file.name
						dest_pathname = Pathname[dest_dir] / dest_name
						
						if src_file.directory? then
							mkpath(dest_pathname)
						else
							log "#{real_zip_path} : #{src_file.name} -> #{dest_pathname.cleanpath.to_s}"
							mkpath(dest_pathname.dirname)
							
							extract_file(src_file, dest_pathname)
						end
						block.call(:on_child_installed, i) if block

						# インストール済みファイルの一覧に出力
						if @installed_file_list_output then
							@installed_file_list_output.puts(dest_pathname.cleanpath.to_s)
						end
					end
				end
			
			}

		end
		
		def extract_file(zip_file, dest)
			# 上書きが必要ないと判断した場合には処理をスキップ
			if not @config.force_overwriting? and not File.basename(dest) == '_nya_rumix' and File.exist?(dest) and zip_file.mtime <= File.mtime(dest) then
				log "skipped." 
				return
			end
			
			if real_operating? then
				open(dest, 'wb'){|out|
					while (got = zip_file.read(ZIP_CHUNK_SIZE)) do
						written_size = out.write(got)
					end
				}
				File.utime(File.atime(dest), zip_file.mtime, dest)
			end
		end
		
		def remove_file(path)
			log "[rm] #{path}"
			if real_operating? then
				FileUtils.rm(path.to_s)
			end
		end
		
		def remove_dir(dir_path)
			log "[rmdir] #{dir_path}"
			if real_operating? then
				FileUtils.remove_entry_secure(dir_path)
			end
		end
		
		def mkpath(dir_path)
			if not File.exist?(dir_path) and not @maked_dir_path_list.include?(dir_path) then
				log "[mkpath]  #{dir_path}"
				if real_operating? then
					FileUtils.mkpath(dir_path.to_s)
				else
					# On no-op mode
					@maked_dir_path_list << dir_path.to_s
				end
			end
		end
		alias make_path mkpath
		
		
		def install_tool(package_name, script_name)
			
			log_name = "../../#{package_name}_install.log"
			#old = ENV['LOG_FILE']
			#ENV['LOG_FILE'] = "../../#{package_name}_install.log"
			ruby_path = File.join(@config.dest_dir, 'ruby/bin/ruby.exe')
			old_ruby_opts = ENV['RUBYOPTS']
			ENV['RUBYOPTS'] = nil
			
			temp_dir = 'temp/'
			dest_dir = File.join(temp_dir, package_name)
			install_from_zip("#{package_name}.zip", dest_dir)
			
			cd(dest_dir) do
				log "[tool install] #{ruby_path} #{script_name} (logfile = #{log_name})"
				if @config.real_operating then
					# 標準出力と標準エラー出力を、ログファイルに変更するため
					# tool_install.rb を介して実行
					success = system("#{ruby_path} ../../res/script/tool_install.rb #{ruby_path} #{script_name} #{log_name} #{package_name}")
					
					if success then
						log "#{package_name} install success."
					else
						log "#{package_name} install failed! (code=#{$?}, see the log)"
					end
				
				
					#result = %x(../../tool/doslog.exe #{ruby_path} #{script_name})
					#code = nil
					
					# doslog.exeの出力を解析して、終了コードをチェック
					#if result =~ Regexp.compile(Uconv.u8tosjis('^終了コード＝(\d+)$'), nil, 's') then
					#	code = $1.to_i
					#	if code == 0 then
					#		log "#{package_name} install success."
					#	else
					#		log "#{package_name} install failed! (code=#{code})"
					#	end
					#else
					#	log "#{package_name} install failed! (code=?)"
					#end
				end
			end
			
			#ENV['LOG_FILE'] = old
			ENV['RUBYOPTS'] = old_ruby_opts
			
			remove_dir(dest_dir)
			
			
		end

		# 指定したファイルが存在すればバックアップする
		def backup_file(src_relative_file_path)
			timestr = @config.timestamp.strftime('%Y%m%d%H%M%S')
			backup_dir_path = File.join(@config.dest_dir, ".rumix-inst/backup\\#{timestr}")
			
			src_path = File.join(@config.dest_dir, src_relative_file_path)
			dest_path = File.join(backup_dir_path, src_relative_file_path)
			
			if File.exist?(src_path) then
				if real_operating? then
					mkpath(File.dirname(dest_path))
					FileUtils.cp(src_path, dest_path)
				end
				log "[BACKUP] #{src_path} -> #{dest_path}"
			end
		end

		# スタートメニューのRumixフォルダに、指定した名前のショートカットを作成
		def create_start_menu_shortcut(name, target_path, properties = {})
			dir_path = start_menu_dir
			lnk_path = File.join(dir_path, Uconv.u8tosjis(name) + '.lnk')
			mkpath(File.dirname(lnk_path))

			target_path = Rumix.winpath(target_path)
			log "[shortcut] #{lnk_path} -> #{target_path}"
		
			if real_operating? then
				shortcut = @shell.CreateShortcut(Rumix.winpath(lnk_path))
				shortcut.TargetPath = target_path
				properties.each do |key, value|
					shortcut.setproperty(key.to_s, value)
				end
				shortcut.Save
			end

		end

		def sh(command_string)
			log("[shell command] #{command_string}")
			puts "[shell command] #{command_string}"
			if real_operating? then
				result = system(command_string)
			else
				result = true
			end

			if result then
				log("  => OK")
				puts "  => OK"
			else
				log("  => FAILED!")
				puts "  => FAILED!"
				raise "shell command execution failed - '#{command_string}'"
			end

			return result
		end

		# インストール先パスの情報をレジストリに書き込む
		def write_install_path_to_registry
			writing_install_path = Rumix.winpath(@config.dest_dir)
			if system_changable? then
				reg = Rumix::Registry.load
				reg.install_path = writing_install_path
				reg.store or raise 'failure to store registry'
			end
			log "[WRITE REGISTRY] InstallPath <= #{writing_install_path}"

		end

		# 前回のインストール先のパスをレジストリから取得（書き込まれていなければnilを返す）
		def read_install_path_from_registry
			reg = Rumix::Registry.load
			log "[READ REGISTRY] InstallPath => #{reg.install_path}"
			reg.install_path
		end

		# 指定した名前の環境変数を取得
		def get_env(key)
			@shell.Environment.item(key)
		end

		# 環境変数を指定した値に設定する
		def set_env(key, value)
			log "Set ENV['#{key}'] = #{value}"
			if system_changable? then
				@shell.Environment.setproperty('item', key, value)
			elsif real_operating? then
				virtual_env_path = File.join(@config.dest_dir, '.virtual-environment/env-writing.yml')
				mkpath(File.dirname(virtual_env_path))

				open(virtual_env_path, 'a'){|f|
					data = {'timestamp' => @config.timestamp, 'key' => key, 'value' => value}
					f.puts data.to_yaml
				}
			end
		end
	end


end