# encoding: utf-8

require 'rubyplus'
require 'ropt'
require 'ini'
require 'yaml'
require 'facets/kernel/try'
require 'facets/pathname'

require 'rumix/const'
require 'rumix/function'

module Rumix
	class Config
		include RubyType
	
		attr_reader :timestamp
		attr_accessor :src_dir, :dest_dir, :ext_dest_type, :ext_dest_dir_manual
		attr_accessor :operation_level
		attr_accessor :rumix_edition_name, :ruby_configs
		attr_accessor :man_type, :shell_type, :installing_tool_ids
		attr_accessor :selected_ruby_type
		bool_attr_accessor :add_path_env, :force_overwriting, :add_start_menu, :no_rdoc_and_no_ri, :usbrumix_mode
		
		def initialize(dest_dir = nil)
			clear
			@dest_dir = dest_dir
		end
		
		public

		def dest_dir_pathname
			Pathname[dest_dir]
		end

		def load_ini_file(path)
			ini = INI.parse_file(path)

			@rumix_edition_name = ini['General']['EditionName']

			# インストール対象となるrubyそれぞれについて、RubyConfigオブジェクトを生成
			ruby_versions = (ini['General']['RubyVersions'] || '').split(',')
			ruby_descriptions = (ini['General']['RubyDescriptions'] || '').split(',')
			@ruby_configs = []
			ruby_versions.each_with_index do |ver_str, i|
				desc = ruby_descriptions[i] || ''
				@ruby_configs << RubyConfig.new(@dest_dir, ver_str, desc, @usbrumix_mode)
			end

			return self
		end
		
		alias package_dir src_dir
		
		def tool_list
			TOOL_LIST_COMMON
		end

		# 実際の変更を行う権限（オペレーションレベル）があればtrue
		def real_operating?
			@operation_level > OperationLevel::DRY_RUN
		end

		# システムにかかわる変更を行う権限（オペレーションレベル）があればtrue
		def system_changable?
			@operation_level > OperationLevel::SYSTEM_SAFE
		end

		# メインで使用するrubyの種類
		def main_ruby_type
			@ruby_types.first
		end
		
		def get_package_file_path(package_zip_name)
			# dependent_path = nil
			# if version_dependent_package_dir then
			# 	dependent_path = File.join(version_dependent_package_dir, package_zip_name)
			# end
			# independent_path = File.join(package_dir, package_zip_name)
			
			# if dependent_path and File.exist?(dependent_path) then
			# 	dependent_path
			# else
			# 	independent_path
			# end

			File.join(package_dir, package_zip_name)
		
		end

		def get_gem_file_path(gem_name)
			gem_pattern = File.join(package_dir, "#{gem_name}-*.gem")
			found_gem_path = Dir.glob(gem_pattern).first
			return found_gem_path
		end
		
		def clear
			@timestamp = Time.now
			@src_dir = 'package/'
			@dest_dir = nil
			@operation_level = OperationLevel::FULL
			@force_overwriting = false
			
			@ruby_configs = []
			@ext_dest_type = nil
			@ext_dest_dir_manual = nil
			@man_type = ManualType::NONE
			@shell_type = ShellType::NONE

			@add_start_menu = false
			@add_path_env = false
			@installing_tool_ids = []
			@selected_ruby_type = nil
			@no_rdoc_and_no_ri = false
			@usbrumix_mode = false
			
			return self
		end
		
		# 標準のruby種別。いずれかのrubyが選択されている場合にはそのrubyを優先
		def default_ruby_type
			selected_ruby_type || @ruby_configs.first.ruby_type
		end

		# 標準のruby設定を表すRubyConfigを返す
		def default_ruby_config
			@ruby_configs.find{|x| x.ruby_type == self.default_ruby_type}
		end

		
		def ruby_zip_path
			
			if (package_dir = version_dependent_package_dir) then
				get_package_file_path('ruby.zip')
			else
				raise "unknown ruby type : #{@ruby_type}"
			end
		end
		
		# pikをインストールする必要があるかどうかを判定
		def pik_should_be_installed?
			ruby_configs.size >= 2
		end

		def pik_dir_pathname
			Pathname[dest_dir] / 'ruby/pik'
		end

		def pik_dir
			Rumix.winpath(pik_dir_pathname)
		end

		def pik_command
			Rumix.winpath(pik_dir_pathname / 'pik.bat')
		end
		

		def devkit_types
			ruby_configs.map{|x| x.devkit_type}.uniq.sort
		end

		def devkit_zip_file_names
			devkit_types.map{|x| "devkit-#{x}.zip"}
		end

		def ruby_manual_zip_file_names
			case @man_type
			when ManualType::CHM_NEW_REMIX
				prefix = "rubyrefm-remix"
			when ManualType::CHM_NEW
				prefix = "rubyrefm"
			when ManualType::NONE
				return []
			else
				raise "unknown reference manual type - #{@man_type}"
			end
			
			names = []
			ruby_configs.each do |ruby_conf|
				case ruby_conf.ruby_type
				when RUBY_1_8
					names << "#{prefix}-1.8.7.zip"
				when RUBY_1_9
					names << "#{prefix}-1.9.3.zip"
				when RUBY_2_0_I386, RUBY_2_0_X64
					names << "#{prefix}-2.0.0.zip"
				else
					raise "unknown ruby type - #{ruby_conf.ruby_type}"
				end
			end

			return names.uniq
		end
		
		
		# 「最終確認」で表示する操作リスト
		def operation_list
			re = []

			ruby_configs.each do |ruby_conf|
				if pik_should_be_installed? and ruby_conf == self.default_ruby_config then
					re << "#{ruby_conf.ruby_short_description}のインストール (このrubyが標準で使用されます)"
				else
					re << "#{ruby_conf.ruby_short_description}のインストール"
				end
			end

			if pik_should_be_installed? then
				re << "pik #{PIK_VERSION}のインストール"
			end


			case @man_type
			when ManualType::CHM_NEW_REMIX
				re << "リファレンスマニュアル chm版リミックスのインストール"
			when ManualType::CHM_NEW
				re << "リファレンスマニュアル chm版のインストール"
			when ManualType::NONE
			else
				raise "unknown manual type - #{@man_type}"
			end
			
			case @shell_type
			when ShellType::NYAOS_CKW
				re << "NYAOS #{NYAOS_VERSION} + ckw-mod #{CKW_VERSION} のインストール"
			when ShellType::NYAOS
				re << "NYAOS #{NYAOS_VERSION} のインストール"
			when ShellType::NONE
			else
				raise "unknown shell type - #{@shell_type}"
			end
			
			@installing_tool_ids.each do |tool_id|
				found = tool_list.find{|x| x.id == tool_id}
				re << "#{found[1]} のインストール"
			end

			if add_start_menu? then
				re << "スタートメニューにRumixを追加"
			end
			
			if add_path_env? then
				re << "環境変数PATHに ruby.exe のパスを追加"
			end

			re
		end


		# インストール時にユーザーが入力した内容を、Hashオブジェクト形式で出力
		def serialize_input_state_to_hash
			ret = {}
			# インストールした日時を出力
			ret['installed_time'] = Time.now
			
			# インストールしたRubyの種別とRumixのバージョンを出力
			ret['ruby_types'] = ruby_configs.map{|x| x.ruby_type}
			ret['rumix_version_number'] = VERSION_NUMBER
			
			# 選択したマニュアルとシェルの情報を出力
			ret['manual_selected'] = @man_type
			ret['shell_selected'] = @shell_type

			# 選択したツールパッケージの情報を出力
			ret['tool_selected'] = {}
			tool_list.each do |tool|
				ret['tool_selected'][tool.id.to_s] = @installing_tool_ids.include?(tool.id)
			end
			
			# 各オプションスイッチの情報を出力
			ret['option_selected'] = {}

			ret['option_selected']['add_path_env'] = @add_path_env
			ret['option_selected']['add_start_menu'] = @add_start_menu
			ret['option_selected']['force_overwriting'] = @force_overwriting
			ret['option_selected']['no_rdoc_and_no_ri'] = @no_rdoc_and_no_ri
			ret['option_selected']['usbrumix_mode'] = @usbrumix_mode
			ret['selected_ruby_type'] = selected_ruby_type
			return ret
		end
		
		# インストール時にユーザーが入力した内容を、YAML文字列で出力
		def serialize_input_state_to_yaml
			serialize_input_state_to_hash.to_yaml
		end
		
		def parse_cli_options(argv)
			clear
			
			ropt_parsed = ROpt.parse(argv, 'd:e:fi:h', 'dest:', 'system-safe', 'dry-run', 'ext:', 'force', 'ini:', 'help', 'usbrumix')
			if ropt_parsed then
				# help
				if ropt_parsed['help'] || ropt_parsed['h'] then
					show_cli_help
					exit
				end

				dest = ropt_parsed['dest'] || ropt_parsed['d']
				unless dest then
					text = <<TEXT
エラー: インストール先のフォルダパスを指定してください。
        '--dest PATH' オプションでインストール先を指定可能です。

        例) #{Rumix.running_file_name} --dest "C:\\rumix" ruby
TEXT
					$stderr.puts(Uconv.u8tosjis(text))
					exit
				end
				@dest_dir = File.expand_path(dest)


				targets = ropt_parsed.args
				if targets.empty? then
					text = <<TEXT
エラー: インストール対象のソフトウェアを1つ以上指定してください。
        詳細は、以下のコマンドで表示されるヘルプを参照してください。

        #{Rumix.running_file_name} --help
TEXT
					$stderr.puts(Uconv.u8tosjis(text))
					exit
				end

				ini_path = ropt_parsed['ini'] || ropt_parsed['i'] || 'rumix.ini'
				unless File.readable?(ini_path) then
					text = <<TEXT
エラー: iniファイル '#{ini_path}' が読み込めませんでした。
TEXT
					$stderr.puts(Uconv.u8tosjis(text))
					exit

				end
				
				# --dry-run and --system-safe 
				if ropt_parsed['dry-run'] then
					@operation_level = OperationLevel::DRY_RUN
				elsif ropt_parsed['system-safe'] then
					@operation_level = OperationLevel::SYSTEM_SAFE
				end
				
				# --force or -f
				if ropt_parsed['force'] || ropt_parsed['f'] then
					@force_overwriting = true
				end

				# --usbrumix
				if ropt_parsed['usbrumix'] then
					@usbrumix_mode = true
				end

				load_ini_file(ini_path)
				ruby_found = false
				
				targets.each do |target|
				
					case target.downcase.delete('-_')
					when 'ruby'
						ruby_found = true
					when 'nyaosckw'
						@shell_type = ShellType::NYAOS_CKW
	
					when 'refmchm'
						@man_type = ManualType::CHM_NEW_REMIX
					when 'refmchmoriginal'
						@man_type = ManualType::CHM_NEW

					when 'pathenv'
						@add_path_env = true
					when 'startmenu'
						@add_start_menu = true

					when 'bundler', 'facets', 'ocra', 'devkit', 'yard'
						@installing_tool_ids << target.downcase.to_sym
					else
						$stderr.puts "エラー: '#{target}' は不正なターゲットです。"
						exit 1
					end
				end

				unless ruby_found then
					text = <<TEXT
エラー: 必ず1種類以上のrubyをインストールする必要があります。
TEXT
					$stderr.puts(Uconv.u8tosjis(text))
					exit

				end

				
				return self
			else
				return nil
			end
			
		end

		def show_cli_help
			help_text = <<TEXT
Rumix #{Rumix::VERSION} - Ruby Starter Package with Installer

使い方1: #{Rumix.running_file_name} [options] ターゲット1 [ターゲット2 ..]
使い方2: #{Rumix.running_file_name} [options] ターゲット1 [ターゲット2 ..]

オプション:
  -d, --dest           【必須】インストール先のフォルダパスを指定します。

      --system-safe    システムセーフモード。レジストリや環境変数の書き込み、
                       スタートメニューのショートカット作成、
                       一部コマンドの実行など、OS環境全体にかかわる
                       変更を行いません。
                       (インストール先フォルダへのファイルの書き込みは
                        通常通り行います)

      --dry-run        dry runモード。上記システムセーフモードの制限に加えて
                       さらに実際のファイル書き込みも行いません。

  -i, --ini            使用するiniファイルのパスを指定します。
                        (デフォルト: rumix.ini)

      --usbrumix       USB Rumixとしてインストールします。
                       NYAOS設定ファイルに設定するパスなど、いくつかの箇所の設定が変わります。

  -f, --force          forceモード。インストールしようとしたファイルが
                       すでにインストール先に存在する場合も
                       常にファイルを上書きします。

  -h, --help           このヘルプメッセージを表示します。

ターゲットの指定:
  インストールを行う、ソフトウェアやドキュメントの種類を
  半角スペースで区切って指定してください。以下のものが選択可能です。

  ruby               ruby (iniファイルで指定されている標準のruby)
  ruby-1.8           ruby 1.8.7
  ruby-1.9           ruby 1.9.3
  ruby-2.0           ruby 2.0.0
  devkit             Devkit

  nyaos-ckw          NYAOS & ckw-mod (コマンドラインシェル)

  refm_chm           Rubyリファレンスマニュアル (chm版リミックス)
  refm_chm_original  Rubyリファレンスマニュアル (chm版)
  refe2              ReFe2 (動作に必要なgem + ドキュメントデータ)

  facets             Ruby Facets (gem)
  bundler            Bundler (gem)
  ocra               Ocra (gem)
  yard               YARD (gem)

  path_env           PATH環境変数にインストールしたrubyのパスを追加
  start_menu         スタートメニューにショートカットを作成

例:
  #{Rumix.running_file_name} -d "c:\rumix" ruby devkit

その他:
  * ターゲット指定では、'_' 記号と '-' 記号は省略することができます。
    また、どちらの記号も同じように扱われます。
     (nyaos-ckw == nyaos_ckw == nyaosckw)

TEXT

			$stderr.puts(Uconv.u8tosjis(help_text))
		end
	end
	
	# 1種類のrubyに対応する設定オブジェクト。1つのConfigは複数のRubyConfigを持つことができる
	class RubyConfig
		include RubyType

		attr_reader :dest_dir, :ruby_type, :ruby_description, :usbrumix_mode
		def initialize(dest_dir, ruby_type, ruby_description, usbrumix_mode)
			@dest_dir = dest_dir
			@ruby_type = ruby_type
			@ruby_description = ruby_description
			@usbrumix_mode = usbrumix_mode
			validate_ruby_type
		end

		def validate_ruby_type
			case ruby_type
			when RUBY_1_8, RUBY_1_9, RUBY_2_0_I386, RUBY_2_0_X64
				return
			else
				raise "unknown installing-ruby type : #{ruby_type}"
			end
		end

		# rubyのバージョンを表す文字列
		def ruby_version
			case ruby_type
			when RUBY_1_8
				'1.8.7'
			when RUBY_1_9
				'1.9.3'
			when RUBY_2_0_I386, RUBY_2_0_X64
				'2.0.0'
			end
		end

		# rubyのプラットフォームを表す文字列
		def ruby_platform
			case ruby_type
			when RUBY_1_8, RUBY_1_9, RUBY_2_0_I386
				'i386-mingw32'
			when RUBY_2_0_X64
				'x64-mingw32'
			end
		end

		# rubyのバージョン＆プラットフォームの組み合わせを表す短縮文字列
		def ruby_short_description
			case ruby_type
			when RUBY_1_8, RUBY_1_9, RUBY_2_0_I386
				"ruby #{ruby_version}"
			when RUBY_2_0_X64
				"ruby #{ruby_version} (x64)"
			end
		end

		# rubyのインストール先ディレクトリパス（Pathname）
		def ruby_dir_pathname
			Pathname[dest_dir] / "ruby/#{ruby_version}/#{ruby_platform}"
		end
		
		# rubyのインストールされるディレクトリパス
		def ruby_dir
			Rumix.winpath(ruby_dir_pathname)
		end

		# rubyのbinディレクトリパス (Pathname)
		def ruby_bin_dir_pathname
			ruby_dir_pathname / 'bin'
		end

		# rubyのbinディレクトリパス
		def ruby_bin_dir
			Rumix.winpath(ruby_bin_dir_pathname)
		end

		# rubyのbinディレクトリパス・NYAOS設定ファイル更新用
		def ruby_bin_dir_for_nyaos
			if @usbrumix_mode then
				File.join('%NYAOSDIR%\\..\\..\\', "ruby\\#{ruby_version}\\#{ruby_platform}\\bin")
			else
				ruby_bin_dir
			end
		end

		# rubyコマンドパス（Pathnameを返す）
		def ruby_command_pathname
			ruby_bin_dir_pathname / 'ruby.exe'
		end

		# rubyコマンドパス（Stringを返す）
		def ruby_command
			Rumix.winpath(ruby_command_pathname)
		end

		# gemコマンドパス（Pathnameを返す）
		def ruby_gem_command_pathname
			ruby_bin_dir_pathname / 'gem.bat'
		end
		# gemコマンドパス（Stringを返す）
		def ruby_gem_command
			Rumix.winpath(ruby_gem_command_pathname)
		end

		# BitClustのデータベースパス
		def bitclust_database_dir_pathname
			ruby_dir_pathname / "share/bitclust/db/#{ruby_version}/"

		end
		def bitclust_database_dir
			Rumix.winpath(bitclust_database_dir_pathname)
		end

		def ruby_zip_file_name
			case ruby_type
			when RUBY_1_8
				"ruby-1.8.7.zip"
			when RUBY_1_9
				"ruby-1.9.3.zip"
			when RUBY_2_0_I386
				"ruby-2.0.0-i386.zip"
			when RUBY_2_0_X64
				"ruby-2.0.0-x64.zip"
			else
				raise "unknown ruby type - #{config.ruby_type}"
			end
		end

		def devkit_type
			case ruby_type
			when RUBY_1_8, RUBY_1_9
				DevkitType::TDM32
			when RUBY_2_0_I386
				DevkitType::MINGW64_32
			when RUBY_2_0_X64
				DevkitType::MINGW64_64
			else
				raise "unknown ruby type - #{config.ruby_type}"
			end
		end
	end

end
