/*
 * The MIT License
 *
 * Copyright 2013 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.dropbox_uploader;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Environment;
import com.dropbox.client2.DropboxAPI;
import com.dropbox.client2.android.AndroidAuthSession;
import com.dropbox.client2.exception.DropboxException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.routine_work.simple_battery_logger.dropbox_uploader.db.CsvUploadStatusDBHelper;
import org.routine_work.simple_battery_logger.dropbox_uploader.db.DBConstants;
import org.routine_work.simple_battery_logger.util.CsvFilenameFilter;
import org.routine_work.util.Log;

/**
 *
 * @author Masahiko, SAWAI <masahiko.sawai@gmail.com>
 */
public class UploadServiceUtils
{

	private static final String LOG_TAG = "simple-battery-logger-dropbox";

	public static boolean checkCanUploadCSV(Context context)
	{
		// Check network connection
		ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
		NetworkInfo activeNetworkInfo = cm.getActiveNetworkInfo();
		if (activeNetworkInfo == null || activeNetworkInfo.isConnected() == false)
		{
			Log.w(LOG_TAG, "CSV file uploading is canceled. The network is not connected.");
			return false;
		}

		// Check Wi-Fi connection
		if (PreferenceUtils.isWifiOnlyUpload(context))
		{
			if (activeNetworkInfo.getType() != ConnectivityManager.TYPE_WIFI)
			{
				Log.w(LOG_TAG, "CSV file uploading is canceled. The network is not connected to Wi-Fi.");
				return false;
			}
		}

		// Check external storage state
		String externalStorageState = Environment.getExternalStorageState();
		if (externalStorageState.equalsIgnoreCase(Environment.MEDIA_MOUNTED) == false)
		{
			Log.w(LOG_TAG, "CSV file uploading is canceled. SD card is not mounted.");
			return false;
		}

		// Check Dropbox session
		DropboxAPI<AndroidAuthSession> dropboxAPI = DropboxAppConfig.getDropboxAPI(context);
		if (dropboxAPI.getSession().isLinked() == false)
		{
			Log.w(LOG_TAG, "CSV file uploading is canceled. Dropbox account is not linked.");
			return false;
		}

		// Check CSV export directory
		String csvExportDirectoryName = PreferenceUtils.getCsvExportDirectoryName(context);
		File externalStorageDirectory = Environment.getExternalStorageDirectory();
		File csvExportDirPath = new File(externalStorageDirectory, csvExportDirectoryName);

		if (csvExportDirPath.exists() == false)
		{
			Log.w(LOG_TAG, "CSV file uploading is canceled. CSV export directory is not exist.");
			return false;
		}

		if (csvExportDirPath.isDirectory() == false)
		{
			Log.w(LOG_TAG, "CSV file uploading is canceled. CSV export directory is not directory.");
			return false;
		}

		return true;
	}

	public static void uploadOneCsvFile(Context context, File csvFile) throws DropboxException, IOException
	{
		Log.v(LOG_TAG, "Hello");

		// Upload csv file
		String deviceName = PreferenceUtils.getDeviceName(context);
		String csvFileName = csvFile.getName();
		String uploadFilePath = "/" + deviceName + "/" + csvFileName;
		Log.d(LOG_TAG, "uploadFilePath => " + uploadFilePath);
		FileInputStream csvFileInputStream = null;
		try
		{
			csvFileInputStream = new FileInputStream(csvFile);
			DropboxAPI<AndroidAuthSession> dropboxAPI = DropboxAppConfig.getDropboxAPI(context);
			DropboxAPI.Entry uploadedFileEntry = dropboxAPI.putFile(uploadFilePath, csvFileInputStream, csvFile.length(), null, null);
			Log.d(LOG_TAG, "uploadedFileEntry => " + uploadedFileEntry);
		}
		finally
		{
			if (csvFileInputStream != null)
			{
				csvFileInputStream.close();
			}
		}

		Log.v(LOG_TAG, "Bye");
	}

	public static File getCsvExportDirectory(Context context)
	{
		String csvExportDirectoryName = PreferenceUtils.getCsvExportDirectoryName(context);
		Log.d(LOG_TAG, "csvExportDirectoryName => " + csvExportDirectoryName);

		File externalStorageDirectory = Environment.getExternalStorageDirectory();
		File exportDir = new File(externalStorageDirectory, csvExportDirectoryName);
		return exportDir;
	}

	public static List<CsvUploadStatus> getCsvUploadStatusList(Context context)
	{
		List<CsvUploadStatus> csvUploadStatusList = new ArrayList<CsvUploadStatus>();
		Log.v(LOG_TAG, "Hello");

		String externalStorageState = Environment.getExternalStorageState();
		if (externalStorageState.equalsIgnoreCase(Environment.MEDIA_MOUNTED))
		{
			File csvExportDir = getCsvExportDirectory(context);

			if (csvExportDir.exists())
			{
				Map<String, CsvUploadStatus> map = new HashMap<String, CsvUploadStatus>();
				File[] csvFiles = csvExportDir.listFiles(new CsvFilenameFilter());
				for (File csvFile : csvFiles)
				{
					//Log.v(LOG_TAG, "csvFile => " + csvFile);
					CsvUploadStatus csvUploadStatus = new CsvUploadStatus(csvFile);
					map.put(csvFile.getPath(), csvUploadStatus);
				}

				// set uploadedModified
				CsvUploadStatusDBHelper dbHelper = new CsvUploadStatusDBHelper(context);
				SQLiteDatabase db = dbHelper.getReadableDatabase();
				if (db != null)
				{
					try
					{
						Cursor cursor = dbHelper.getAllCSVFileUploadStatusCursor(db);
						if (cursor != null)
						{
							try
							{
								if (cursor.moveToFirst())
								{
									do
									{
										int index;
										index = cursor.getColumnIndex(DBConstants.CsvUploadStatus.Columns.FILEPATH);
										String filePath = cursor.getString(index);
										CsvUploadStatus csvUploadStatus = map.get(filePath);
										if (csvUploadStatus != null)
										{
											index = cursor.getColumnIndex(DBConstants.CsvUploadStatus.Columns.MODIFIED);
											long uploadedModified = cursor.getLong(index);
											csvUploadStatus.setUploadedModified(uploadedModified);
										}
									}
									while (cursor.moveToNext());
								}
							}
							finally
							{
								cursor.close();
							}
						}
					}
					finally
					{
						db.close();
					}
				}

				csvUploadStatusList.addAll(map.values());
				Collections.sort(csvUploadStatusList);
				Collections.reverse(csvUploadStatusList);
			}
		}

		Log.v(LOG_TAG, " Bye");
		return csvUploadStatusList;
	}

	public static List<String> getNewCsvFilePathList(Context context)
	{
		List<String> csvFilePathList = new ArrayList<String>();
		Log.v(LOG_TAG, "Hello");

		List<CsvUploadStatus> statusList = getCsvUploadStatusList(context);
		for (CsvUploadStatus csvUploadStatus : statusList)
		{
			if (csvUploadStatus.checkNecessaryToUpload())
			{
				File csvFile = csvUploadStatus.getFile();
				String csvFilePath = csvFile.getAbsolutePath();
				csvFilePathList.add(csvFilePath);
			}
		}

		Log.v(LOG_TAG, "Bye");
		return csvFilePathList;
	}
}
