#include "PDrawTest003.h"
#include "SKINI.msg"

#define NUM_BUTTONS 5
#define NUM_BUTTONS_INIT_PARAM 4

// Button Note Pad
static const double PadX = 0.06;
static const double PadY = 0.06;
static const double PadWidth = 0.29;
static const double PadHeight = 0.08;

// Button X, Y, Size table
static const double ButtonOffsetX = 0.06;
static const double ButtonOffsetY = 0.2;
static const double ButtonWidth = 0.05;
static const double ButtonHeight = 0.06;

static const double mButtonParam[NUM_BUTTONS * NUM_BUTTONS_INIT_PARAM] = {
	ButtonOffsetX+0.00, ButtonOffsetY, ButtonWidth, ButtonHeight,
	ButtonOffsetX+0.06, ButtonOffsetY, ButtonWidth, ButtonHeight,
	ButtonOffsetX+0.12, ButtonOffsetY, ButtonWidth, ButtonHeight,
	ButtonOffsetX+0.18, ButtonOffsetY, ButtonWidth, ButtonHeight,
	ButtonOffsetX+0.24, ButtonOffsetY, ButtonWidth, ButtonHeight
};

static const double mButtonNamePosition[NUM_BUTTONS * 2] = {
	ButtonOffsetX+0.00, ButtonOffsetY + (ButtonHeight * 1.7),
	ButtonOffsetX+0.06, ButtonOffsetY + (ButtonHeight * 1.7),
	ButtonOffsetX+0.12, ButtonOffsetY + (ButtonHeight * 1.7),
	ButtonOffsetX+0.18, ButtonOffsetY + (ButtonHeight * 1.7),
	ButtonOffsetX+0.24, ButtonOffsetY + (ButtonHeight * 1.7)
};

// Dial X, Y, Size table
#define NUM_DIALS 10
#define NUM_DIAL_PARAM 3

static const double DialOffsetX = 0.06;
static const double DialOffsetY = 0.35;
static const double DialSize    = 0.08;

static double mDialParam[NUM_DIALS * NUM_DIAL_PARAM] = {
	DialOffsetX + 0.00, DialOffsetY + 0.00, (DialSize * 0.5),
	DialOffsetX + 0.06, DialOffsetY + 0.00, (DialSize * 0.5),
	DialOffsetX + 0.12, DialOffsetY + 0.00, (DialSize * 0.5),
	DialOffsetX + 0.18, DialOffsetY + 0.00, (DialSize * 0.5),
	DialOffsetX + 0.24, DialOffsetY + 0.00, (DialSize * 0.5),
	DialOffsetX + 0.00, DialOffsetY + 0.15, (DialSize * 0.5),
	DialOffsetX + 0.06, DialOffsetY + 0.15, (DialSize * 0.5),
	DialOffsetX + 0.12, DialOffsetY + 0.15, (DialSize * 0.5),
	DialOffsetX + 0.18, DialOffsetY + 0.15, (DialSize * 0.5),
	DialOffsetX + 0.24, DialOffsetY + 0.15, (DialSize * 0.5)
};

// Dial Param Name
static string soundParamName[NUM_DIALS * NUM_BUTTONS] = {
	// Sin
	"freq", "gain", "", "", "", "", "", "", "", "",
	// Saw
	"freq", "gain", "", "", "", "", "", "", "", "",
	// Clarinet
	"freq", "gain", "Reed", "Noise", "ModFreq", "ModWheel", "", "", "", "", 
	// Saxofony
	"freq", "gain", "Reed", "Noise", "ModWheel", "Blow", "", "", "", "",
	// Rhodey
	"freq", "gain", "", "", "", "", "", "", "", "", 
};

// Sound name
static string soundName[NUM_BUTTONS] = {
	"Sin",
	"Saw",
	"Clar",
	"Sax",
	"Rhodey"
};
	
// Signal Monitor Area
static const double OscilloX = 0.40;
static const double OscilloY = 0.06;
static const double OscilloWidth = 0.54;
static const double OscilloHeight = 0.4;

static const int FFT_SIZE = 1024;

PDrawTest003::PDrawTest003(GrMain *grmain) : parent(grmain),
	mSaxopony(20), mSoundNumber(0), mFreq(440), mGain(0.5),
	mHold(false), mNote(false), mFftPowdB(new float[FFT_SIZE])
{
	//double ratio = ofGetHeight() / (double)ofGetWidth();
	
	// setup GUI (button)

	// Note On button
	parent->add(mNoteHold = new GrButton((int)(PadX * ofGetWidth()),
											 (int)(PadY * ofGetHeight()),
											 (int)(PadWidth * ofGetWidth()),
											 (int)(PadHeight * ofGetHeight())));
	mNoteHold->setColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1]);
	mNoteHold->addButtonListener(this);

	// Sound kind Switch
	for(int i = 0; i < NUM_BUTTONS; i++) {
		GrButton *button = new GrButton((int)(mButtonParam[i * NUM_BUTTONS_INIT_PARAM    ] * ofGetWidth()),
										(int)(mButtonParam[i * NUM_BUTTONS_INIT_PARAM + 1] * ofGetHeight()),
										(int)(mButtonParam[i * NUM_BUTTONS_INIT_PARAM + 2] * ofGetWidth()),
										(int)(mButtonParam[i * NUM_BUTTONS_INIT_PARAM + 3] * ofGetHeight()));
		button->addButtonListener(this);
		button->setColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1]);
		mButtons.push_back(button);
		parent->add(button);
	}

	// Sound Parameter Dial
	for(int i = 0; i < NUM_DIALS; i++) {
		GrDial *dial = new GrDial(parent,
								  (int)(mDialParam[i * NUM_DIAL_PARAM    ] * ofGetWidth()),
								  (int)(mDialParam[i * NUM_DIAL_PARAM + 1] * ofGetHeight()),
								  (int)(mDialParam[i * NUM_DIAL_PARAM + 2] * ofGetHeight()));
		dial->addGrDialListener(this);
		mDials.push_back(dial);
		parent->add(dial);
	}

	// setup AudioSignal
	mAdsr.setAllTimes(0.1, 0.3, 0.5, 0.5);

	AuMain *au = AuMain::getInstance();
	au->add(this);

	mAudioBuffer.assign(AuMain::getBufferSize(), 0.0);

	mfft.setup(FFT_SIZE, FFT_SIZE>>1, FFT_SIZE>>2);
}

PDrawTest003::~PDrawTest003()
{
	AuMain::getInstance()->remove(this);
	delete[] mFftPowdB;
}

void PDrawTest003::draw()
{
	int drawWidth = OscilloWidth * ofGetWidth();
	int drawHeight = OscilloHeight * ofGetHeight();
	
	ofPushStyle();
	ofNoFill();

	ofPushMatrix();
	ofSetColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1+1]);
	
	// ---- draw Hold Button
	if(mHold) {
		ofDrawBitmapString("Hold On", PadX * ofGetWidth(), PadY * ofGetHeight());
	} else {
		ofDrawBitmapString("Hold Off", PadX * ofGetWidth(), PadY * ofGetHeight());
	}
	
	// ---- draw Sound Names
	for(int i = 0; i < NUM_BUTTONS; i++) {
		ofDrawBitmapString(soundName[i],
						   (int)(mButtonNamePosition[i * 2    ] * ofGetWidth()),
						   (int)(mButtonNamePosition[i * 2 + 1] * ofGetHeight()));
	}
	
	// ---- draw Dial Names
	for(int i = 0; i < NUM_DIALS; i++) {
		ofDrawBitmapString(soundParamName[i + NUM_DIALS * mSoundNumber],
						   (int)(mDialParam[i * NUM_DIAL_PARAM    ] * ofGetWidth()),
						   (int)(mDialParam[i * NUM_DIAL_PARAM + 1] * ofGetHeight() +
								 mDialParam[i * NUM_DIAL_PARAM + 2] * ofGetHeight() * 3));
	}
	
	// ---- draw Signal Area
	ofTranslate(OscilloX * ofGetWidth(), OscilloY * ofGetHeight());
	
	ofSetColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1+1]);
	ofRect(0, 0, drawWidth, drawHeight);

	ofSetColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1+2]);
	int audioBufferSize = AuMain::getBufferSize();

	ofBeginShape();
	for(int i = 0; i < audioBufferSize; i++) {
		double x = ofMap(i, 0, audioBufferSize, 0, drawWidth);
		ofVertex(x, mAudioBuffer[i] * drawHeight * 0.5 + (drawHeight >> 1));
	}
	ofEndShape();

	// ---- draw FFT maxi
	ofTranslate(0, (OscilloY + OscilloHeight) * ofGetHeight());

	ofSetColor(colorSet[COLOR_COOL_4C_QUIET_PURPLE1+1]);
	ofRect(0, 0, drawWidth, drawHeight);

	ofBeginShape();
	for(int i = 0; i < FFT_SIZE/2; i++) {
		double x = ofMap(i, 0, FFT_SIZE/2, 0, drawWidth);
		double drawPower = (mFftPowdB[i] * drawHeight * 0.02);
		if(drawPower > drawHeight) drawPower = drawHeight;
		ofVertex(x, drawHeight - drawPower);
	}
	ofEndShape();

	ofPopMatrix();
	ofPopStyle();

}

void PDrawTest003::audioOut(double *output, int size, int channels)
{
	for(int i = 0; i < size; i++) {
		double v = 0;
		switch(mSoundNumber) {
		case SoundKindSin:
			v = mSin.tick();
			break;
		case SoundKindSaw:
			v = mSaw.tick();
			break;
		case SoundKindClarinet:
			v = mClarinet.tick();
			break;
		case SoundKindSaxofony:
			v = mSaxopony.tick();
			break;
		case SoundKindRhodey:
			v = mRhodey.tick();
			break;
		}
		
		v *= mAdsr.tick() * mGain;
		for(int c = 0; c < channels; c++) {
			output[i * channels + c] = v;
		}

		mAudioBuffer[i] = v;

		// ---- maxi fft
		if(mfft.process(v)) {
			float *magdb = mfft.magsToDB();
			for(int i = 0; i < (FFT_SIZE>>1); i++) {
				mFftPowdB[i] = magdb[i];
			}
		}
	}
}

void PDrawTest003::buttonNotify(eGrButtonEventKind kind, GrButton *button)
{
	bool note = false;
	
	// Hold on/off
	if(button == mNoteHold) {
		if(kind == GR_BUTTON_PRESSED) {
			mHold = !mHold;
		}
	}

	// ADSR
	if(kind == GR_BUTTON_PRESSED) {
		if(mHold & mNote) {
			mAdsr.keyOff();
			mNote = false;
		} else {
			mAdsr.keyOn();
			mNote = true;
		}
	} else if(kind == GR_BUTTON_RELEASED) {
		if(!mHold & mNote) {
			mAdsr.keyOff();
			mNote = false;
		}
	}
	
	// Sound Button
	for(int i = 0; i < NUM_BUTTONS; i++) {
		// Change kind of sound
		if(button == mButtons[i]) {
			mSoundNumber = i;
			printf("change kind of sound %s.\n", soundName[mSoundNumber].c_str());
		}
	}

	// Note On
	switch(mSoundNumber) {
	case SoundKindSin:
		mSin.freq = mFreq / 44100.0;
		break;
	case SoundKindSaw:
		mSaw.setFrequency(mFreq);
		break;
	case SoundKindClarinet:
		// if(kind == GR_BUTTON_PRESSED) {
		// 	mClarinet.noteOn(mFreq, 1.0);
		// } else if(kind == GR_BUTTON_RELEASED) {
		// 	mClarinet.noteOff(1.0);
		// }
		if(mNote) {
			mClarinet.noteOn(mFreq, 1.0);
		} else {
			mClarinet.noteOff(1.0);
		}
		break;
	case SoundKindSaxofony:
		// if(kind == GR_BUTTON_PRESSED) {
		// 	mSaxopony.noteOn(mFreq, 1.0);
		// } else if(kind == GR_BUTTON_RELEASED) {
		// 	mSaxopony.noteOff(1.0);
		// }
		if(mNote) {
			mSaxopony.noteOn(mFreq, 1.0);
		} else {
			mSaxopony.noteOff(1.0);
		}
		break;
	case SoundKindRhodey:
		// if(kind == GR_BUTTON_PRESSED) {
		// 	mRhodey.noteOn(mFreq, 1.0);
		// } else if(kind == GR_BUTTON_RELEASED) {
		// 	mRhodey.noteOff(1.0);
		// }
		if(mNote) {
			mRhodey.noteOn(mFreq, 1.0);
		} else {
			mRhodey.noteOff(1.0);
		}
		break;
	}
}

void PDrawTest003::dialValueChanged(GrDial *grDial, double value)
{
	for(int i = 0; i < NUM_DIALS; i++) {
		if(grDial == mDials[i]) {
			printf("dial value changed sound=%s, param=%s, value=%f\n",
				   soundName[mSoundNumber].c_str(),
				   soundParamName[i].c_str(),
				   value);
		}
	}

	double valueForMidi = ofMap(value, 0, 1, 0, 128);
	
	// common parameter
	if(grDial == mDials[0]) { // freq
		mFreq = ofMap(value, 0, 1, 20, 2000); // Hz
	} else if(grDial == mDials[1]) { // gain
		mGain = value;
	} else {
		// sound specific parameter
		switch(mSoundNumber) {
		case SoundKindSin:
			break;
		case SoundKindSaw:
			break;
		case SoundKindClarinet:
			//printf("%f\n", valueForMidi);
			if(grDial == mDials[2]) { // Reed
				mClarinet.controlChange(__SK_ReedStiffness_, valueForMidi);
			} else if(grDial == mDials[3]) { // Noise
				mClarinet.controlChange(__SK_NoiseLevel_, valueForMidi);
			} else if(grDial == mDials[4]) { // ModFreq
				mClarinet.controlChange(__SK_ModFrequency_, valueForMidi);
			} else if(grDial == mDials[5]) { // ModWheel
				mClarinet.controlChange(__SK_ModWheel_, valueForMidi);
			}
			break;
		case SoundKindSaxofony:
			if(grDial == mDials[2]) { // Reed
				mSaxopony.controlChange(__SK_ReedStiffness_, valueForMidi);
			} else if(grDial == mDials[3]) { // Noise
				mSaxopony.controlChange(__SK_NoiseLevel_, valueForMidi);
			} else if(grDial == mDials[4]) { // ModFreq
				mSaxopony.controlChange(__SK_ModWheel_, valueForMidi);
			} else if(grDial == mDials[5]) { // Blow
				mSaxopony.controlChange(11, valueForMidi);
			}
			break;
		case SoundKindRhodey:
			break;
		}
	}
}

