#ifndef __P_SIG__
#define __P_SIG__

#include <math.h>
#include "ofMain.h"
#include "debug.h"
#include "PSigMacro.h"

namespace psig {
	
void init();

extern int mouseMoveX;
extern int mouseMoveY;
extern int mouseDragX;
extern int mouseDragY;
extern int mousePressX;
extern int mousePressY;
extern int mouseReleaseX;
extern int mouseReleaseY;
	
/**
 * @brief PGen (basic Signal interface)
 * @note super class of all signal class
 */
class PGen {
public:
	PGen() : ref_cnt(0) {
		static int id_cnt = 0;
		this->id = id_cnt++;
		debugf("id=%d, ref_cnt=%d\n", id, ref_cnt);
	}
	
	virtual ~PGen() {
		debugf("id=%d, ref_cnt=%d\n", id, ref_cnt);
	}
	
    virtual double tick() = 0;
	
	void addRef() {
		ref_cnt++;
	}
	
	void release() {
		ref_cnt--;
		debugf("id=%d, ref_cnt=%d\n", id, ref_cnt);
		if(ref_cnt <= 0) {
			delete this;
		}
	}
	
private:
	int ref_cnt;
	int id;
};

/**
 * @brief PValue (Port static value signal class)
 * @note static value signal class. this is default implements of PPort.
 */
class PGenValue : public PGen {
public:
    PGenValue(double value) : mValue(value) {}
	/* virtual ~PGenValue() { */
	/* 	debugl(); */
	/* } */
	
    PGenValue &operator =(double value) {
        mValue = value;
        return *this;
    }
    double tick() {
        return mValue;
    }
private:
    double mValue;
};

/**
 * @brief PPort (basic Port class)
 * @note Port class
 */
class PPort {
public:
    PPort() : mValue(0), mGen( &mValue ) {
		setIdCnt();

	}
	
	explicit PPort(PGen *gen) : mValue(0), mGen( gen ) {
		setIdCnt();
		gen->addRef();
	}

    explicit PPort(double value) : mValue(value), mGen( &mValue ) {
		static int id_cnt = 0;
		this->id = id_cnt++;
		debugf("id=%d\n", id);
	}
	
	~PPort() {
		debugf("id=%d\n", id);
		if(mGen != &mValue) {
			mGen->release();
		}
	}
    
	//! set Generator to this port 
	PPort& operator=(PGen *gen) {
		mGen = gen;
		gen->addRef();
		return *this;
	}
	
	//! set Static value to this port 
	PPort& operator=(double value) {
		//printf("%d, %d\n", mGen, &mValue);
		if(mGen != &mValue) {
			mGen->release();
		}
        mGen = &mValue;
        mValue = value;
		return *this;
	}

	//! get value form gen (generator or static gen)
    operator double() {
        return get();
    }
	
	//! get value form gen (generator or static gen)
	operator float() {
		return (float)get();
	}

    double get() {
        return mGen->tick();
    }
	
private:
	void setIdCnt() {
		static int id_cnt = 0;
		this->id = id_cnt++;
		debugf("id=%d\n", id);
	}
	
    PGenValue mValue;
    PGen *mGen;
	int id;
};

/**
 * @brief PAdd
 * @note sub class of PGen. Add Unit.
 */
class PAdd : public PGen {
public:
	PGenConstructor2Arg(PAdd, a, b);
	
    PPort a, b;
    double tick() {
        return ( a.get() + b.get() );
    }
};

/**
 * @brief PMlt
 * @note sub class of PGen. Multiply Unit.
 */
class PMlt : public PGen {
public:
	PGenConstructor2Arg(PMlt, a, b);

    PPort a, b;
    double tick() {
        return ( a.get() * b.get() );
    }
};

/**
 * @brief PCnt
 * @note sub class of PGen. Counter Unit.
 */
class PCnt : public PGen {
public:
	PCnt(int cnt, int min = 0, int max = 255, int step = 1) :
		cnt(cnt), min(min), max(max), step(step)
	{
		this->cnt = (cnt < min) ? min : cnt;
		this->cnt = (cnt > max) ? max : cnt;
	}

	virtual ~PCnt() {
		debugl();
	}

	double tick() {
		double _c = cnt++;
		if(cnt == max) {
			cnt = min;
		}
		return _c;
	}

private:
	int cnt;
	int min;
	int max;
	int step;
};

/**
 * @brief PCntd
 * @note sub class of PGen. Counter Unit.
 */
class PCntd : public PGen {
public:
	PCntd(double start = 0, double end = 1.0, int numStep = 10)
	{
		if(start < end) {
			min = start;
			max = end;
		} else {
			min = end;
			max = start;
		}
		step = (end - start) / numStep;
		cnt = start;
	}

	PCntd(double start, double end, int numStep, double init)
	{
		PCntd(start, end, numStep);
		cnt = init;
	}


	virtual ~PCntd() {
		debugl();
	}

	double tick() {
		double _c = cnt;
		cnt += step;
		cnt = (cnt < min) ? max : cnt;
		cnt = (cnt > max) ? min : cnt;
		return _c;
	}

private:
	double cnt;
	double min;
	double max;
	double step;
};

#define _PI2 6.28318530717958

/**
 * @brief PSin
 * @note sub class of PGen. Sin Unit.
 * @param freq : frequency(Hz) / FS(Hz)
 * @param phase : first phase value. input value = 0 ~ 1.0 -> set value = 0 ~ PI2
 */
class PSin : public PGen {
public:
	PSin(double freq = (440.0/44100.0), double phase = 0) : freq(freq),	phase(phase * _PI2) {}
	PSin(PGen *freq, double phase = 0) : freq(freq),	phase(phase * _PI2) {}
	
	virtual ~PSin() {
		debugl();
	}
	
    double tick() {
		phase += _PI2 * freq.get();
        return sin(phase);
    }
	
    PPort freq;
	
private:
	double phase;
};

/**
 * @brief PInput
 * @note sub class of PGen. Add Unit.
 * @param input : input pointer of data
 */
class PInput : public PGen {
public:
    PInput(int *input) : in(input) {}
    double tick() {
        return *in;
    }
private:
	int *in;
};

/**
 * @brief PGate
 * @note sub class of PGen. Add Unit.
 * @param input : input pointer of data
 * @param enable : if enable is not 0, tick value is updated.
 */
class PGate : public PGen {
public:
	//PGate(PGen *input, PGen *enable) : input(input), enable(enable) {}
	PGenConstructor2Arg(PGate, input, enable);
	PPort input, enable;

	double tick() {
		double en = enable;
		if(en) {
			last = input.get();
		}
		return last;
	}
private:
	double last;
};


} // namespace
#endif // __P_SIG__


