#include "ofxSimpleDial.h"

void ofxSimpleDial::setup(int x, int y, int size, bool fill)
{
	this->mX = x;
	this->mY = y;
	this->mSize = size; // radius of dial
	this->mFill = fill;
	this->mValue = 0.5; // 0.0 ~ 1.0
	this->mDrawPrecision = 100; // num draw points
	this->mRingOpenRatio = 0.04;
	this->mPressX = 0;
	this->mPressY = 0;
	this->mDialSpeed = 1.5;
	
	this->mOnColor = ofColor(120, 120, 100);
	this->mOffColor = ofColor(70, 70, 70);
	this->mCurrentColor = mOffColor;
 
	ofAddListener(ofEvents().draw, this, &ofxSimpleDial::draw);
	ofAddListener(ofEvents().mousePressed, this, &ofxSimpleDial::mousePressed);	
	ofAddListener(ofEvents().mouseReleased, this, &ofxSimpleDial::mouseReleased);	
	ofAddListener(ofEvents().mouseDragged, this, &ofxSimpleDial::mouseDragged);	
}

void ofxSimpleDial::draw(ofEventArgs& event)
{
	ofPushStyle();
	ofPushMatrix();

	(mFill) ? ofFill() : ofNoFill();

	ofSetColor(mCurrentColor);

	double value = mValue;

	double startRadian = ofMap(      mRingOpenRatio, 0, 1, 0, PI * 2);
	double endRadian   = ofMap(1.0 - mRingOpenRatio, 0, 1, 0, PI * 2);

	//double precisionAdjustmentInv = 100.0 / ((w + h) >> 2);
	double stepRadian  = (endRadian - startRadian) / mDrawPrecision;
	int numStep = (int)(mDrawPrecision);

	// // stepRadian /= precisionAdjustmentInv;
	// // numStep *= (int)(numStep * precisionAdjustmentInv);
	
	ofTranslate(mX + mSize, mY + mSize);
	ofRotate(mRotateDegree, 0, 0, 1);
	
	ofBeginShape();

	double rad = startRadian;
	for(int i = 0; i < numStep+1; i++) {
		double px = -sin(rad);
		double py = -cos(rad);
		ofVertex(px * mSize, py * mSize);
		rad += stepRadian;
	}

	rad = endRadian;
	for(int i = 0; i < numStep+1; i++) {
		double px = -sin(rad);
		double py = -cos(rad);
		ofVertex((px * mSize) * 0.7, (py * mSize) * 0.7);
		rad -= stepRadian;
	}
	
	ofEndShape();

	ofPopMatrix();
	ofPopStyle();
}
	
void ofxSimpleDial::mousePressed(ofMouseEventArgs& event)
{
	ofVec2f center(mX + mSize, mY + mSize);
	ofVec2f mouse(event.x, event.y);
	
	float dist = center.distance(mouse);
	if(dist < mSize) {
		mState = DIAL_PRESSED;
		mPressX = event.x;
		mPressY = event.y;

		sendNotify(DIAL_PRESSED);
		
		// update color
		mCurrentColor = mOnColor;
	}
}

void ofxSimpleDial::mouseReleased(ofMouseEventArgs& event)
{
	mState = DIAL_RELEASED;
	sendNotify(DIAL_RELEASED);
	// update color
	mCurrentColor = mOffColor;
}

void ofxSimpleDial::mouseDragged(ofMouseEventArgs& event)
{
	if(mState == DIAL_PRESSED) {
		int distY = mPressY - event.y;
		mPressY = event.y;
		mValue += distY * mDialSpeed / (double)ofGetHeight();
		
		if(mValue > 1) {
			mValue = 1;
		} else if(mValue < 0) {
			mValue = 0;
		}
		mRotateDegree = (double)(-150 + (300 * mValue));

		sendNotify(DIAL_CHANGED);
	}
}

void ofxSimpleDial::addListener(ofxSimpleDialListener *listener)
{
	mDialListeners.push_back(listener);
}

void ofxSimpleDial::removeListner(ofxSimpleDialListener *listener)
{
	vector<ofxSimpleDialListener*>::iterator end = std::remove(mDialListeners.begin(),
															   mDialListeners.end(), listener);
	mDialListeners.erase(end, mDialListeners.end());
}

void ofxSimpleDial::sendNotify(ofxSimpleDialEventType type)
{
	ofxSimpleDialEvent event;
	event.dial = this;
	event.type = type;
	event.value = mValue;
	for(vector<ofxSimpleDialListener*>::iterator itr = mDialListeners.begin();
		itr != mDialListeners.end(); itr++) {
		(*itr)->dialNotify(event);
	}
}
