/*
 * locale-info.c
 *
 * Utility program to enumerate, and assist in identification of, the entity
 * references within a named Windows locale; (requires Vista, or later).
 *
 * $Id: locale-info.c,v ee7de7b264dd 2019/03/04 21:59:17 keith $
 *
 * Written by Keith Marshall <keith@users.osdn.me>
 * Copyright (C) 2019, MinGW.org Project.
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice, this permission notice, and the following
 * disclaimer shall be included in all copies or substantial portions of
 * the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OF OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *
 * Usage:
 *   locale-info [-c] [-l <IETF-locale>] [-k <key-value>]
 *   locale-info [-c] [-l <IETF-locale>] [-r [<first-key>,]<last-key>]
 *
 * Options:
 *   -c				   Query the "calendar" category,
 *   				   rather than the "general" category
 *   				   of the specified locale.
 *
 *   -k <key-value>		   Query only one <key-value>.
 *
 *   -l <IETF-locale>		   Set IETF name of locale to query;
 *   				   if unspecified, the "en-GB" locale
 *   				   is queried.
 *
 *   -r [<first-key>,]<last-key>   Set the query range to key values
 *   				   from <first-key> to <last-key>; if
 *   				   omitted, <first-key> defaults to
 *   				   zero.  If neither "-k" or "-r" is
 *   				   specified, the default key range
 *   				   is from zero to UINT_MAX>>4.
 *
 * Compile time options:
 *   In addition to the usual GCC options, users may choose to enable:
 *
 *   -D BUILD_DLL -shared	   Build the DLL component of the
 *				   application.  The application must
 *				   be compiled and linked twice; once
 *				   with BOTH of these options, and
 *				   again with NEITHER, to build the
 *				   DLL and main EXE components of
 *				   the application respectively.
 *
 *   -D SETLOCALE		   Enables synchronization of the CRT
 *   				   locale with the specified Windows
 *   				   locale.
 *
 *   -D ALT_BUFSIZ=<value>	   Adjust the size of the buffer, in
 *   				   which query results are captured;
 *   				   default is BUFSIZ (nominally 512);
 *   				   increase if "insufficient buffer"
 *   				   errors are observed.
 *
 */
#define _GNU_SOURCE
#define _WIN32_WINNT _WIN32_WINNT_VISTA

static const char *error = "Error ***";
#define errprintf(fmt, ...)  fprintf(stderr, "%s " fmt, error, ##__VA_ARGS__)

#include <stdio.h>
#include <winbase.h>
#include <stdlib.h>

/* The primary duty of the application is served by a DLL component...
 */
#if BUILD_DLL
/* ...which is compiled in this case; (it is assumed that the -shared
 * option is also specified, for the compile and link).
 */
#include <limits.h>
#include <unistd.h>
#include <winnls.h>
#include <locale.h>

/* Missing from <winnls.h> */
WINAPI int GetCalendarInfoEx (LPCWSTR, CALID, LPCWSTR, CALTYPE, LPWSTR, int, LPDWORD);
WINAPI int GetLocaleInfoEx (LPCWSTR, int, LPWSTR, int);

#ifndef ALT_BUFSIZ
#define ALT_BUFSIZ  BUFSIZ
#endif

static wchar_t *locale = L"en-GB";
static union { unsigned int i; wchar_t s[ALT_BUFSIZ]; } buf;
static CALID calendar;

static int insufficient_buffer( unsigned int id, int needs )
{
  /* Helper function to print an error message, if the alloted buffer is
   * too small to capture the description of the specified locale entity.
   */
  return printf( "0x%08X: insufficient buffer; need %d words\n", id, needs ); }

static int report( unsigned int id, int len )
{ /* Helper function to display the identifier, and description, for any
   * one Windows locale entity; display is suppressed, for any entity code
   * which does not yield a description of non-zero length.
   */
  if( len > 0 )
    return printf( "0x%1$08X (%1$04d): %2$04d words: %3$S\n", id, len, buf.s );
  return 0;
}

static void get_locale_info( unsigned int lookup )
{
  /* Helper function to look up, retrieve, and display the descriptive text
   * associated with any one numeric key, within the "general" category of a
   * Windows locale; becomes a no-op, for any specified key value with which
   * no description is associated.
   */
  int len = GetLocaleInfoEx( locale, lookup, NULL, 0 );
  if( len > ALT_BUFSIZ ) insufficient_buffer( lookup, len );
  else report( lookup, GetLocaleInfoEx( locale, lookup, buf.s, ALT_BUFSIZ ) );
}

static void get_calendar_info( unsigned int lookup )
{
  /* Helper function to look up, retrieve, and display the descriptive text
   * associated with any one numeric key, within the "calendar" category of
   * a Windows locale; becomes a no-op, for any key value with which there
   * is no associated description.
   */
  int len = GetCalendarInfoEx( locale, calendar, NULL, lookup, NULL, 0, NULL );
  if( len > ALT_BUFSIZ ) insufficient_buffer( lookup, len );
  else report( lookup, GetCalendarInfoEx(
       	 locale, calendar, NULL, lookup, buf.s, ALT_BUFSIZ, NULL )
      );
}

#if SETLOCALE
static int build_locale_string( wchar_t delim, int offset, int category )
{
  /* Helper function to construct a locale identification string, which
   * is suitable for passing to MSVCRT.DLL's _wsetlocale() function, by
   * piecing together the appropriate elements from the Windows locale.
   */
  int len = GetLocaleInfoEx( locale, category, buf.s + offset, ALT_BUFSIZ - offset );
  if( len > 0 )
  { /* When this is not the first element of the constructed string,
     * insert the specified delimiter character, in place of the NUL
     * terminator on the preceding element.
     */
    if( offset > 0 ) buf.s[ offset - 1 ] = delim;
    return offset + len;
  }
  return offset;
}
#endif

__declspec(dllexport)
int query_locale_info( int argc, char **argv )
{
  unsigned int base = 0, span = UINT_MAX >> 4;
  int opt; void (*action)(unsigned int) = get_locale_info;

  /* Interpret any command line options, which may have been specified.
   */
  while( (opt = getopt( argc, argv, "ck:l:r:" )) != -1 )
  { switch( opt )
    {
      case '?':
	/* An unrecognized option ws specified; bail out.
	 */
	return EXIT_FAILURE;

      case 'c':
	/* The "-c" option was specified; select the inspection
	 * action which is specific to the "calendar" category of
	 * locale information.
	 */
	action = get_calendar_info;
	break;

      case 'k':
	/* The "-k" option restricts the look-up to just one key
	 * value; set both "base" and "span" for the look-up cycle
	 * to this single value.
	 */
	base = span = strtoul( optarg, NULL, 0 );
	break;

      case 'l':
	/* A locale specification has been given, with the "-l" option;
	 * nominally, this should be an IETF format specification, given
	 * as an ASCII string, but we need is as UTF-16LE.
	 */
        if( ALT_BUFSIZ >= MultiByteToWideChar( CP_OEMCP, 0, optarg, -1, NULL, 0 ) )
	{ MultiByteToWideChar( CP_OEMCP, 0, optarg, -1, buf.s, ALT_BUFSIZ );
	  locale = wcsdup( buf.s );
	}
	break;

      case 'r':
	/* The "-r" option specifies a key value range for the look-up
	 * cycle; this may be in the form "-r <first>,<last>", (in which
	 * case both the "base" and "span" key values are assigned), or
	 * in the simplified "-r <last>" form, (in which case, only the
	 * "span" setting is assigned).
	 */
	{ char *brk;
	  span = strtoul( optarg, &brk, 0 );
	  if( *brk == ',' )
	  { /* This is the "-r <first>,<last>" form; capture the <last>
	     * specification, then ensure that <first> and <last> values
	     * are assigned to "base" and "span", in ascending order.
	     */
	    unsigned int tmp = strtoul( ++brk, NULL, 0 );
	    if( tmp > span )
	    { base = span; span = tmp; }
	    else base = tmp;
	  }
	  else if( base > span )
	  { /* This is the "-r <last>" form, but a prior "base" setting
	     * has made it greater than "span", (assigned from <last>);
	     * swap them, to maintain ascending order.
	     */
	    unsigned int tmp = base; base = span; span = tmp;
	  }
	}
    }
  }

# if SETLOCALE
  /* Compiled with the "-D SETLOCALE" option; synchronize the CRT
   * notion of locale with the specified Windows locale.
   *
   * Note that, contrary to Microsoft documentation, on my Win-7 box,
   * _wsetlocale() does NOT seem to comprehend IETF format locale specs;
   * neither does it seem to accept ISO-639/ISO-3166 abbreviated forms,
   * nor even Microsoft's own long form names, in any language other
   * than English ... yuck!
   */
  opt = build_locale_string( 0, 0, LOCALE_SENGLANGUAGE );
  opt = build_locale_string( L'_', opt, LOCALE_SENGCOUNTRY );
  build_locale_string( L'.', opt, LOCALE_IDEFAULTANSICODEPAGE );
  if( _wsetlocale( LC_ALL, buf.s ) == NULL )
  { errprintf( "failed to set CRT locale to '%S'\n", buf.s );
    return EXIT_FAILURE;
  }
# endif

  if( action == get_calendar_info )
  { /* The "-c" option was specifed; thus, we will be inspecting the
     * "calendar" category of the local, and we need to establish the
     * appropriate calendar type, to pass to GetCalendarInfo().
     */
    calendar = GetLocaleInfoEx( locale,
	LOCALE_RETURN_NUMBER | LOCALE_ICALENDARTYPE, buf.s, 2
      );
    if( calendar == 2 ) calendar = buf.i;
    else
    { errprintf( "cannot establish calendar for locale '%S'\n", locale );
      return EXIT_FAILURE;
    }
  }
  /* Finally, cycle over the specified key value range, performing
   * the specified look-up action, ensuring that we do not wrap at
   * UINT_MAX, into an interminable loop.
   */
  for( unsigned int lookup = base; span >= lookup; lookup++ )
  { action( lookup ); if( lookup == UINT_MAX ) break; }

  return EXIT_SUCCESS;
}

#else	/* ! BUILD_DLL */
/* When not building the DLL component, we simply furnish the main()
 * function; this checks that the necessary GetLocaleInfoEx() function
 * is actually present within kernel32.dll, (it isn't, on any Windows
 * version pre-dating Vista), before delegating the actual query to
 * the handler in the supporting DLL component.
 */
int main( int argc, char **argv )
{
  typedef int (*query)( int, char ** );

  HMODULE ref; const char *api = "kernel32.dll";
  if( (ref = GetModuleHandleA( api )) != NULL )
  {
    if( GetProcAddress( ref, api = "GetLocaleInfoEx" ) == NULL )
    {
      errprintf( "function '%s' is unsupported on this host\n", api );
      errprintf( "Windows-Vista (or later) is required\n" );
    }
    else if( (ref = LoadLibraryA( api = "locale-info.dll" )) == NULL )
      errprintf( "cannot load '%s'\n", api );

    else
    { api = "query_locale_info";
      query locale_info = (query)(GetProcAddress( ref, api ));
      if( locale_info == NULL )
	errprintf( "cannot access '%s()' API\n", api );

      else return locale_info( argc, argv );
    }
  }
  else errprintf( "'%s' was not mapped within this process\n", api );
  return EXIT_FAILURE;
}

#endif	/* ! BUILD_DLL */

/* $RCSfile: locale-info.c,v $: end of file */
