package org.dyndns.nuda.tools.util.binary;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * バイナリアレイを表すオブジェクトです
 * 
 * @author nkoseki
 * 
 */
public class Binary {
	/*
	 * 生成元
	 */
	private byte[]					binaryArray	= null;
	
	/*
	 * リソースバンドル
	 */
	private static ResourceBundle	rb			= null;
	
	/*
	 * 固定値:'0'
	 */
	private static final String		ZERO		= "0";
	
	/*
	 * リソースバンドル読み込み
	 */
	static {
		rb = ResourceBundle.getBundle(Binary.class.getCanonicalName());
	}
	
	/**
	 * 引数にバイト配列を指定してインスタンスを生成します.<br />
	 * もし、引数に指定されたバイト配列が初期化されていない場合は<br />
	 * すべての２進値が0で初期化されたものと認識されます.
	 * 
	 * @param binaryArray
	 *            インスタンスの生成元となるバイト配列
	 * 
	 * @throws IllegalArgumentException
	 *             引数に指定されたバイト配列がnullまたは配列長がゼロの場合にスローされます
	 */
	public Binary(final byte[] binaryArray) {
		
		if (binaryArray == null) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_01));
		}
		if (binaryArray.length == 0) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_02));
		}
		this.binaryArray = Arrays.copyOf(binaryArray, binaryArray.length);
	}
	
	/**
	 * このクラスの生成元となったバイト配列を返します
	 * 
	 * @return このクラスの生成元バイトアレイ
	 */
	public byte[] getSource() {
		byte[] result =
			Arrays.copyOf(this.binaryArray, this.binaryArray.length);
		return result;
	}
	
	/**
	 * バイナリアレイ同士の論理積を算出します
	 * 
	 * @param binary
	 * @return
	 */
	public Binary and(final Binary binary) {
		if (binary == null) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_05));
		}
		return null;
	}
	
	/**
	 * バイナリアレイ同士の論理和を算出します
	 * 
	 * @param binary
	 * @return
	 */
	public Binary or(final Binary binary) {
		if (binary == null) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_05));
		}
		return null;
	}
	
	/**
	 * バイナリアレイの否定を算出します
	 * 
	 * @return
	 */
	public Binary not() {
		
		return null;
	}
	
	/**
	 * バイナリアレイ同士の排他的論理和を算出します
	 * 
	 * @param binary
	 * @return
	 */
	public Binary xor(final Binary binary) {
		if (binary == null) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_05));
		}
		return null;
	}
	
	/**
	 * このクラスのバイナリ表現の2進表記を返します
	 * 
	 * @return 2進表記された文字列
	 */
	public String getBinaryForm() {
		
		StringBuilder builder = new StringBuilder();
		
		for (byte b : this.getSource()) {
			// 上位4bit
			String upper = Integer.toBinaryString((b & 0xF0) >> 4);
			// 下位4bit
			String under = Integer.toBinaryString(b & 0xF);
			
			StringBuilder upperBuilder = new StringBuilder();
			
			/*
			 * 以下、上位/下位ビットともに0フィルされる可能性があるため
			 * 足りない桁数を埋める
			 */

			/*
			 * 上位4bit調整
			 */
			if (upper.length() != 0) {
				int fill = this.getFillLength(upper, 4);
				for (int i = 0; i < fill; i++) {
					upperBuilder.append(ZERO);
				}
				
				upperBuilder.append(upper);
				
				builder.append(upperBuilder.toString());
			}
			
			/*
			 * 下位4ビット調整
			 */
			StringBuilder underBuilder = new StringBuilder();
			if (under.length() != 0) {
				int fill = this.getFillLength(under, 4);
				for (int i = 0; i < fill; i++) {
					underBuilder.append(ZERO);
				}
				
				underBuilder.append(under);
				builder.append(underBuilder.toString());
			}
			
		}
		return builder.toString();
	}
	
	/**
	 * このバイナリオブジェクトの16進表現を返します(1-9, a-f)
	 * 
	 * @return 16進表現文字列
	 */
	public String getHexForm() {
		StringBuilder b = new StringBuilder();
		String binForm = this.getBinaryForm();
		Pattern p = Pattern.compile("([01]{4})");
		Matcher m = p.matcher(binForm);
		while (m.find()) {
			String digit = m.group(1);
			
			if ("0000".equals(digit)) {
				b.append("0");
			} else if ("0001".equals(digit)) {
				b.append("1");
			} else if ("0010".equals(digit)) {
				b.append("2");
			} else if ("0011".equals(digit)) {
				b.append("3");
			} else if ("0100".equals(digit)) {
				b.append("4");
			} else if ("0101".equals(digit)) {
				b.append("5");
			} else if ("0110".equals(digit)) {
				b.append("6");
			} else if ("0111".equals(digit)) {
				b.append("7");
			} else if ("1000".equals(digit)) {
				b.append("8");
			} else if ("1001".equals(digit)) {
				b.append("9");
			} else if ("1010".equals(digit)) {
				b.append("a");
			} else if ("1011".equals(digit)) {
				b.append("b");
			} else if ("1100".equals(digit)) {
				b.append("c");
			} else if ("1101".equals(digit)) {
				b.append("d");
			} else if ("1110".equals(digit)) {
				b.append("e");
			} else if ("1111".equals(digit)) {
				b.append("f");
			}
		}
		
		return b.toString();
	}
	
	/**
	 * このバイナリオブジェクトの32進表現(0-9, a-v)を返します
	 * 
	 * @return 32進表現文字列
	 */
	public String get32dForm() {
		StringBuilder b = new StringBuilder();
		String binForm = this.getBinaryForm();
		
		int fill = this.getFillLength(binForm, 5);
		
		StringBuilder tmp = new StringBuilder();
		for (int i = 0; i < fill; i++) {
			tmp.append("0");
		}
		
		tmp.append(binForm);
		binForm = tmp.toString();
		
		Pattern p = Pattern.compile("([01]{5})");
		Matcher m = p.matcher(binForm);
		while (m.find()) {
			String digit = m.group(1);
			
			if ("00000".equals(digit)) {
				b.append("0");
			} else if ("00001".equals(digit)) {
				b.append("1");
			} else if ("00010".equals(digit)) {
				b.append("2");
			} else if ("00011".equals(digit)) {
				b.append("3");
			} else if ("00100".equals(digit)) {
				b.append("4");
			} else if ("00101".equals(digit)) {
				b.append("5");
			} else if ("00110".equals(digit)) {
				b.append("6");
			} else if ("00111".equals(digit)) {
				b.append("7");
			} else if ("01000".equals(digit)) {
				b.append("8");
			} else if ("01001".equals(digit)) {
				b.append("9");
			} else if ("01010".equals(digit)) {
				b.append("a");
			} else if ("01011".equals(digit)) {
				b.append("b");
			} else if ("01100".equals(digit)) {
				b.append("c");
			} else if ("01101".equals(digit)) {
				b.append("d");
			} else if ("01110".equals(digit)) {
				b.append("e");
			} else if ("01111".equals(digit)) {
				b.append("f");
			} else if ("10000".equals(digit)) {
				b.append("g");
			} else if ("10001".equals(digit)) {
				b.append("h");
			} else if ("10010".equals(digit)) {
				b.append("i");
			} else if ("10011".equals(digit)) {
				b.append("j");
			} else if ("10100".equals(digit)) {
				b.append("k");
			} else if ("10101".equals(digit)) {
				b.append("l");
			} else if ("10110".equals(digit)) {
				b.append("m");
			} else if ("10111".equals(digit)) {
				b.append("n");
			} else if ("11000".equals(digit)) {
				b.append("o");
			} else if ("11001".equals(digit)) {
				b.append("p");
			} else if ("11010".equals(digit)) {
				b.append("q");
			} else if ("11011".equals(digit)) {
				b.append("r");
			} else if ("11100".equals(digit)) {
				b.append("s");
			} else if ("11101".equals(digit)) {
				b.append("t");
			} else if ("11110".equals(digit)) {
				b.append("u");
			} else if ("11111".equals(digit)) {
				b.append("v");
			}
		}
		
		return b.toString();
	}
	
	/**
	 * このバイナリオブジェクトの64進表現(0-9, a-z, A-Z, +-)を返します
	 * 
	 * @return 64進表現文字列
	 */
	public String get64dForm() {
		StringBuilder b = new StringBuilder();
		String binForm = this.getBinaryForm();
		
		int fill = this.getFillLength(binForm, 6);
		
		StringBuilder tmp = new StringBuilder();
		for (int i = 0; i < fill; i++) {
			tmp.append("0");
		}
		
		tmp.append(binForm);
		binForm = tmp.toString();
		
		Pattern p = Pattern.compile("([01]{6})");
		Matcher m = p.matcher(binForm);
		while (m.find()) {
			String digit = m.group(1);
			
			if ("000000".equals(digit)) {
				b.append("0");
			} else if ("000001".equals(digit)) {
				b.append("1");
			} else if ("000010".equals(digit)) {
				b.append("2");
			} else if ("000011".equals(digit)) {
				b.append("3");
			} else if ("000100".equals(digit)) {
				b.append("4");
			} else if ("000101".equals(digit)) {
				b.append("5");
			} else if ("000110".equals(digit)) {
				b.append("6");
			} else if ("000111".equals(digit)) {
				b.append("7");
			} else if ("001000".equals(digit)) {
				b.append("8");
			} else if ("001001".equals(digit)) {
				b.append("9");
			} else if ("001010".equals(digit)) {
				b.append("a");
			} else if ("001011".equals(digit)) {
				b.append("b");
			} else if ("001100".equals(digit)) {
				b.append("c");
			} else if ("001101".equals(digit)) {
				b.append("d");
			} else if ("001110".equals(digit)) {
				b.append("e");
			} else if ("001111".equals(digit)) {
				b.append("f");
			} else if ("010000".equals(digit)) {
				b.append("g");
			} else if ("010001".equals(digit)) {
				b.append("h");
			} else if ("010010".equals(digit)) {
				b.append("i");
			} else if ("010011".equals(digit)) {
				b.append("j");
			} else if ("010100".equals(digit)) {
				b.append("k");
			} else if ("010101".equals(digit)) {
				b.append("l");
			} else if ("010110".equals(digit)) {
				b.append("m");
			} else if ("010111".equals(digit)) {
				b.append("n");
			} else if ("011000".equals(digit)) {
				b.append("o");
			} else if ("011001".equals(digit)) {
				b.append("p");
			} else if ("011010".equals(digit)) {
				b.append("q");
			} else if ("011011".equals(digit)) {
				b.append("r");
			} else if ("011100".equals(digit)) {
				b.append("s");
			} else if ("011101".equals(digit)) {
				b.append("t");
			} else if ("011110".equals(digit)) {
				b.append("u");
			} else if ("011111".equals(digit)) {
				b.append("v");
			}
			// ここから17進
			// wxyz A-Z, +-
			else if ("100000".equals(digit)) {
				b.append("w");
			} else if ("100001".equals(digit)) {
				b.append("x");
			} else if ("100010".equals(digit)) {
				b.append("y");
			} else if ("100011".equals(digit)) {
				b.append("z");
			} else if ("100100".equals(digit)) {
				b.append("A");
			} else if ("100101".equals(digit)) {
				b.append("B");
			} else if ("100110".equals(digit)) {
				b.append("C");
			} else if ("100111".equals(digit)) {
				b.append("D");
			} else if ("101000".equals(digit)) {
				b.append("E");
			} else if ("101001".equals(digit)) {
				b.append("F");
			} else if ("101010".equals(digit)) {
				b.append("G");
			} else if ("101011".equals(digit)) {
				b.append("H");
			} else if ("101100".equals(digit)) {
				b.append("I");
			} else if ("101101".equals(digit)) {
				b.append("J");
			} else if ("101110".equals(digit)) {
				b.append("K");
			} else if ("101111".equals(digit)) {
				b.append("L");
			} else if ("110000".equals(digit)) {
				b.append("M");
			} else if ("110001".equals(digit)) {
				b.append("N");
			} else if ("110010".equals(digit)) {
				b.append("O");
			} else if ("110011".equals(digit)) {
				b.append("P");
			} else if ("110100".equals(digit)) {
				b.append("Q");
			} else if ("110101".equals(digit)) {
				b.append("R");
			} else if ("110110".equals(digit)) {
				b.append("S");
			} else if ("110111".equals(digit)) {
				b.append("T");
			} else if ("111000".equals(digit)) {
				b.append("U");
			} else if ("111001".equals(digit)) {
				b.append("V");
			} else if ("111010".equals(digit)) {
				b.append("W");
			} else if ("111011".equals(digit)) {
				b.append("X");
			} else if ("111100".equals(digit)) {
				b.append("Y");
			} else if ("111101".equals(digit)) {
				b.append("Z");
			} else if ("111110".equals(digit)) {
				b.append("+");
			} else if ("111111".equals(digit)) {
				b.append("-");
			}
		}
		
		return b.toString();
	}
	
	/**
	 * 64進文字列表現からバイナリオブジェクトを生成します
	 * 
	 * @param source
	 *            64進文字列表現
	 * @return バイナリオブジェクト
	 */
	public static Binary decode64DFrom(final String source) {
		Pattern p = Pattern.compile("([0-9a-zA-Z\\+\\-])+");
		Pattern p2 = Pattern.compile("([0-9a-zA-Z\\+\\-]{1})");
		Matcher m = p.matcher(source);
		StringBuilder b = new StringBuilder();
		if (!m.matches()) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_03));
		} else {
			Matcher m2 = p2.matcher(source);
			
			while (m2.find()) {
				String digit = m2.group(1);
				
				if ("0".equals(digit)) {
					b.append("000000");
				} else if ("1".equals(digit)) {
					b.append("000001");
				} else if ("2".equals(digit)) {
					b.append("000010");
				} else if ("3".equals(digit)) {
					b.append("000011");
				} else if ("4".equals(digit)) {
					b.append("000100");
				} else if ("5".equals(digit)) {
					b.append("000101");
				} else if ("6".equals(digit)) {
					b.append("000110");
				} else if ("7".equals(digit)) {
					b.append("000111");
				} else if ("8".equals(digit)) {
					b.append("001000");
				} else if ("9".equals(digit)) {
					b.append("001001");
				} else if ("a".equals(digit)) {
					b.append("001010");
				} else if ("b".equals(digit)) {
					b.append("001011");
				} else if ("c".equals(digit)) {
					b.append("001100");
				} else if ("d".equals(digit)) {
					b.append("001101");
				} else if ("e".equals(digit)) {
					b.append("001110");
				} else if ("f".equals(digit)) {
					b.append("001111");
				} else if ("g".equals(digit)) {
					b.append("010000");
				} else if ("h".equals(digit)) {
					b.append("010001");
				} else if ("i".equals(digit)) {
					b.append("010010");
				} else if ("j".equals(digit)) {
					b.append("010011");
				} else if ("k".equals(digit)) {
					b.append("010100");
				} else if ("l".equals(digit)) {
					b.append("010101");
				} else if ("m".equals(digit)) {
					b.append("010110");
				} else if ("n".equals(digit)) {
					b.append("010111");
				} else if ("o".equals(digit)) {
					b.append("011000");
				} else if ("p".equals(digit)) {
					b.append("011001");
				} else if ("q".equals(digit)) {
					b.append("011010");
				} else if ("r".equals(digit)) {
					b.append("011011");
				} else if ("s".equals(digit)) {
					b.append("011100");
				} else if ("t".equals(digit)) {
					b.append("011101");
				} else if ("u".equals(digit)) {
					b.append("011110");
				} else if ("v".equals(digit)) {
					b.append("011111");
				}
				// ここから17進
				// wxyz A-Z, +-
				else if ("w".equals(digit)) {
					b.append("100000");
				} else if ("x".equals(digit)) {
					b.append("100001");
				} else if ("y".equals(digit)) {
					b.append("100010");
				} else if ("z".equals(digit)) {
					b.append("100011");
				} else if ("A".equals(digit)) {
					b.append("100100");
				} else if ("B".equals(digit)) {
					b.append("100101");
				} else if ("C".equals(digit)) {
					b.append("100110");
				} else if ("D".equals(digit)) {
					b.append("100111");
				} else if ("E".equals(digit)) {
					b.append("101000");
				} else if ("F".equals(digit)) {
					b.append("101001");
				} else if ("G".equals(digit)) {
					b.append("101010");
				} else if ("H".equals(digit)) {
					b.append("101011");
				} else if ("I".equals(digit)) {
					b.append("101100");
				} else if ("J".equals(digit)) {
					b.append("101101");
				} else if ("K".equals(digit)) {
					b.append("101110");
				} else if ("L".equals(digit)) {
					b.append("101111");
				} else if ("M".equals(digit)) {
					b.append("110000");
				} else if ("N".equals(digit)) {
					b.append("110001");
				} else if ("O".equals(digit)) {
					b.append("110010");
				} else if ("P".equals(digit)) {
					b.append("110011");
				} else if ("Q".equals(digit)) {
					b.append("110100");
				} else if ("R".equals(digit)) {
					b.append("110101");
				} else if ("S".equals(digit)) {
					b.append("110110");
				} else if ("T".equals(digit)) {
					b.append("110111");
				} else if ("U".equals(digit)) {
					b.append("111000");
				} else if ("V".equals(digit)) {
					b.append("111001");
				} else if ("W".equals(digit)) {
					b.append("111010");
				} else if ("X".equals(digit)) {
					b.append("111011");
				} else if ("Y".equals(digit)) {
					b.append("111100");
				} else if ("Z".equals(digit)) {
					b.append("111101");
				} else if ("+".equals(digit)) {
					b.append("111110");
				} else if ("-".equals(digit)) {
					b.append("111111");
				}
			}
			
		}
		
		String binForm = b.toString();
		
		int sup = binForm.length() % 8;
		
		binForm = binForm.substring(sup, binForm.length());
		
		Binary result = decode2DForm(binForm);
		
		return result;
	}
	
	/**
	 * 32進文字列表現からバイナリオブジェクトを生成します
	 * 
	 * @param source
	 *            32進文字列表現
	 * @return バイナリオブジェクト
	 */
	public static Binary decode32DFrom(final String source) {
		Pattern p = Pattern.compile("([0-9a-v])+");
		Pattern p2 = Pattern.compile("([0-9a-v]{1})");
		Matcher m = p.matcher(source);
		StringBuilder b = new StringBuilder();
		if (!m.matches()) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_04));
		} else {
			Matcher m2 = p2.matcher(source);
			
			while (m2.find()) {
				String digit = m2.group(1);
				
				if ("0".equals(digit)) {
					b.append("00000");
				} else if ("1".equals(digit)) {
					b.append("00001");
				} else if ("2".equals(digit)) {
					b.append("00010");
				} else if ("3".equals(digit)) {
					b.append("00011");
				} else if ("4".equals(digit)) {
					b.append("00100");
				} else if ("5".equals(digit)) {
					b.append("00101");
				} else if ("6".equals(digit)) {
					b.append("00110");
				} else if ("7".equals(digit)) {
					b.append("00111");
				} else if ("8".equals(digit)) {
					b.append("01000");
				} else if ("9".equals(digit)) {
					b.append("01001");
				} else if ("a".equals(digit)) {
					b.append("01010");
				} else if ("b".equals(digit)) {
					b.append("01011");
				} else if ("c".equals(digit)) {
					b.append("01100");
				} else if ("d".equals(digit)) {
					b.append("01101");
				} else if ("e".equals(digit)) {
					b.append("01110");
				} else if ("f".equals(digit)) {
					b.append("01111");
				} else if ("g".equals(digit)) {
					b.append("10000");
				} else if ("h".equals(digit)) {
					b.append("10001");
				} else if ("i".equals(digit)) {
					b.append("10010");
				} else if ("j".equals(digit)) {
					b.append("10011");
				} else if ("k".equals(digit)) {
					b.append("10100");
				} else if ("l".equals(digit)) {
					b.append("10101");
				} else if ("m".equals(digit)) {
					b.append("10110");
				} else if ("n".equals(digit)) {
					b.append("10111");
				} else if ("o".equals(digit)) {
					b.append("11000");
				} else if ("p".equals(digit)) {
					b.append("11001");
				} else if ("q".equals(digit)) {
					b.append("11010");
				} else if ("r".equals(digit)) {
					b.append("11011");
				} else if ("s".equals(digit)) {
					b.append("11100");
				} else if ("t".equals(digit)) {
					b.append("11101");
				} else if ("u".equals(digit)) {
					b.append("11110");
				} else if ("v".equals(digit)) {
					b.append("11111");
				}
				
			}
			
		}
		
		String binForm = b.toString();
		
		int sup = binForm.length() % 8;
		
		binForm = binForm.substring(sup, binForm.length());
		
		Binary result = decode2DForm(binForm);
		
		return result;
	}
	
	/**
	 * 16進文字列表現からバイナリオブジェクトを生成します
	 * 
	 * @param source
	 *            16進文字列表現
	 * @return バイナリオブジェクト
	 */
	public static Binary decode16DFrom(final String source) {
		Pattern p = Pattern.compile("([0-9a-f])+");
		Pattern p2 = Pattern.compile("([0-9a-f]{1})");
		Matcher m = p.matcher(source);
		StringBuilder b = new StringBuilder();
		if (!m.matches()) {
			throw new IllegalArgumentException(
				rb.getString(BinaryPropertiesKey.ERR_04));
		} else {
			Matcher m2 = p2.matcher(source);
			
			while (m2.find()) {
				String digit = m2.group(1);
				
				if ("0".equals(digit)) {
					b.append("0000");
				} else if ("1".equals(digit)) {
					b.append("0001");
				} else if ("2".equals(digit)) {
					b.append("0010");
				} else if ("3".equals(digit)) {
					b.append("0011");
				} else if ("4".equals(digit)) {
					b.append("0100");
				} else if ("5".equals(digit)) {
					b.append("0101");
				} else if ("6".equals(digit)) {
					b.append("0110");
				} else if ("7".equals(digit)) {
					b.append("0111");
				} else if ("8".equals(digit)) {
					b.append("1000");
				} else if ("9".equals(digit)) {
					b.append("1001");
				} else if ("a".equals(digit)) {
					b.append("1010");
				} else if ("b".equals(digit)) {
					b.append("1011");
				} else if ("c".equals(digit)) {
					b.append("1100");
				} else if ("d".equals(digit)) {
					b.append("1101");
				} else if ("e".equals(digit)) {
					b.append("1110");
				} else if ("f".equals(digit)) {
					b.append("1111");
				}
				
			}
		}
		
		String binForm = b.toString();
		
		int sup = binForm.length() % 8;
		
		binForm = binForm.substring(sup, binForm.length());
		
		Binary result = decode2DForm(binForm);
		
		return result;
	}
	
	/**
	 * 2進文字列表現からバイナリオブジェクトを生成します
	 * 
	 * @param source
	 *            2進文字列表現
	 * @return バイナリオブジェクト
	 */
	public static Binary decode2DForm(final String source) {
		Pattern p = Pattern.compile("([01]{8})");
		Matcher m = p.matcher(source);
		List<String> digitList = new LinkedList<String>();
		while (m.find()) {
			String digit = m.group(1);
			digitList.add(digit);
			
		}
		
		byte[] result = new byte[digitList.size()];
		
		int count = 0;
		for (String digit : digitList) {
			String upper = digit.substring(0, 4);
			String under = digit.substring(4, 8);
			
			int upp = getIntFromBinForm(upper);
			int und = getIntFromBinForm(under);
			
			// 上位桁と下位桁をまとめてbyte値を作成
			byte out = (byte) ((upp << 4) | und);
			result[count] = out;
			
			count++;
		}
		
		Binary resultObj = new Binary(result);
		
		return resultObj;
	}
	
	private static int getIntFromBinForm(final String source) {
		int result = 0;
		
		if ("0000".equals(source)) {
			result = 0;
		} else if ("0001".equals(source)) {
			result = 1;
		} else if ("0010".equals(source)) {
			result = 2;
		} else if ("0011".equals(source)) {
			result = 3;
		} else if ("0100".equals(source)) {
			result = 4;
		} else if ("0101".equals(source)) {
			result = 5;
		} else if ("0110".equals(source)) {
			result = 6;
		} else if ("0111".equals(source)) {
			result = 7;
		} else if ("1000".equals(source)) {
			result = 8;
		} else if ("1001".equals(source)) {
			result = 9;
		} else if ("1010".equals(source)) {
			result = 10;
		} else if ("1011".equals(source)) {
			result = 11;
		} else if ("1100".equals(source)) {
			result = 12;
		} else if ("1101".equals(source)) {
			result = 13;
		} else if ("1110".equals(source)) {
			result = 14;
		} else if ("1111".equals(source)) {
			result = 15;
		}
		
		return result;
	}
	
	private int getFillLength(final String source, final int baseLength) {
		if (source == null || source.isEmpty()) {
			return 0;
		}
		int fill = baseLength - (source.length() % baseLength);
		if (fill == baseLength) {
			fill = 0;
		}
		return fill;
	}
	
	public int length() {
		int len = this.getSource().length;
		
		return len * 8;
	}
	
}
