<?php

require_once('simg-functions.php');
require_once('SimgLog.php');

define('_ERROR_INDEX', '_e');
define('_INFO_INDEX', '_i');

/**
 * 出力クラス
 * @author System Concept
 *
 */
abstract class SimgOutput {
	/**
	 * Smartyインスタンス
	 */
	protected $smarty;
	/**
	 * ページ情報
	 * key: テンプレートファイル名
	 * value: パーシャル配列
	 */
	protected $page;
	/**
	 * テンプレートファイル名
	 */
	private $template;
	/**
	 * エラー情報
	 */
	private $error;

	/**
	 * 構築
	 */
	public function __construct() {
		$this->initialize();
	}

	/**
	 * ホームテンプレート取得
	 * @return テンプレート
	 */
	abstract public function getHomeTemplate();

	/**
	 * システムエラーテンプレート取得
	 * @return テンプレート
	 */
	abstract public function getSystemErrorTemplate();

	/**
	 * 初期化
	 * @param $parameter パラメータ
	 */
	private function initialize() {
		$this->page = $this->createPage();
		$this->smarty = $this->createSmarty();
		$this->tempalte = null;
		$this->error = array();
	}
	/**
	 * ページ情報作成
	 * @return ページ情報
	 */
	protected function createPage() {
		$maps = simg_make_definemap($this->getPageConfigPath(), array('partial', 'smarty', 'template:partial'), array());
		return simg_get_array_value($maps, 'template', null);
	}
	/**
	 * ページ情報configファイルパス取得
	 * @return ファイルパス
	 */
	protected function getPageConfigPath() {
		return simg_combine_path(SIMG_PATH_CONFIG, SIMG_FN_CONFIG_PAGE);
	}

	/**
	 * Smarty生成
	 */
	protected function createSmarty() {
		$smarty = new Smarty();
		$smarty->template_dir = SIMG_PATH_TEMPLATE . '/';
		$smarty->compile_dir = SIMG_PATH_TEMPLATE_COMPILE . '/';
		$smarty->cache_dir    = SIMG_PATH_TEMPLATE_CACHE . '/';
		$this->loadResourceOnDirectory($smarty, SIMG_PATH_RESOURCE);
		simg_smarty_initialize($smarty);
		return $smarty;
	}
	/**
	 * ディレクトリ配下のリソースファイルをSmarty定数として読み込む
	 * @param $smarty ビュー
	 * @param $dir ディレクトリ
	 */
	protected function loadResourceOnDirectory($smarty, $dir) {
		$dir_handle = opendir($dir);
		while($file_name = readdir($dir_handle)) {
			$ext = pathinfo($file_name, PATHINFO_EXTENSION);
			//if($ext == 'ini') {
			if($ext == SIMG_RESOURCE_EXT) {
				$w = simg_combine_path($dir, pathinfo($file_name, PATHINFO_BASENAME));
				$smarty->configLoad($w);
				simg_config_load($w);
			}
		}
	}
	/**
	 * テンプレートファイル名設定
	 * @param $v テンプレートファイル名
	 */
	public function setTemplate($v) {
		$this->template = $v;
	}
	/**
	 * テンプレートファイル名取得
	 * @return テンプレートファイル名
	 */
	public function getTemplate() {
		return $this->template;
	}
	/**
	 * パーシャルを保持しているかを調べる
	 * @param $partial パーシャル
	 * @param $template テンプレート名
	 * @return true/false=保持している/していない
	 */
	public function hasPartial($partial, $template = null) {
		$template = (is_null($template))? $this->template: $template;
		//$ar = simg_get_value($this->page[$template], array());
		$ar = simg_get_array_value($this->page, $template, array());
		return in_array($partial, $ar);
	}
	/**
	 * テンプレートに値を割り当てる
	 * @param $name 名前
	 * @param $val 値
	 */
	public function assign($name, $val) {
		$this->smarty->assign($name, $val);
	}
	/**
	 * テンプレートを表示する
	 * @param $template テンプレート名
	 * @return true/false=表示した/テンプレートが無い
	 */
	public function display($template = null) {
		$ret = false;
		$template = (is_null($template))? $this->template: $template;
		if(!is_null($template)) {
			$this->smarty->display($template);
			$ret = true;
		}
		return $ret;
	}
	/**
	 * ホーム遷移
	 */
	public function toHome() {
		$this->setTemplate($this->getHomeTemplate());
	}
	/**
	 * システムエラー遷移
	 */
	public function toSystemError() {
		$this->setTemplate($this->getSystemErrorTemplate());
	}
	/**
	 * エラーコード設定
	 * @param $err エラーコード
	 */
	public function setError($err) {
		$this->error[_ERROR_INDEX] = $err;
	}
	/**
	 * エラーコード取得
	 * @return エラーコード
	 */
	public function getError() {
		return simg_get_array_value($this->error, _ERROR_INDEX, null);
	}
	/**
	 * エラーコード判定
	 * @return true/false=エラーあり/なし
	 */
	public function isExistError() {
		return !is_null($this->getError());
	}
	/**
	 * エラー情報設定
	 * @param $info エラーコード
	 */
	public function setErrorInfo($info) {
		$this->error[_INFO_INDEX] = $err;
	}
	/**
	 * エラー情報取得
	 * @return エラー情報
	 */
	public function getErrorInfo() {
		return simg_get_array_value($this->error, _INFO_INDEX, null);
	}
	/**
	 * エラー情報有無判定
	 * @return true/false=エラー情報あり/なし
	 */
	public function isExistErrorInfo() {
		return !is_null($this->getErrorInfo());
	}
	/**
	 * エラー処理
	 * @param $e 例外
	 */
	public function error($e) {
		$this->toSystemError();
		$this->assign(SIMG_ASSIGN_ERROR, $e->getMessage());
	}
}

?>

