/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.task.reduce;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.mapred.Counters;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.TaskStatus;
import org.apache.hadoop.mapreduce.TaskAttemptID;
import org.apache.hadoop.mapreduce.TaskID;
import org.apache.hadoop.mapreduce.task.reduce.ExceptionReporter;
import org.apache.hadoop.mapreduce.task.reduce.MapHost;
import org.apache.hadoop.mapreduce.task.reduce.MapOutput;
import org.apache.hadoop.util.Progress;

class ShuffleScheduler<K, V> {
    static ThreadLocal<Long> shuffleStart = new ThreadLocal<Long>(){

        @Override
        protected Long initialValue() {
            return 0L;
        }
    };
    private static final Log LOG = LogFactory.getLog(ShuffleScheduler.class);
    private static final int MAX_MAPS_AT_ONCE = 20;
    private static final long INITIAL_PENALTY = 10000L;
    private static final float PENALTY_GROWTH_RATE = 1.3f;
    private static final int REPORT_FAILURE_LIMIT = 10;
    private final boolean[] finishedMaps;
    private final int totalMaps;
    private int remainingMaps;
    private Map<String, MapHost> mapLocations = new HashMap<String, MapHost>();
    private Set<MapHost> pendingHosts = new HashSet<MapHost>();
    private Set<TaskAttemptID> obsoleteMaps = new HashSet<TaskAttemptID>();
    private final Random random = new Random(System.currentTimeMillis());
    private final DelayQueue<Penalty> penalties = new DelayQueue();
    private final Referee referee = new Referee();
    private final Map<TaskAttemptID, IntWritable> failureCounts = new HashMap<TaskAttemptID, IntWritable>();
    private final Map<String, IntWritable> hostFailures = new HashMap<String, IntWritable>();
    private final TaskStatus status;
    private final ExceptionReporter reporter;
    private final int abortFailureLimit;
    private final Progress progress;
    private final Counters.Counter shuffledMapsCounter;
    private final Counters.Counter reduceShuffleBytes;
    private final Counters.Counter failedShuffleCounter;
    private final long startTime;
    private long lastProgressTime;
    private int maxMapRuntime = 0;
    private int maxFailedUniqueFetches = 5;
    private int maxFetchFailuresBeforeReporting;
    private long totalBytesShuffledTillNow = 0L;
    private DecimalFormat mbpsFormat = new DecimalFormat("0.00");
    private boolean reportReadErrorImmediately = true;
    private long maxDelay = 60000L;

    public ShuffleScheduler(JobConf job, TaskStatus status, ExceptionReporter reporter, Progress progress, Counters.Counter shuffledMapsCounter, Counters.Counter reduceShuffleBytes, Counters.Counter failedShuffleCounter) {
        this.totalMaps = job.getNumMapTasks();
        this.abortFailureLimit = Math.max(30, this.totalMaps / 10);
        this.remainingMaps = this.totalMaps;
        this.finishedMaps = new boolean[this.remainingMaps];
        this.reporter = reporter;
        this.status = status;
        this.progress = progress;
        this.shuffledMapsCounter = shuffledMapsCounter;
        this.reduceShuffleBytes = reduceShuffleBytes;
        this.failedShuffleCounter = failedShuffleCounter;
        this.lastProgressTime = this.startTime = System.currentTimeMillis();
        this.referee.start();
        this.maxFailedUniqueFetches = Math.min(this.totalMaps, this.maxFailedUniqueFetches);
        this.maxFetchFailuresBeforeReporting = job.getInt("mapreduce.reduce.shuffle.maxfetchfailures", 10);
        this.reportReadErrorImmediately = job.getBoolean("mapreduce.reduce.shuffle.notify.readerror", true);
        this.maxDelay = job.getLong("mapreduce.reduce.shuffle.retry-delay.max.ms", 60000L);
    }

    public synchronized void copySucceeded(TaskAttemptID mapId, MapHost host, long bytes, long millis, MapOutput<K, V> output) throws IOException {
        this.failureCounts.remove(mapId);
        this.hostFailures.remove(host.getHostName());
        int mapIndex = mapId.getTaskID().getId();
        if (!this.finishedMaps[mapIndex]) {
            output.commit();
            this.finishedMaps[mapIndex] = true;
            this.shuffledMapsCounter.increment(1L);
            if (--this.remainingMaps == 0) {
                this.notifyAll();
            }
            this.totalBytesShuffledTillNow += bytes;
            this.updateStatus();
            this.reduceShuffleBytes.increment(bytes);
            this.lastProgressTime = System.currentTimeMillis();
            LOG.debug((Object)("map " + mapId + " done " + this.status.getStateString()));
        }
    }

    private void updateStatus() {
        float mbs = (float)this.totalBytesShuffledTillNow / 1048576.0f;
        int mapsDone = this.totalMaps - this.remainingMaps;
        long secsSinceStart = (System.currentTimeMillis() - this.startTime) / 1000L + 1L;
        float transferRate = mbs / (float)secsSinceStart;
        this.progress.set((float)mapsDone / (float)this.totalMaps);
        String statusString = mapsDone + " / " + this.totalMaps + " copied.";
        this.status.setStateString(statusString);
        this.progress.setStatus("copy(" + mapsDone + " of " + this.totalMaps + " at " + this.mbpsFormat.format(transferRate) + " MB/s)");
    }

    public synchronized void copyFailed(TaskAttemptID mapId, MapHost host, boolean readError, boolean connectExcpt) {
        host.penalize();
        int failures = 1;
        if (this.failureCounts.containsKey(mapId)) {
            IntWritable x = this.failureCounts.get(mapId);
            x.set(x.get() + 1);
            failures = x.get();
        } else {
            this.failureCounts.put(mapId, new IntWritable(1));
        }
        String hostname = host.getHostName();
        if (this.hostFailures.containsKey(hostname)) {
            IntWritable x = this.hostFailures.get(hostname);
            x.set(x.get() + 1);
        } else {
            this.hostFailures.put(hostname, new IntWritable(1));
        }
        if (failures >= this.abortFailureLimit) {
            try {
                throw new IOException(failures + " failures downloading " + mapId);
            }
            catch (IOException ie) {
                this.reporter.reportException(ie);
            }
        }
        this.checkAndInformJobTracker(failures, mapId, readError, connectExcpt);
        this.checkReducerHealth();
        long delay = (long)(10000.0 * Math.pow(1.3f, failures));
        if (delay > this.maxDelay) {
            delay = this.maxDelay;
        }
        this.penalties.add(new Penalty(host, delay));
        this.failedShuffleCounter.increment(1L);
    }

    private void checkAndInformJobTracker(int failures, TaskAttemptID mapId, boolean readError, boolean connectExcpt) {
        if (connectExcpt || this.reportReadErrorImmediately && readError || failures % this.maxFetchFailuresBeforeReporting == 0) {
            LOG.info((Object)("Reporting fetch failure for " + mapId + " to jobtracker."));
            this.status.addFetchFailedMap((org.apache.hadoop.mapred.TaskAttemptID)mapId);
        }
    }

    private void checkReducerHealth() {
        boolean reducerStalled;
        int doneMaps;
        float MAX_ALLOWED_FAILED_FETCH_ATTEMPT_PERCENT = 0.5f;
        float MIN_REQUIRED_PROGRESS_PERCENT = 0.5f;
        float MAX_ALLOWED_STALL_TIME_PERCENT = 0.5f;
        long totalFailures = this.failedShuffleCounter.getValue();
        boolean reducerHealthy = (float)totalFailures / (float)(totalFailures + (long)(doneMaps = this.totalMaps - this.remainingMaps)) < 0.5f;
        boolean reducerProgressedEnough = (float)doneMaps / (float)this.totalMaps >= 0.5f;
        int stallDuration = (int)(System.currentTimeMillis() - this.lastProgressTime);
        int shuffleProgressDuration = (int)(this.lastProgressTime - this.startTime);
        int minShuffleRunDuration = shuffleProgressDuration > this.maxMapRuntime ? shuffleProgressDuration : this.maxMapRuntime;
        boolean bl = reducerStalled = (float)stallDuration / (float)minShuffleRunDuration >= 0.5f;
        if (!(this.failureCounts.size() < this.maxFailedUniqueFetches && this.failureCounts.size() != this.totalMaps - doneMaps || reducerHealthy || reducerProgressedEnough && !reducerStalled)) {
            LOG.fatal((Object)"Shuffle failed with too many fetch failures and insufficient progress!");
            String errorMsg = "Exceeded MAX_FAILED_UNIQUE_FETCHES; bailing-out.";
            this.reporter.reportException(new IOException(errorMsg));
        }
    }

    public synchronized void tipFailed(TaskID taskId) {
        if (!this.finishedMaps[taskId.getId()]) {
            this.finishedMaps[taskId.getId()] = true;
            if (--this.remainingMaps == 0) {
                this.notifyAll();
            }
            this.updateStatus();
        }
    }

    public synchronized void addKnownMapOutput(String hostName, String hostUrl, TaskAttemptID mapId) {
        MapHost host = this.mapLocations.get(hostName);
        if (host == null) {
            host = new MapHost(hostName, hostUrl);
            this.mapLocations.put(hostName, host);
        }
        host.addKnownMap(mapId);
        if (host.getState() == MapHost.State.PENDING) {
            this.pendingHosts.add(host);
            this.notifyAll();
        }
    }

    public synchronized void obsoleteMapOutput(TaskAttemptID mapId) {
        this.obsoleteMaps.add(mapId);
    }

    public synchronized void putBackKnownMapOutput(MapHost host, TaskAttemptID mapId) {
        host.addKnownMap(mapId);
    }

    public void reportLocalError(IOException ioe) {
        try {
            LOG.error((Object)("Shuffle failed : local error on this node: " + InetAddress.getLocalHost()));
        }
        catch (UnknownHostException e) {
            LOG.error((Object)"Shuffle failed : local error on this node");
        }
        this.reporter.reportException(ioe);
    }

    public synchronized MapHost getHost() throws InterruptedException {
        while (this.pendingHosts.isEmpty()) {
            this.wait();
        }
        MapHost host = null;
        Iterator<MapHost> iter = this.pendingHosts.iterator();
        int numToPick = this.random.nextInt(this.pendingHosts.size());
        for (int i = 0; i <= numToPick; ++i) {
            host = iter.next();
        }
        this.pendingHosts.remove(host);
        host.markBusy();
        LOG.info((Object)("Assiging " + host + " with " + host.getNumKnownMapOutputs() + " to " + Thread.currentThread().getName()));
        shuffleStart.set(System.currentTimeMillis());
        return host;
    }

    public synchronized List<TaskAttemptID> getMapsForHost(MapHost host) {
        TaskAttemptID id;
        List<TaskAttemptID> list = host.getAndClearKnownMaps();
        Iterator<TaskAttemptID> itr = list.iterator();
        ArrayList<TaskAttemptID> result = new ArrayList<TaskAttemptID>();
        int includedMaps = 0;
        int totalSize = list.size();
        while (itr.hasNext()) {
            id = itr.next();
            if (this.obsoleteMaps.contains(id) || this.finishedMaps[id.getTaskID().getId()]) continue;
            result.add(id);
            if (++includedMaps < 20) continue;
            break;
        }
        while (itr.hasNext()) {
            id = itr.next();
            if (this.obsoleteMaps.contains(id) || this.finishedMaps[id.getTaskID().getId()]) continue;
            host.addKnownMap(id);
        }
        LOG.info((Object)("assigned " + includedMaps + " of " + totalSize + " to " + host + " to " + Thread.currentThread().getName()));
        return result;
    }

    public synchronized void freeHost(MapHost host) {
        if (host.getState() != MapHost.State.PENALIZED && host.markAvailable() == MapHost.State.PENDING) {
            this.pendingHosts.add(host);
            this.notifyAll();
        }
        LOG.info((Object)(host + " freed by " + Thread.currentThread().getName() + " in " + (System.currentTimeMillis() - shuffleStart.get()) + "s"));
    }

    public synchronized void resetKnownMaps() {
        this.mapLocations.clear();
        this.obsoleteMaps.clear();
        this.pendingHosts.clear();
    }

    public synchronized boolean waitUntilDone(int millis) throws InterruptedException {
        if (this.remainingMaps > 0) {
            this.wait(millis);
            return this.remainingMaps == 0;
        }
        return true;
    }

    public void close() throws InterruptedException {
        this.referee.interrupt();
        this.referee.join();
    }

    public synchronized void informMaxMapRunTime(int duration) {
        if (duration > this.maxMapRuntime) {
            this.maxMapRuntime = duration;
        }
    }

    private class Referee
    extends Thread {
        public Referee() {
            this.setName("ShufflePenaltyReferee");
            this.setDaemon(true);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void run() {
            try {
                while (true) {
                    MapHost host = ((Penalty)((ShuffleScheduler)ShuffleScheduler.this).penalties.take()).host;
                    ShuffleScheduler shuffleScheduler = ShuffleScheduler.this;
                    synchronized (shuffleScheduler) {
                        if (host.markAvailable() == MapHost.State.PENDING) {
                            ShuffleScheduler.this.pendingHosts.add(host);
                            ShuffleScheduler.this.notifyAll();
                        }
                    }
                }
            }
            catch (InterruptedException ie) {
                return;
            }
            catch (Throwable t) {
                ShuffleScheduler.this.reporter.reportException(t);
                return;
            }
        }
    }

    private static class Penalty
    implements Delayed {
        MapHost host;
        private long endTime;

        Penalty(MapHost host, long delay) {
            this.host = host;
            this.endTime = System.currentTimeMillis() + delay;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long remainingTime = this.endTime - System.currentTimeMillis();
            return unit.convert(remainingTime, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            long other = ((Penalty)o).endTime;
            return this.endTime == other ? 0 : (this.endTime < other ? -1 : 1);
        }
    }
}

