/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.fs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import org.apache.flink.core.fs.BlockLocation;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.HadoopFileSystemWrapper;
import org.apache.flink.core.fs.Path;
import org.apache.flink.util.ClassUtils;
import org.apache.flink.util.OperatingSystem;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystem.class);
    private static final String LOCAL_FILESYSTEM_CLASS = "org.apache.flink.core.fs.local.LocalFileSystem";
    private static final String HADOOP_WRAPPER_FILESYSTEM_CLASS = "org.apache.flink.runtime.fs.hdfs.HadoopFileSystem";
    private static final String MAPR_FILESYSTEM_CLASS = "org.apache.flink.runtime.fs.maprfs.MapRFileSystem";
    private static final String S3_FILESYSTEM_CLASS = "org.apache.flink.runtime.fs.s3.S3FileSystem";
    private static final String HADOOP_WRAPPER_SCHEME = "hdwrapper";
    private static final Object SYNCHRONIZATION_OBJECT = new Object();
    private static final Map<FSKey, FileSystem> CACHE = new HashMap<FSKey, FileSystem>();
    private static final Map<String, String> FSDIRECTORY = new HashMap<String, String>();
    private static HadoopFileSystemWrapper hadoopWrapper;

    public static FileSystem getLocalFileSystem() throws IOException {
        URI localUri;
        try {
            localUri = OperatingSystem.isWindows() ? new URI("file:/") : new URI("file:///");
        }
        catch (URISyntaxException e) {
            throw new IOException("Cannot create URI for local file system");
        }
        return FileSystem.get(localUri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static FileSystem get(URI uri) throws IOException {
        FileSystem fs = null;
        Object object = SYNCHRONIZATION_OBJECT;
        synchronized (object) {
            if (uri.getScheme() == null) {
                try {
                    uri = new URI("file", null, uri.getPath(), null);
                }
                catch (URISyntaxException e) {
                    try {
                        uri = new URI("file", null, new Path(new File(uri.getPath()).getAbsolutePath()).toUri().getPath(), null);
                    }
                    catch (URISyntaxException ex) {
                        throw new IOException("The file URI '" + uri.toString() + "' is not valid.");
                    }
                }
            }
            if (uri.getScheme().equals("file") && uri.getAuthority() != null && !uri.getAuthority().isEmpty()) {
                String supposedUri = "file:///" + uri.getAuthority() + uri.getPath();
                throw new IOException("Found local file path with authority '" + uri.getAuthority() + "' in path '" + uri.toString() + "'. Hint: Did you forget a slash? (correct path would be '" + supposedUri + "')");
            }
            FSKey key = new FSKey(uri.getScheme(), uri.getAuthority());
            if (CACHE.containsKey(key)) {
                return CACHE.get(key);
            }
            if (!FSDIRECTORY.containsKey(uri.getScheme())) {
                Class<?> wrapperClass = FileSystem.getHadoopWrapperClassNameForFileSystem(uri.getScheme());
                if (wrapperClass == null) throw new IOException("No file system found with scheme " + uri.getScheme() + ", referenced in file URI '" + uri.toString() + "'.");
                FSKey wrappedKey = new FSKey("hdwrapper+" + uri.getScheme(), uri.getAuthority());
                if (CACHE.containsKey(wrappedKey)) {
                    return CACHE.get(wrappedKey);
                }
                fs = FileSystem.instantiateHadoopFileSystemWrapper(wrapperClass);
                fs.initialize(uri);
                CACHE.put(wrappedKey, fs);
            } else {
                String fsClass = FSDIRECTORY.get(uri.getScheme());
                fs = fsClass.equals(HADOOP_WRAPPER_FILESYSTEM_CLASS) ? FileSystem.instantiateHadoopFileSystemWrapper(null) : FileSystem.instantiateFileSystem(fsClass);
                fs.initialize(uri);
                CACHE.put(key, fs);
            }
            return fs;
        }
    }

    private static FileSystem instantiateHadoopFileSystemWrapper(Class<?> wrappedFileSystem) throws IOException {
        FileSystem fs = null;
        try {
            Class<? extends FileSystem> fsClass = ClassUtils.getFileSystemByName(HADOOP_WRAPPER_FILESYSTEM_CLASS);
            Constructor<? extends FileSystem> fsClassCtor = fsClass.getConstructor(Class.class);
            fs = fsClassCtor.newInstance(wrappedFileSystem);
        }
        catch (Throwable e) {
            throw new IOException("Error loading Hadoop FS wrapper", e);
        }
        return fs;
    }

    private static FileSystem instantiateFileSystem(String className) throws IOException {
        Class<? extends FileSystem> fsClass;
        FileSystem fs = null;
        try {
            fsClass = ClassUtils.getFileSystemByName(className);
        }
        catch (ClassNotFoundException e1) {
            throw new IOException(StringUtils.stringifyException(e1));
        }
        try {
            fs = fsClass.newInstance();
        }
        catch (InstantiationException e) {
            throw new IOException("Could not instantiate file system class: " + e.getMessage(), e);
        }
        catch (IllegalAccessException e) {
            throw new IOException("Could not instantiate file system class: " + e.getMessage(), e);
        }
        return fs;
    }

    private static Class<?> getHadoopWrapperClassNameForFileSystem(String scheme) {
        if (hadoopWrapper == null) {
            try {
                hadoopWrapper = (HadoopFileSystemWrapper)((Object)FileSystem.instantiateHadoopFileSystemWrapper(null));
            }
            catch (IOException e) {
                throw new RuntimeException("Error creating new Hadoop wrapper", e);
            }
        }
        return hadoopWrapper.getHadoopWrapperClassNameForFileSystem(scheme);
    }

    public abstract Path getWorkingDirectory();

    public abstract URI getUri();

    public abstract void initialize(URI var1) throws IOException;

    public abstract FileStatus getFileStatus(Path var1) throws IOException;

    public abstract BlockLocation[] getFileBlockLocations(FileStatus var1, long var2, long var4) throws IOException;

    public abstract FSDataInputStream open(Path var1, int var2) throws IOException;

    public abstract FSDataInputStream open(Path var1) throws IOException;

    public long getDefaultBlockSize() {
        return 0x2000000L;
    }

    public abstract FileStatus[] listStatus(Path var1) throws IOException;

    public boolean exists(Path f) throws IOException {
        try {
            return this.getFileStatus(f) != null;
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    public abstract boolean delete(Path var1, boolean var2) throws IOException;

    public abstract boolean mkdirs(Path var1) throws IOException;

    public abstract FSDataOutputStream create(Path var1, boolean var2, int var3, short var4, long var5) throws IOException;

    public abstract FSDataOutputStream create(Path var1, boolean var2) throws IOException;

    public abstract boolean rename(Path var1, Path var2) throws IOException;

    public boolean initOutPathLocalFS(Path outPath, WriteMode writeMode, boolean createDirectory) throws IOException {
        if (this.isDistributedFS()) {
            return false;
        }
        long now = System.currentTimeMillis();
        long deadline = now + 30000L;
        IOException lastError = null;
        do {
            FileStatus status = null;
            try {
                status = this.getFileStatus(outPath);
            }
            catch (FileNotFoundException e) {
                // empty catch block
            }
            if (status != null) {
                switch (writeMode) {
                    case NO_OVERWRITE: {
                        if (status.isDir() && createDirectory) {
                            return true;
                        }
                        throw new IOException("File or directory already exists. Existing files and directories are not overwritten in " + WriteMode.NO_OVERWRITE.name() + " mode. Use " + WriteMode.OVERWRITE.name() + " mode to overwrite existing files and directories.");
                    }
                    case OVERWRITE: {
                        if (status.isDir()) {
                            if (createDirectory) {
                                return true;
                            }
                            try {
                                this.delete(outPath, true);
                            }
                            catch (IOException e) {
                                lastError = e;
                            }
                            break;
                        }
                        try {
                            this.delete(outPath, false);
                        }
                        catch (IOException e) {
                            lastError = e;
                        }
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Invalid write mode: " + (Object)((Object)writeMode));
                    }
                }
            }
            if (createDirectory) {
                try {
                    if (!this.exists(outPath)) {
                        this.mkdirs(outPath);
                    }
                }
                catch (IOException e) {
                    lastError = e;
                }
                try {
                    FileStatus check = this.getFileStatus(outPath);
                    if (check != null) {
                        if (check.isDir()) {
                            return true;
                        }
                        lastError = new IOException("FileSystem should create an output directory, but the path points to a file instead.");
                    }
                }
                catch (FileNotFoundException e) {}
            } else {
                return !this.exists(outPath);
            }
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                throw new IOException("Thread was interrupted");
            }
        } while (System.currentTimeMillis() < deadline);
        if (lastError != null) {
            throw new IOException("File system failed to prepare output path " + outPath + " with write mode " + writeMode.name(), lastError);
        }
        return false;
    }

    public boolean initOutPathDistFS(Path outPath, WriteMode writeMode, boolean createDirectory) throws IOException {
        if (!this.isDistributedFS()) {
            return false;
        }
        if (this.exists(outPath)) {
            switch (writeMode) {
                case NO_OVERWRITE: {
                    throw new IOException("File or directory already exists. Existing files and directories are not overwritten in " + WriteMode.NO_OVERWRITE.name() + " mode. Use " + WriteMode.OVERWRITE.name() + " mode to overwrite existing files and directories.");
                }
                case OVERWRITE: {
                    try {
                        this.delete(outPath, true);
                    }
                    catch (IOException ioe) {}
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid write mode: " + (Object)((Object)writeMode));
                }
            }
        }
        if (createDirectory) {
            try {
                if (!this.exists(outPath)) {
                    this.mkdirs(outPath);
                }
            }
            catch (IOException ioe) {
                // empty catch block
            }
            return this.exists(outPath) && this.getFileStatus(outPath).isDir();
        }
        return !this.exists(outPath);
    }

    public abstract boolean isDistributedFS();

    public int getNumberOfBlocks(FileStatus file) throws IOException {
        FileStatus[] files;
        int numberOfBlocks = 0;
        if (file == null) {
            return 0;
        }
        if (!file.isDir()) {
            return this.getNumberOfBlocks(file.getLen(), file.getBlockSize());
        }
        for (FileStatus file1 : files = this.listStatus(file.getPath())) {
            if (file1.isDir()) continue;
            numberOfBlocks += this.getNumberOfBlocks(file1.getLen(), file1.getBlockSize());
        }
        return numberOfBlocks;
    }

    private int getNumberOfBlocks(long length, long blocksize) {
        if (blocksize != 0L) {
            int numberOfBlocks = (int)(length / blocksize);
            if (length % blocksize != 0L) {
                ++numberOfBlocks;
            }
            return numberOfBlocks;
        }
        return 1;
    }

    static {
        FSDIRECTORY.put("hdfs", HADOOP_WRAPPER_FILESYSTEM_CLASS);
        FSDIRECTORY.put("maprfs", MAPR_FILESYSTEM_CLASS);
        FSDIRECTORY.put("file", LOCAL_FILESYSTEM_CLASS);
        FSDIRECTORY.put("s3", S3_FILESYSTEM_CLASS);
    }

    public static class FSKey {
        private String scheme;
        private String authority;

        public FSKey(String scheme, String authority) {
            this.scheme = scheme;
            this.authority = authority;
        }

        public boolean equals(Object obj) {
            if (obj instanceof FSKey) {
                FSKey key = (FSKey)obj;
                if (!this.scheme.equals(key.scheme)) {
                    return false;
                }
                if (this.authority == null || key.authority == null) {
                    return this.authority == null && key.authority == null;
                }
                return this.authority.equals(key.authority);
            }
            return false;
        }

        public int hashCode() {
            if (this.scheme != null) {
                return this.scheme.hashCode();
            }
            if (this.authority != null) {
                return this.authority.hashCode();
            }
            return super.hashCode();
        }
    }

    public static enum WriteMode {
        NO_OVERWRITE,
        OVERWRITE;

    }
}

