/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.source;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.EventDrivenSource;
import org.apache.flume.channel.ChannelProcessor;
import org.apache.flume.conf.Configurable;
import org.apache.flume.event.EventBuilder;
import org.apache.flume.instrumentation.SourceCounter;
import org.apache.flume.source.AbstractSource;
import org.apache.flume.source.SyslogParser;
import org.apache.flume.source.SyslogUtils;
import org.apache.mina.core.buffer.IoBuffer;
import org.apache.mina.core.service.IoHandler;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.transport.socket.nio.NioSocketAcceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultiportSyslogTCPSource
extends AbstractSource
implements EventDrivenSource,
Configurable {
    public static final Logger logger = LoggerFactory.getLogger(MultiportSyslogTCPSource.class);
    private final ConcurrentMap<Integer, ThreadSafeDecoder> portCharsets;
    private List<Integer> ports = Lists.newArrayList();
    private String host;
    private NioSocketAcceptor acceptor;
    private Integer numProcessors;
    private int maxEventSize;
    private int batchSize;
    private int readBufferSize;
    private String portHeader;
    private SourceCounter sourceCounter = null;
    private Charset defaultCharset;
    private ThreadSafeDecoder defaultDecoder;
    private boolean keepFields;

    public MultiportSyslogTCPSource() {
        this.portCharsets = new ConcurrentHashMap<Integer, ThreadSafeDecoder>();
    }

    @Override
    public void configure(Context context) {
        String portsStr = context.getString("ports");
        Preconditions.checkNotNull((Object)portsStr, (Object)"Must define config parameter for MultiportSyslogTCPSource: ports");
        for (String portStr : portsStr.split("\\s+")) {
            Integer port = Integer.parseInt(portStr);
            this.ports.add(port);
        }
        this.host = context.getString("host");
        this.numProcessors = context.getInteger("numProcessors");
        this.maxEventSize = context.getInteger("eventSize", SyslogUtils.DEFAULT_SIZE);
        String defaultCharsetStr = context.getString("charset.default", "UTF-8");
        try {
            this.defaultCharset = Charset.forName(defaultCharsetStr);
        }
        catch (Exception ex) {
            throw new IllegalArgumentException("Unable to parse charset string (" + defaultCharsetStr + ") from port configuration.", ex);
        }
        this.defaultDecoder = new ThreadSafeDecoder(this.defaultCharset);
        this.portCharsets.clear();
        ImmutableMap portCharsetCfg = context.getSubProperties("charset.port.");
        for (Map.Entry entry : portCharsetCfg.entrySet()) {
            String portStr = (String)entry.getKey();
            String charsetStr = (String)entry.getValue();
            Integer port = Integer.parseInt(portStr);
            Preconditions.checkNotNull((Object)port, (Object)"Invalid port number in config");
            try {
                Charset charset = Charset.forName(charsetStr);
                this.portCharsets.put(port, new ThreadSafeDecoder(charset));
            }
            catch (Exception ex) {
                throw new IllegalArgumentException("Unable to parse charset string (" + charsetStr + ") from port configuration.", ex);
            }
        }
        this.batchSize = context.getInteger("batchSize", Integer.valueOf(100));
        this.portHeader = context.getString("portHeader");
        this.readBufferSize = context.getInteger("readBufferBytes", Integer.valueOf(1024));
        this.keepFields = context.getBoolean("keepFields", Boolean.valueOf(false));
        if (this.sourceCounter == null) {
            this.sourceCounter = new SourceCounter(this.getName());
        }
    }

    @Override
    public void start() {
        logger.info("Starting {}...", (Object)this);
        this.acceptor = this.numProcessors != null ? new NioSocketAcceptor(this.numProcessors.intValue()) : new NioSocketAcceptor();
        this.acceptor.setReuseAddress(true);
        this.acceptor.getSessionConfig().setReadBufferSize(this.readBufferSize);
        this.acceptor.getSessionConfig().setIdleTime(IdleStatus.BOTH_IDLE, 10);
        this.acceptor.setHandler((IoHandler)new MultiportSyslogHandler(this.maxEventSize, this.batchSize, this.getChannelProcessor(), this.sourceCounter, this.portHeader, this.defaultDecoder, this.portCharsets, this.keepFields));
        for (int port : this.ports) {
            InetSocketAddress addr = this.host != null ? new InetSocketAddress(this.host, port) : new InetSocketAddress(port);
            try {
                this.acceptor.bind((SocketAddress)addr);
            }
            catch (IOException ex) {
                logger.error("Could not bind to address: " + String.valueOf(addr), (Throwable)ex);
            }
        }
        this.sourceCounter.start();
        super.start();
        logger.info("{} started.", (Object)this);
    }

    @Override
    public void stop() {
        logger.info("Stopping {}...", (Object)this);
        this.acceptor.unbind();
        this.acceptor.dispose();
        this.sourceCounter.stop();
        super.stop();
        logger.info("{} stopped. Metrics: {}", (Object)this, (Object)this.sourceCounter);
    }

    @Override
    public String toString() {
        return "Multiport Syslog TCP source " + this.getName();
    }

    static class ThreadSafeDecoder
    extends ThreadLocal<CharsetDecoder> {
        private final Charset charset;

        public ThreadSafeDecoder(Charset charset) {
            this.charset = charset;
        }

        @Override
        protected CharsetDecoder initialValue() {
            return this.charset.newDecoder();
        }
    }

    static class ParsedBuffer {
        public IoBuffer buffer = null;
        public boolean incomplete = false;

        ParsedBuffer() {
        }
    }

    static class LineSplitter {
        private static final byte NEWLINE = 10;
        private final int maxLineLength;

        public LineSplitter(int maxLineLength) {
            this.maxLineLength = maxLineLength;
        }

        public boolean parseLine(IoBuffer buf, IoBuffer savedBuf, ParsedBuffer parsedBuf) {
            int msgPos;
            parsedBuf.buffer = null;
            parsedBuf.incomplete = false;
            buf.mark();
            boolean seenNewline = false;
            for (msgPos = savedBuf.position(); !seenNewline && buf.hasRemaining() && msgPos < this.maxLineLength; ++msgPos) {
                byte curByte = buf.get();
                if (curByte != 10) continue;
                seenNewline = true;
            }
            if (seenNewline) {
                int end = buf.position();
                buf.reset();
                int start = buf.position();
                if (savedBuf.position() > 0) {
                    byte[] tmp = new byte[end - start];
                    buf.get(tmp);
                    savedBuf.put(tmp);
                    int len = savedBuf.position() - 1;
                    savedBuf.flip();
                    parsedBuf.buffer = savedBuf.getSlice(len);
                    savedBuf.clear();
                } else {
                    parsedBuf.buffer = buf.getSlice(end - start - 1);
                    buf.get();
                }
                return true;
            }
            if (msgPos == this.maxLineLength) {
                int end = buf.position();
                buf.reset();
                int start = buf.position();
                if (savedBuf.position() > 0) {
                    byte[] tmp = new byte[end - start];
                    buf.get(tmp);
                    savedBuf.put(tmp);
                    savedBuf.flip();
                    parsedBuf.buffer = savedBuf.getSlice(msgPos);
                    savedBuf.clear();
                } else {
                    parsedBuf.buffer = buf.getSlice(msgPos);
                }
                logger.warn("Event size larger than specified event size: {}. Consider increasing the max event size.", (Object)this.maxLineLength);
                parsedBuf.incomplete = true;
                return true;
            }
            if (!buf.hasRemaining()) {
                int end = buf.position();
                buf.reset();
                int start = buf.position();
                byte[] tmp = new byte[end - start];
                buf.get(tmp);
                savedBuf.put(tmp);
                return false;
            }
            throw new IllegalStateException("unexpected buffer state: msgPos=" + msgPos + ", buf.hasRemaining=" + buf.hasRemaining() + ", savedBuf.hasRemaining=" + savedBuf.hasRemaining() + ", seenNewline=" + seenNewline + ", maxLen=" + this.maxLineLength);
        }
    }

    static class MultiportSyslogHandler
    extends IoHandlerAdapter {
        private static final String SAVED_BUF = "savedBuffer";
        private final ChannelProcessor channelProcessor;
        private final int maxEventSize;
        private final int batchSize;
        private final SourceCounter sourceCounter;
        private final String portHeader;
        private final SyslogParser syslogParser;
        private final LineSplitter lineSplitter;
        private final ThreadSafeDecoder defaultDecoder;
        private final ConcurrentMap<Integer, ThreadSafeDecoder> portCharsets;
        private final boolean keepFields;

        public MultiportSyslogHandler(int maxEventSize, int batchSize, ChannelProcessor cp, SourceCounter ctr, String portHeader, ThreadSafeDecoder defaultDecoder, ConcurrentMap<Integer, ThreadSafeDecoder> portCharsets, boolean keepFields) {
            this.channelProcessor = cp;
            this.sourceCounter = ctr;
            this.maxEventSize = maxEventSize;
            this.batchSize = batchSize;
            this.portHeader = portHeader;
            this.defaultDecoder = defaultDecoder;
            this.portCharsets = portCharsets;
            this.keepFields = keepFields;
            this.syslogParser = new SyslogParser();
            this.lineSplitter = new LineSplitter(maxEventSize);
        }

        public void exceptionCaught(IoSession session, Throwable cause) throws Exception {
            logger.error("Error in syslog message handler", cause);
            if (cause instanceof Error) {
                Throwables.propagate((Throwable)cause);
            }
        }

        public void sessionCreated(IoSession session) {
            logger.info("Session created: {}", (Object)session);
            session.setAttribute((Object)SAVED_BUF, (Object)IoBuffer.allocate((int)this.maxEventSize, (boolean)false));
        }

        public void sessionOpened(IoSession session) {
            logger.debug("Session opened: {}", (Object)session);
        }

        public void sessionClosed(IoSession session) {
            logger.info("Session closed: {}", (Object)session);
        }

        public void messageReceived(IoSession session, Object message) {
            IoBuffer buf = (IoBuffer)message;
            IoBuffer savedBuf = (IoBuffer)session.getAttribute((Object)SAVED_BUF);
            ParsedBuffer parsedLine = new ParsedBuffer();
            ArrayList events = Lists.newArrayList();
            CharsetDecoder decoder = (CharsetDecoder)this.defaultDecoder.get();
            int port = ((InetSocketAddress)session.getLocalAddress()).getPort();
            if (this.portCharsets.containsKey(port)) {
                decoder = (CharsetDecoder)((ThreadSafeDecoder)this.portCharsets.get(port)).get();
            }
            while (buf.hasRemaining()) {
                events.clear();
                for (int num = 0; num < this.batchSize && buf.hasRemaining(); ++num) {
                    if (this.lineSplitter.parseLine(buf, savedBuf, parsedLine)) {
                        Event event = this.parseEvent(parsedLine, decoder);
                        if (this.portHeader != null) {
                            event.getHeaders().put(this.portHeader, String.valueOf(port));
                        }
                        events.add(event);
                        continue;
                    }
                    logger.trace("Parsed null event");
                }
                if (events.isEmpty()) {
                    logger.trace("Empty set!");
                    return;
                }
                int numEvents = events.size();
                this.sourceCounter.addToEventReceivedCount(numEvents);
                try {
                    this.channelProcessor.processEventBatch(events);
                    this.sourceCounter.addToEventAcceptedCount(numEvents);
                }
                catch (Throwable t) {
                    logger.error("Error writing to channel, event dropped", t);
                    if (!(t instanceof Error)) continue;
                    Throwables.propagate((Throwable)t);
                }
            }
        }

        Event parseEvent(ParsedBuffer parsedBuf, CharsetDecoder decoder) {
            Event event;
            String msg = null;
            try {
                msg = parsedBuf.buffer.getString(decoder);
            }
            catch (Throwable t) {
                logger.info("Error decoding line with charset (" + decoder.charset() + "). Exception follows.", t);
                if (t instanceof Error) {
                    Throwables.propagate((Throwable)t);
                }
                byte[] bytes = new byte[parsedBuf.buffer.remaining()];
                parsedBuf.buffer.get(bytes);
                Event event2 = EventBuilder.withBody((byte[])bytes);
                event2.getHeaders().put("flume.syslog.status", SyslogUtils.SyslogStatus.INVALID.getSyslogStatus());
                return event2;
            }
            logger.trace("Seen raw event: {}", (Object)msg);
            try {
                event = this.syslogParser.parseMessage(msg, decoder.charset(), this.keepFields);
                if (parsedBuf.incomplete) {
                    event.getHeaders().put("flume.syslog.status", SyslogUtils.SyslogStatus.INCOMPLETE.getSyslogStatus());
                }
            }
            catch (IllegalArgumentException ex) {
                event = EventBuilder.withBody((String)msg, (Charset)decoder.charset());
                event.getHeaders().put("flume.syslog.status", SyslogUtils.SyslogStatus.INVALID.getSyslogStatus());
                logger.debug("Error parsing syslog event", (Throwable)ex);
            }
            return event;
        }
    }
}

