/*
 * bone flags for PMX
 *
 * License : The MIT License
 * Copyright(c) 2010 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx;

import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

/**
 * ボーン種別.
 * <ul>
 * <li>OFFSET(0x0001) オフセット. 0: オフセットセット指定, 1:先ボーン指定</li>
 * <li>ROTATE(0x0002) 回転.</li>
 * <li>MOVE(0x0004) 移動.</li>
 * <li>VISIBLE(0x0008) 表示.</li>
 * <li>OP(0x0010) 操作.</li>
 * <li>IK(0x0020) IK.</li>
 * <li>LINK_FLAG(0x0080) ローカル付与フラグ.</li>
 * <li>ROTATE_LINK(0x0100) 回転付与.</li>
 * <li>MOVE_LINK(0x0200) 移動付与.</li>
 * <li>AXIS_ROTATE(0x0400) 回転軸固定.</li>
 * <li>LOCAL_AXIS(0x0800) ローカル座標軸.</li>
 * <li>PHYSICAL(0x1000) 物理後変形.</li>
 * <li>EXTRA(0x2000) 外部親変形.</li>
 * </ul>
 */
public enum BoneFlags {

    /** オフセット. (0:のときオフセット. 1:のときボーン.) */
    OFFSET(0x0001),
    /** 回転. */
    ROTATE(0x0002),
    /** 移動. */
    MOVE(0x0004),
    /** 表示. */
    VISIBLE(0x0008),
    /** 操作. */
    OP(0x0010),
    /** IK. */
    IK(0x0020),

    /** ローカル付与フラグ. */
    LINK_FLAG(0x0080),
    /** 回転付与. */
    ROTATE_LINK(0x0100),
    /** 移動付与. */
    MOVE_LINK(0x0200),
    /** 回転軸固定. */
    AXIS_ROTATE(0x0400),
    /** ローカル座標軸. */
    LOCAL_AXIS(0x0800),
    /** 物理後変形 */
    PHYSICAL(0x1000),
    /** 外部親変形 */
    EXTRA(0x2000)
    ;

    private static final ResourceBundle.Control NOFALLBACK;
    private static final String FAMILY_NAME =
            "jp.sfjp.mikutoga.pmx.resources.BoneFlags";

    static{
        List<String> rbforms = ResourceBundle.Control.FORMAT_DEFAULT;
        NOFALLBACK = ResourceBundle.Control.getNoFallbackControl(rbforms);

        String name = ROTATE.getClass().getPackage().getName();
        assert FAMILY_NAME.startsWith(name);
    }


    private final short encoded;

    /**
     * コンストラクタ。
     * @param code 符号化int値
     */
    private BoneFlags(int code){
        this((short)code);
    }

    /**
     * コンストラクタ。
     * @param code 符号化short値
     */
    private BoneFlags(short code){
        this.encoded = code;
    }

    /**
     * short値からデコードする。
     * @param code short値
     * @return デコードされた列挙子。該当するものがなければnull
     */
    public static BoneFlags decode(short code){
        BoneFlags result = null;

        for(BoneFlags type : values()){
            if(type.encode() == code){
                result = type;
                break;
            }
        }

        return result;
    }

    /**
     * short値にエンコードする。
     * @return short値
     */
    public short encode(){
        return this.encoded;
    }

    /**
     * フラグがonかテストする.
     * @param objective テスト対象.
     * @return on なら true
     */
    public boolean check(short objective){
        return ((objective & this.encoded) >0);
    }
    
    /**
     * デフォルトロケールでの表示名を返す。
     * @return 表示名
     */
    public String getGuiName(){
        Locale locale = Locale.getDefault();
        assert locale != null;
        return getGuiName(locale);
    }

    /**
     * ロケールに準じた表示名を返す。
     * @param locale ロケール。nullならデフォルトロケールと解釈される。
     * @return 表示名
     */
    public String getGuiName(Locale locale){
        if(locale == null) return getGuiName();
        ResourceBundle rb =
                ResourceBundle.getBundle(FAMILY_NAME, locale, NOFALLBACK);
        String key = name();
        String result = rb.getString(key);
        return result;
    }

}
